/*
 * Licensed Materials - Property of HCL
 * UrbanCode Deploy
 * (c) Copyright HCL Technologies Ltd. 2018. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * Trademark of HCL Technologies Limited
 */

package com.urbancode.air.plugin.loadrunner

import java.io.File
import java.text.SimpleDateFormat
import java.util.List
import java.util.regex.Matcher
import java.util.regex.Pattern
import org.apache.commons.io.FileUtils
import org.ini4j.Wini

import com.urbancode.air.CommandHelper

class LRCLIHelper {
    private File workDir
    private CommandHelper cmdHelper
    private File lrInstallDir
    private File wlRunExe
    private File analysisUIExe

    /* Non-configurable fields */
    private final String UCD_ANALYSIS_TEMPLATE = "UCD_ANALYSIS_TEMPLATE" // Custom template name
    private final String LR_CONFIG_FILE = "LRAnalysis80.ini" // INI file used to import templates
    private final String TEMPLATE_SECTION = "AvailableTemplates" // Section of the INI file that contains templates

    public LRCLIHelper(String lrInstallPath, File workDir) {
        this.workDir = workDir

        cmdHelper = new CommandHelper(workDir)
        lrInstallDir = resolveFilePath(lrInstallPath)
        wlRunExe = new File(lrInstallDir, "bin" + File.separator + "Wlrun.exe")
        analysisUIExe = new File(lrInstallDir, "bin" + File.separator + "AnalysisUI.exe")
    }

    /* Execute an LRS file and analyze the results. */
    public int runTestScenario(String testPath, String lrResultsName) {
        File testFile = resolveFilePath(testPath)

        List<String> cmdArgs = [
            wlRunExe.absolutePath,
            "-TestPath",
            testFile.absolutePath,
            "-Run",
            "-ResultLocation",
            workDir.absolutePath,
            "-ResultCleanName",
            lrResultsName]

        println("${getTimestamp()} [Action] Running test scenario '${testPath}'.")
        int exitVal = executeCLICommand(cmdArgs)
        println("${getTimestamp()} [OK] Test scenario has executed successfully.")

        return exitVal
    }

    /* Execute analysis on the generated LRR file and create an HTML report using the given template. */
    public int runTestAnalysis(String templateName, String lrResultsName) {
        File resultsFile = new File(workDir, lrResultsName + File.separator + "${lrResultsName}.lrr")

        if (!resultsFile.isFile()) {
            println("[Error] Results file doesn't exist at '${resultsFile.absolutePath}'.")
            return -1
        }

        /* Copy and import the UCD analysis template if it has been specified */
        if (UCD_ANALYSIS_TEMPLATE.equals(templateName)
            && (copyTemplate() != 0 || importUCDTemplate() != 0))
        {
            return -1
        }

        List<String> cmdArgs = [
            analysisUIExe.absolutePath,
            "-RESULTPATH",
            resultsFile.absolutePath,
            "-TEMPLATENAME",
            templateName]

        /* Generate the HTML analysis file */
        println("${getTimestamp()} [Action] Analyzing the test results.")
        int exitVal = executeCLICommand(cmdArgs)
        println("${getTimestamp()} [OK] Completed test result analysis.")

        return exitVal
    }

    public static String getTimestamp() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("[MM/dd/yyyy HH:mm:ss]")

        return dateFormat.format(new Date())
    }

    /* Copy the UCD analysis template from PLUGIN_HOME to the working directory
     * This must be done, otherwise HP LoadRunner would delete the template permanently if the user deleted
     * the template using the HP LoadRunner Analysis tool.
     */
    private int copyTemplate() {
        File sourceTemplateDir = new File(System.getenv("PLUGIN_HOME"), UCD_ANALYSIS_TEMPLATE)
        File destTemplateDir = new File(workDir, UCD_ANALYSIS_TEMPLATE)

        try {
            println("${getTimestamp()} [Action] Copying UCD analysis template from ${sourceTemplateDir} to ${workDir}.")
            FileUtils.copyDirectory(sourceTemplateDir, destTemplateDir)
        }
        catch (IOException ex) {
            println("[Error] Failed to copy UCD analysis template from ${sourceTemplateDir} to ${destTemplateDir}.")
            ex.printStackTrace()
            return -1
        }

        println("${getTimestamp()} [OK] Successfully copied UCD analysis template.")
        return 0
    }

    /* Import UCD analysis template if it has not already been imported */
    private int importUCDTemplate() {
        File analysisConfig = new File(lrInstallDir, "config" + File.separator + LR_CONFIG_FILE)

        println("${getTimestamp()} [Action] Importing UCD analysis template in HP LoadRunner Analysis.")

        /* Import/Update the template if the agent user has permissions to edit analysis config */
        if (analysisConfig.canRead() && analysisConfig.canWrite()) {
            Wini analysisIni = new Wini(analysisConfig)

            /* INI file must contain AvailableTemplates section */
            if (analysisIni.keySet().contains(TEMPLATE_SECTION)) {
                String newTemplatePath = workDir.absolutePath + File.separator
                int numOfTemplates = analysisIni.get(TEMPLATE_SECTION, "NumOfTemplates", int.class)

                /* Update the path to the template if it has already been imported */
                boolean templateImported = updateTemplatePath(analysisIni, newTemplatePath)

                /* Import UCD_ANALYSIS_TEMPLATE if it has never been imported */
                if (!templateImported) {
                    /* Insert the new template entry into the INI file and update template count */
                    analysisIni.put(TEMPLATE_SECTION, "Template${numOfTemplates}_Name", UCD_ANALYSIS_TEMPLATE)
                    analysisIni.put(TEMPLATE_SECTION, "Template${numOfTemplates}_Path", newTemplatePath)
                    analysisIni.put(TEMPLATE_SECTION, "NumOfTemplates", ++numOfTemplates)
                    analysisIni.store()

                    println("${getTimestamp()} [OK] UCD analysis template has been successfully imported.")
                }
                else {
                    println("${getTimestamp()} [OK] UCD analysis template path has been updated.")
                }
            }
            else {
                println("[Error] ${analysisConfig.absolutePath} file is missing 'AvailableTemplates' section.")
                return -1
            }
        }
        else {
            println("[Error] Template import failed. UCD agent user doesn't have read/write permissions to " +
                "file path '${analysisConfig.absolutePath}'")
            println("[Solution] You may create your own custom analysis template or fix the file permissions. " +
                "Please see the plugin documentation for further details.")
            return -1
        }

        return 0
    }

    /* Update current UCD analysis template path and return if it exists in the analysis config. */
    private boolean updateTemplatePath(Wini analysisIni, String templatePath) {
        Pattern p = Pattern.compile("(Template[0-9]+)_Name") // Pattern to match Template<number>_Name key

        /* Iterate through all templates to check if UCD_ANALYSIS_TEMPLATE already exists */
        for (String keyName : analysisIni.get(TEMPLATE_SECTION).keySet()) {
            Matcher m = p.matcher(keyName)

            if (m.find()) {
                /* Get the template name from the whole matched expression */
                String templateName = analysisIni.get(TEMPLATE_SECTION, m.group(0))

                /* Template named UCD_ANALYSIS_TEMPLATE will exist if it's already been imported */
                if (UCD_ANALYSIS_TEMPLATE.equals(templateName)) {
                    println("${getTimestamp()} [Action] UCD analysis template was already imported, updating path.")
                    String pathKey = m.group(1) + "_Path"   // The name of the template's path key
                    analysisIni.put(TEMPLATE_SECTION, pathKey, templatePath)
                    analysisIni.store()
                    return true
                }
            }
        }

        return false
    }

    /* Resolve a relative or absolute file system path to a file or directory */
    private File resolveFilePath(String path) {
        File entry = new File(path)

        /* Do not build path with workDir if user specifies an absolute file path */
        if (!entry.isAbsolute()) {
            entry = new File(workDir, path)
        }

        if (!entry.exists()) {
            throw new FileNotFoundException("File system path '${entry.absolutePath}' does not exist.")
        }

        return entry
    }

    /* Run the CLI command. */
    private int executeCLICommand(List<String> cmdArgs) {
        return cmdHelper.runCommand(cmdArgs.join(' '), cmdArgs)
    }
}
