/*
 * Licensed Materials - Property of HCL
 * UrbanCode Deploy
 * (c) Copyright HCL Technologies Ltd. 2018. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * Trademark of HCL Technologies Limited
 */
package com.urbancode.air.plugin.infosphere

import java.util.regex.Matcher
import java.util.regex.Pattern

import com.urbancode.air.CommandHelper

class ISDHelper {
    /* Fields common for running all ISD CLI Commands */
    private File importExport
    private File workDir
    private String username
    private String password
    private CommandHelper cmdHelper
    private List<String> additionalArgs

    public ISDHelper(String importExportPath, File workDir, String username, String password) {
        this.workDir = workDir
        this.username = username
        this.password = password

        importExport = resolveFilePath(importExportPath)
        cmdHelper = new CommandHelper(workDir)
    }

    /* Execute a design time export to generate an XML file */
    public int doDesignExport(
        String action,
        List<String> projects,
        List<String> applications,
        List<String> services,
        String outputPath,
        boolean omitPassword)
    {
        File output = resolveFilePath(outputPath)   // Output directory or file
        List<String> cmdArgs = [
            importExport.absolutePath,
            "-action",
            action,
            "-output",
            output.absolutePath]

        if (omitPassword) {
            cmdArgs << "-omitPassword"
        }

        for (String project in projects) {
            cmdArgs += ["-p", project]
        }

        /**
         * When multiple projects are specified, the applications/services specified by the user are ignored
         * Instead, all applications and services of the corresponding project will be exported.
         */
        if (projects.size() > 1) {
            if (!applications.isEmpty() || !services.isEmpty()) {
                println("[Warning] Applications and services cannot be specified with multiple projects.")
                println("[OK] All applications and services of the specified projects will be exported.")
            }
            if (!output.isDirectory()) {
                println("[Error] When exporting multiple projects you must specify an output directory." +
                    " instead of a file. The output path '${output.absolutePath}' is not a directory.")
                return 1
            }
        }
        else {

            for (String application in applications) {
                cmdArgs += ["-a", application]
            }

            /**
             * When multiple applications are specified, the services specified by the user are ignored
             * Instead, all services of the corresponding application will be exported.
             */
            if (applications.size() > 1 && !services.isEmpty()) {
                println("[Warning] Services cannot be specified with multiple applications.")
                println("[OK] All services of specified the applications will be exported.")
            }
            else {
                for (String service in services) {
                    cmdArgs += ["-s", service]
                }
            }
        }

        println("[Action] Executing design time export with output path '${output.absolutePath}'...")
        return executeISDCommand(cmdArgs)
    }

    /* Execute a runtime export of running applications to generate a DAT file */
    public int doRuntimeExport(
        String action,
        List<String> applications,
        String outputPath,
        boolean omitPassword)
    {
        File output = resolveFilePath(outputPath)   // Output directory or file
        List<String> cmdArgs = [
            importExport.absolutePath,
            "-action",
            action,
            "-output",
            output.absolutePath]

        if (applications.size() > 1 && !output.isDirectory()) {
            println("[Error] When exporting multiple runtime applications you must specify an output" +
                " directory instead of a file. The output path '${output.absolutePath}' is not a directory.")
            return 1
        }

        if (omitPassword) {
            cmdArgs << "-omitPassword"
        }

        for (String application in applications) {
            cmdArgs += ["-a", application]
        }

        println("[Action] Exporting runtime applications with output path '${output.absolutePath}'...")
        return executeISDCommand(cmdArgs)
    }

   /* Execute a design time import from an XML file */
   public int doDesignImport(
        String action,
        String project,
        List<String> applications,
        List<String> services,
        String inputPath,
        boolean replace)
    {
        File input = resolveFilePath(inputPath)
        List<String> cmdArgs = [
            importExport.absolutePath,
            "-action",
            action,
            "-input",
            input.absolutePath]

        if (replace) {
            cmdArgs << "-replace"
        }

        if (!input.isDirectory()) {
            if (project) {
                cmdArgs += ["-p", project]
            }

            /* Import only specific applications */
            for (String application in applications) {
                cmdArgs += ["-a", application]
            }

            /**
             * When multiple applications are specified, the services specified by the user are ignored
             * Instead, all services of the corresponding application will be imported.
             */
            if (applications.size() > 1 && !services.isEmpty()) {
                println("[Warning] Services cannot be specified with multiple applications.")
                println("[OK] All services of specified applications will be imported")
            }
            else {
                for (String service in services) {
                    cmdArgs += ["-s", service]
                }
            }
        }
        else {
            if (project || !applications.isEmpty() || !services.isEmpty()) {
                println("[Warning] When an input directory is used instead of a file you may not specify" +
                    " individual projects, applications, or services.")
                println("[OK] All projects, applications, and services in the input directory will be imported.")
            }
        }


        println("[Action] Executing design time import with input path '${input.absolutePath}'")
        return executeISDCommand(cmdArgs)
    }

    /* Execute a runtime export of running applications to generate a DAT file */
    public int doRuntimeImport(String action, String inputPath, boolean replace) {
        File input = resolveFilePath(inputPath)   // Output directory or file
        List<String> cmdArgs = [
            importExport.absolutePath,
            "-action",
            action,
            "-input",
            input.absolutePath]

        if (replace) {
            cmdArgs << "-replace"
        }

        println("[Action] Importing runtime applications with input path '${input.absolutePath}'.")
        return executeISDCommand(cmdArgs)
    }

    /* Set arguments that will be passed to the ISDImportExport command */
    public setAdditionalArgs(List<String> additionalArgs) {
        this.additionalArgs = additionalArgs
    }

    /* Get a list of arguments separated by spaces or quotation-wrapped arguments.*/
    public List<String> getCommandArgs(String unparsedString) {
        List<String> parsedArgs = []

        /* Match any number of non-space characters OR any number of characters surrounded in quotes */
        Matcher m = Pattern.compile("([^\"]\\S*|\".+?\")\\s*").matcher(unparsedString)
        while (m.find()) {
            parsedArgs << m.group(1)
        }

        return parsedArgs*.trim()
    }

    /* Run an ISDImportExport CLI command */
    private int executeISDCommand(List<String> cmdArgs) {
        if (username && password) {
            cmdArgs += ["-user", username, "-password", password]
        }

        for (String additionalArg in additionalArgs) {
            cmdArgs << additionalArg
        }

        cmdHelper.runCommand(cmdArgs.join(' '), cmdArgs)
        return 0
    }

    /* Resolve a relative or absolute file system path to a file or directory */
    private File resolveFilePath(String path) {
        File entry = new File(path)

        /* Do not build path with workDir if user specifies an absolute file path */
        if (!entry.isAbsolute()) {
            entry = new File(workDir, path)
        }

        if (!entry.exists()) {
            throw new FileNotFoundException("File system path '${entry.absolutePath}' does not exist.")
        }

        return entry
    }
}