/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2016. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/

package com.urbancode.air.plugin.kubernetes

import com.urbancode.air.CommandHelper
import com.urbancode.air.ExitCodeException

class KubernetesHelper {

    private CommandHelper ch
    private String home

    /**
    *  @param workDir The working directory for the CommandHelper CLI
    *  @param home The full path to the kubectl script location
    */
    public KubernetesHelper(File workDir, String home) {
        if (home) {
            this.home = home
        }
        else {
            this.home = 'kubectl'
        }
        ch = new CommandHelper(workDir)
        ArrayList args = []
        if (!runCommand('[action]  Checking kubectl home...', args, 'kubectl home set.', 'Could not set kubectl home.')) {
            System.exit(1)
        }
    }

    /**
    *  @param proc The process to retrieve the standard output and standard error from
    *  @return An array containing the standard output and standard error of the process
    */
    public String[] captureCommand(Process proc) {
        StringBuffer sout = new StringBuffer()
        StringBuffer serr = new StringBuffer()
        proc.waitForProcessOutput(sout, serr)
        proc.out.close()
        return [sout.toString(), serr.toString()]
    }

    /**
    *  @param message The message to output before running the command
    *  @param args An ArrayList of arguments to be executed by the command prompt
    *  @param success The message to display if the command completes without errors
    *  @param failure The message to display if the command completes with errors
    *  @return true if the command is run without any Standard Errors, false otherwise
    */
    public boolean runCommand(String message, ArrayList args, String success, String failure) {
        ArrayList commandLine = [home]
        args.each() { arg ->
            commandLine << arg
        }
        boolean status
        try {
            ch.runCommand(message, commandLine) { Process proc ->
                def (String sout, String serr) = captureCommand(proc)
                if (serr) {
                    println ("[error]  ${failure}")
                    println serr
                    status = false
                }
                else {
                    println ("[ok]  ${success}")
                    if (args.size() > 0) {
                        println sout
                    }
                    status = true
                }
            }
        }
        catch (ExitCodeException ece) {
            return false
        }
        return status
    }

    /**
    *  @param args The list of arguments to add the global configuration options to
    *  @param url A url to specify for the Kubernetes server
    *  @param username A username to specify for authentication to the Kubernetes server
    *  @param password A password to specify for authentication to the kubernetes server
    *  @param namespce A namespce to specify the scope on the Kubernetes server of the CLI request
    *  @param globals A list of any other global arugments
    */
    public void setGlobals(ArrayList args, String url, String username, String password, String namespace, String globals) {
        if (url) {
            args << '--server=' + url
        }
        if (username) {
            args << '--username=' + username
        }
        if (password) {
            args << '--password=' + password
        }
        if (namespace) {
            args << '--namespace=' + namespace
        }
        if (globals) {
            globals.split("[\r\n]+").each() { global ->
                args << global
            }
        }
    }

    /**
    *  @param args The list of arguments to add the global configuration options to
    *  @param flags The list of flags to add to the args
    */
    public void setFlags(ArrayList args, String flags) {
        if (flags) {
            flags.split("[\r\n]+").each() { flag ->
                args << flag
            }
        }
    }
}
