/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Build
 * IBM UrbanCode Deploy
 * IBM UrbanCode Release
 * IBM AnthillPro
 * (c) Copyright IBM Corporation 2017. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
import org.codehaus.jettison.json.JSONArray
import org.codehaus.jettison.json.JSONObject
import com.urbancode.air.AirPluginTool
import com.urbancode.air.XTrustProvider
import com.urbancode.air.plugin.kubernetes.KubernetesHelper
import com.urbancode.ud.client.ApplicationClient
import com.urbancode.ud.client.ComponentClient
import com.urbancode.ud.client.EnvironmentClient
import com.urbancode.ud.client.ResourceClient
import com.urbancode.ud.client.VersionClient
import com.urbancode.ud.client.PropertyClient
import com.urbancode.ud.client.UDRestClient

XTrustProvider.install()
def COMPONENT_TAG='kubernetes.image'

// Get step properties
def airTool = new AirPluginTool(this.args[0], this.args[1])
def props = airTool.getStepProperties()
String serverUrl                    = props['serverUrl']
String username                     = props['username']
String password                     = props['password']
String resourceId                   = props['resource']
String applicationId                = props['application']
String applicationProcessRequestId  = props['applicationProcessRequest']
String componentProcessRequestId    = props['componentProcessRequest']
String environmentId                = props['environment']
String environmentName              = props['environmentName']
String kubeFilePath                 = props['yamlFile']
String sourceConfigType             = props['sourceConfigType']
String componentTemplate            = props['componentTemplate']

// Sanitize properties
def serverUri = KubernetesHelper.stringToUri(serverUrl)
if (serverUri.equals(new URI(''))) {
    System.exit(1)
}
if (!username) {
    username = airTool.getAuthTokenUsername()
}
if (!password) {
    password = airTool.getAuthToken()
}


ApplicationClient applicationClient = new ApplicationClient(serverUri, username, password)
ComponentClient componentClient = new ComponentClient(serverUri, username, password)
EnvironmentClient environmentClient = new EnvironmentClient(serverUri, username, password)
ResourceClient resourceClient = new ResourceClient(serverUri,  username, password)
VersionClient versionClient = new VersionClient(serverUri, username, password)
PropertyClient propertyClient = new PropertyClient(serverUri, username, password)
UDRestClient restClient = new UDRestClient(serverUri, username, password)

applicationName = KubernetesHelper.parseTextAsJson(applicationClient.getApplication(applicationId)).name
String deploymentRequestId = componentClient.getComponentProcessRequest(componentProcessRequestId)?.deploymentRequestId
def existingResources = KubernetesHelper.parseTextAsJson(resourceClient.getResourceChildren(resourceId))
resource = resourceClient.getResourceById(resourceId);

// Get the current inventory for any existing image components
// We need this in case the user did not specify a version for the component
Map<String, String> desiredInventory = new HashMap<String,String>()
components = KubernetesHelper.parseTextAsJson(applicationClient.getApplicationComponents(applicationId));
components.each { component ->
    componentId = componentClient.getComponentUUID(component.name).toString();
    // if this component has the yaml file (call it root resource), we don't want to create components/update inventory
    // if following the out-of-the box tutorial, the resource is a tag ("Kubernetes")
    // So, check components to see if they have the resource tag. If not using tags,
    // determine if component name = resource name.
    hasResourceNameAsTag = false
    if (component.tags?.find{resource.name.equals(it.name)}) {
        hasResourceNameAsTag = true;
    }
    isRootResourceComponent = true
    if (!hasResourceNameAsTag && !component.name.equals(resource.name)) {
        isRootResourceComponent = false
    }

    // UDRestClient currently does not encode the component name for getLatestEnvironmentInventoryByComponent, so we need to
    sanitizedComponentName = restClient.sanitizePathSegment(component.name);
    inventory = KubernetesHelper.parseTextAsJson(environmentClient.getLatestEnvironmentInventoryByComponent(environmentName, applicationId, sanitizedComponentName));
    if ((inventory) && !isRootResourceComponent) {
        imageName = component.name.substring(applicationName.length()+1);
        desiredInventory.put(imageName, inventory.version.name);
    }

    // if the component exists but is not mapped to our resource, map it
    if (!existingResources?.find{component.name.equals(it.name)} && !isRootResourceComponent) {
        println "Mapping " + component.name + " to base resource"
        try {
            createdResource = KubernetesHelper.parseTextAsJson(resourceClient.createResource(component.name, null, null, resourceId, component.name))
            // need to set desired/actual inventory for these new resources
            if (inventory.version.name) {
                def existingVersions = KubernetesHelper.parseTextAsJson(componentClient.getComponentVersionsJsonArray(component.name, false))
                def versionId
                existingVersions.each { def existingVersion ->
                    if (inventory.version.name.equals(existingVersion.name)) {
                        versionId = existingVersion.id
                    }
                }
                if (versionId) {
                    println "Updating desired/actual inventory for " + component.name
                    environmentClient.createDesiredInventoryEntry(deploymentRequestId, environmentId, componentId, versionId.toString(), 'Active')
                    resourceClient.createResourceInventoryEntry(deploymentRequestId, createdResource.id, componentId, versionId.toString(), 'Active')
                }
            }
        }
        catch (IOException e){
            // do nothing; the resource probably exists
        }
    }
}


// Get the user selected component versions
userSelectedVersions = KubernetesHelper.parseTextAsJson(applicationClient.getApplicationProcessRequestVersions(applicationProcessRequestId));
userSelectedVersions.each{ version ->
    componentName = version.component.name;
    component = KubernetesHelper.parseTextAsJson(componentClient.getComponent(componentName));
    // if this component has the yaml file (call it root resource), we don't want to create components/update inventory
    // if following the out-of-the box tutorial, the resource is a tag ("Kubernetes")
    // So, check components to see if they have the resource tag. If not using tags,
    // determine if component name = resource name.
    hasResourceNameAsTag = false
    if (component.tags?.find{resource.name.equals(it.name)}) {
        hasResourceNameAsTag = true;
    }
    isRootResourceComponent = true
    if (!hasResourceNameAsTag && !component.name.equals(resource.name)) {
        isRootResourceComponent = false
    }
    // components (for images) names are in the format ApplicationName-imageName
    if (!isRootResourceComponent) {
        imageName = componentName.substring(applicationName.length()+1);
        desiredInventory.put(imageName, version.name)
    }
}
println "The desired versions for existing image components is " + desiredInventory

// Read the yaml file, replacing image versions as needed
if (!kubeFilePath) {
    // if user did not specify a yaml file, assume they have downloaded one (and only one)
    // yaml file to the working directory
    String currentDir = new File(".").getAbsoluteFile().getParent();
    baseDir = new File(currentDir);
    allFilesInDir = baseDir.listFiles();
    if (allFilesInDir.size() == 0) {
      throw new Exception("No Kubernetes YAML file was specified and no files were found in " + baseDir);
    }
    if (allFilesInDir.size() > 1) {
      throw new Exception("No Kubernetes YAML file was specified and more than one file was found in  " + baseDir);
    }
    kubeFilePath = allFilesInDir[0];
}
def kubeFile = new File(kubeFilePath);
println "Loading ${kubeFile}"
imagesToCreateComponentsFor = [];
def List newYamlLines = [];
originalYamlLines = kubeFile.readLines();
originalYamlLines.each { String line ->
    trimmedLine = line.trim();
    if (trimmedLine.contains("image: ")) {
        // get everything after "image: "
        image = trimmedLine.substring(trimmedLine.indexOf("image: ")+7);
        // there could be comments after the image name
        indexOfComment = image.indexOf("#");
        if (indexOfComment != -1) {
          image = image.substring(0, indexOfComment).trim();
        }
        parsedImage = KubernetesHelper.parseImageSpec(image);
        imageName = parsedImage.image;
        imageVersion = parsedImage.version;
        if (desiredInventory != null) {
            imageNameWithoutNameSpace = getImageNameWithoutNamespace(imageName)
            userSelectedVersionForThisImage = desiredInventory.get(imageNameWithoutNameSpace);
            if (userSelectedVersionForThisImage != null) {
                // we found a user selected version for this image
                line = line.replace(":" + imageVersion, ":" + userSelectedVersionForThisImage);
            }
            else {
                // we need to create a component for this image
                imagesToCreateComponentsFor.add(parsedImage);
            }
        }
        else {
            // we need to create a component for this image
            imagesToCreateComponentsFor.add(parsedImage);
        }
  }
  newYamlLines << line;
}


// Create an new yaml file containing user selected versions for images
println "Creating ibm-ucd-kubernetes.yaml"
new File("ibm-ucd-kubernetes.yaml").withWriter { out ->
    newYamlLines.each {
        out.println it
    }
}


// Create components
def applicationComponents = KubernetesHelper.parseTextAsJson(applicationClient.getApplicationComponents(applicationId))
imagesToCreateComponentsFor.each{ entry ->
    // image names may contain a slash (such as google-samples/gb-frontend)
    // however, having slashes in a component name will prevent us from using
    // the uDeployRestClient. So, we will remove the namespace
    // (everything before and including the slash) when naming our component
    imageNameWithoutNameSpace = getImageNameWithoutNamespace(entry.image);
    componentName = "${applicationName.replace(' ', '')}-" + imageNameWithoutNameSpace;
    def registry = entry.registry
    def image = entry.image
    def tag
    if (entry.version) {
        tag = entry.version
    }
    else {
        tag = 'latest'
    }

    def sourceProps = [:]
    if (registry) {
        sourceProps.put('DockerTagImport/dockerRegistryName', registry)
    }
    else {
        sourceProps.put('DockerTagImport/dockerRegistryName', '')
    }
    sourceProps.put('DockerTagImport/dockerImageName', image)
    sourceProps.put('DockerTagImport/namingConvention', 'tag_only')

    def componentProps = new Properties()
    componentProps.setProperty('docker.image.name', image)
    if (registry) {
        componentProps.setProperty('docker.registry.name', registry)
    }
    else {
        componentProps.setProperty('docker.registry.name', '')
    }

    String componentId = createComponent(componentClient, componentName, sourceProps, componentProps,
                                         sourceConfigType, componentTemplate).toString()

    //map components to this base resource
    //Currently assigns name componentName - if creating multiple instances of a container, should be unique

    if (!existingResources?.find{ componentName.equals(it.name) }) {
        println "Mapping to base reource ${componentName}"
        println(existingResources)
        try {
            resourceClient.createResource(componentName, null, null, resourceId, componentName)
        }
        catch (IOException e){
          // do nothing; the resource probably exists
        }
    }

    if (applicationComponents && !applicationComponents.find{ resourceId.equals(it.id) }) {
        applicationClient.addComponentToApplication(applicationId, componentId)
    }

    componentClient.addTagToComponent(componentId, COMPONENT_TAG)

    def existingVersions = KubernetesHelper.parseTextAsJson(componentClient.getComponentVersionsJsonArray(componentName, false))

    def versionId

    existingVersions.each { def existingVersion ->
        if (tag.equals(existingVersion.name)) {
            versionId = existingVersion.id
        }
    }

    if (!versionId) {
        versionId = versionClient.createVersion(componentId, tag, "Imported from Kubernetes yaml")

        versionClient.setVersionProperty(tag, componentId, 'dockerImageTag', tag, false)
        if (registry) {
            versionClient.setVersionProperty(tag, componentId, 'registry', registry, false)
        }
        else {
            versionClient.setVersionProperty(tag, componentId, 'registry', '', false)
        }
    }

    try {
        environmentClient.createDesiredInventoryEntry(deploymentRequestId, environmentId, componentId, versionId.toString(), 'Active')
    }
    catch (IOException ex) {
        throw new IOException("[Error] Unable to update the Application manifest.\n " +
            "[Possible Solution] IBM UrbanCode Deploy v6.2+ is required for this step to complete successfully.", ex)
    }
}

// Update inventory for any components that were created
if (imagesToCreateComponentsFor.size() > 0) {
    println "Updating inventory for components which were created"
    def children = KubernetesHelper.parseTextAsJson(resourceClient.getResourceChildren(resourceId))
    children.each { childResource ->
        def roles = KubernetesHelper.parseTextAsJson(resourceClient.getResourceRoles(childResource.id))
        roles.each { role ->
            // if child resource has role, update inventory
            def componentName = role.name
            def componentId = componentClient.getComponentUUID(componentName).toString();
            if (componentName.startsWith(applicationName + "-")) {
                imageName = componentName.substring(applicationName.length()+1);
                // get desired version
                desiredVersion = "";
                imagesToCreateComponentsFor.each{ entry ->
                    entryImageWithoutNamespace = getImageNameWithoutNamespace(entry.image)
                    if (entryImageWithoutNamespace == imageName) {
                        desiredVersion = entry.version;
                    }
                }
                if (desiredVersion != "") {
                    // need to get the id of the version
                    def existingVersions = KubernetesHelper.parseTextAsJson(componentClient.getComponentVersionsJsonArray(componentName, false))
                    def versionId
                    existingVersions.each { def existingVersion ->
                        if (desiredVersion.equals(existingVersion.name)) {
                            versionId = existingVersion.id
                        }
                    }
                    if (versionId) {
                        println "Updating inventory for " + componentName
                        resourceClient.createResourceInventoryEntry(deploymentRequestId, childResource.id, componentId, versionId, 'Active')
                    }
                }
            }
        }
    }
}

/**
 *  @param imageName Name of a container image (which may be in the format namespace/image)
 *  @return The image name with the namespace and slash removed, if they were present
 */
private String getImageNameWithoutNamespace(String imageName) {
    String imageNameWithoutNamespace = imageName;
    int indexOfSlash = imageName.indexOf("/");
    if (indexOfSlash != -1) {
      imageNameWithoutNamespace = imageName.substring(indexOfSlash+1);
    }
    return imageNameWithoutNamespace;
}

/**
 *  @param name The name of the component to create
 *  @param sourceProps The list of source properties to set for the component
 *  @param componentProps The list of properties to set for the component
 */
private def createComponent(def client, String name, def sourceProps, def componentProps, def configType, def template) {
    def result
    def image = sourceProps['DockerTagImport/dockerImageName']
    try {
        result = KubernetesHelper.parseTextAsJson(client.getComponent(name)).id
    }
    catch (IOException e){
        println "Creating component: ${name}"
        result = client.createComponent(
                 name,
                 "Kubernetes image: ${image}",
                 configType,
                 'FULL',
                 template,
                 -1,
                 true,
                 false,
                 sourceProps)
    }

    componentProps.each { componentProp ->
        client.setComponentProperty(result.toString(), componentProp.key, componentProp.value, false)
    }

    return result
}
