/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Deploy
 * (c) Copyright IBM Corporation 2016. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */

package com.urbancode.air.plugin.osbcm

import javax.management.JMX
import javax.management.MalformedObjectNameException
import javax.management.ObjectName

import org.apache.xmlbeans.XmlException

import com.ibm.issr.iet.weblogic.WebLogicConnection
import com.ibm.issr.iet.weblogic.WebLogicMBeanServer
import com.bea.wli.config.customization.Customization
import com.bea.wli.config.importexport.ImportResult
import com.bea.wli.config.Ref
import com.bea.wli.config.resource.Diagnostics
import com.bea.wli.sb.management.configuration.ALSBConfigurationMBean
import com.bea.wli.sb.management.configuration.SessionManagementMBean
import com.bea.wli.sb.management.importexport.ALSBImportPlan
import com.bea.wli.sb.management.importexport.ALSBJarInfo

public class OsbcmHelper {
    private static WebLogicConnection connection
    private static SessionManagementMBean session
    private static ALSBConfigurationMBean configuration
    private static String sessionName

    /**
    *  @param hostname The hostname of the WebLogic server
    *  @param port The port number of the WebLogic server
    *  @param user The user id to authenticate with OSB
    *  @param password the password to authenticate with OSB
    *  @param timeout The maximum amount of milliseconds to keep the connection open
    *  @return True if connected successfully, false otherwise
    */
    public static boolean createSession(String hostname, String port, String user, String password, String timeout) {
        println ('[action]  Connecting to WebLogic...')
        try {
            connection = new WebLogicConnection(hostname, port, user, password, WebLogicMBeanServer.RUNTIME, timeout)
        }
        catch (MalformedURLException e) {
            println ("[error]  Could not connect to Weblogic due to bad URL: ${e.getMessage()}")
            return false
        }
        catch (IOException e) {
            println ("[error]  Could not connect to Weblogic: ${e.getMessage()}")
            return false
        }
        println ('[ok]  Weblogic connection aquired.')
        println ('[action]  Activating MBean session...')
        try {
            session = JMX.newMBeanProxy(connection.getServerConnection(),
                                        ObjectName.getInstance(SessionManagementMBean.OBJECT_NAME),
                                        SessionManagementMBean.class)
            sessionName = "session_" + System.currentTimeMillis()
            session.createSession(sessionName)
        }
        catch (MalformedObjectNameException e) {
            println ("[error]  Could not create bean: ${e.getMessage()}")
            println ('[possible solution]  It\'s likely that your version of WebLogic OSB is unsupported')
            return false
        }
        catch (NullPointerException e) {
            println ("[error]  Could not create bean: ${e.getMessage()}")
            return false
        }
        catch (Exception e) {
            println ("[error]  Could not create session: ${e.getMessage()}")
            return false
        }
        println ('[ok]  MBean session activated.')
        return true
    }

    /**
    *  @return True if configured successfully, false otherwise
    */
    public static boolean configureSession() {
        println ('[action]  Aquiring ALSB Configuration...')
        String alsbConfigBean = "com.bea:Name=${ALSBConfigurationMBean.NAME}.${sessionName},Type=${ALSBConfigurationMBean.TYPE}"
        try {
            configuration = JMX.newMBeanProxy(connection.getServerConnection(),
                                              ObjectName.getInstance(alsbConfigBean),
                                              ALSBConfigurationMBean.class)
        }
        catch (MalformedObjectNameException e) {
            println ("[error]  Could not create ALSB connection due to bad name: ${e.getMessage()}")
            return false
        } catch (NullPointerException e) {
            println ("[error]  Could not create ALSB connection: ${e.getMessage()}")
            return false
        }
        println ('[ok]  ALSB configuration created.')
        return true
    }

    /**
    *  @return True if activated successfully, false otherwise
    */
    public static boolean activateSession() {
        println ('[action]  Attempting to activate session...')
        try {
            session.activateSession(sessionName, "Session activated by UCD agent.");
        }
        catch (Exception e) {
            println ("[error]  Could not activate session: ${e.getMessage()}")
            return false
        }
        println ('[ok]  Session activated.')
        return true
    }

    /**
    *
    */
    public static void closeConnection() {
        PrintStream originalStream = System.out
        PrintStream dummyStream    = new PrintStream(new OutputStream(){
            public void write(int b) {
                // No Output
            }
        })

        println ('[action]  Closing connection...')
        if (connection) {
            System.setOut(dummyStream)    // prevent printout from WebLogicConnection class
            connection.closeConnection()
            System.setOut(originalStream)    // reset output to print out normally
            println ('[ok]  Connection closed.')
        }
        else {
            println ('[ok]  Connection already closed.')
        }
    }

    /**
    *  @param configFilePaths The comma separated list of configuration jars to loadConfigurations
    *  @param fastFail True to stop after first failed jar, false to process all jars even if some fail
    *  @return True if no failures, false otherwise
    */
    public static boolean loadConfigurations(String configFilePaths, boolean fastFail) {
        boolean failure = false
        configFilePaths.split(',').each { configFilePath ->
            if (!failure || !fastFail) {
                ImportResult importResult
                println ("[action]  Loading jar ${configFilePath} into the server...")
                File configFile = new File (configFilePath.trim())
                if (!configFile.isFile()) {
                    println ("[error]  Configuration file ${configFilePath} does not exist.")
                    println ('[possible solution]  Please ensure all Configuration JAR Files exist in the step configuration.')
                    failure = true
                    return false    // continue to next jar
                }
                byte[] bytes
                try {
                    bytes = new byte[(int) configFile.length()]
                    DataInputStream inputStream = new DataInputStream(new FileInputStream(configFile))
                    inputStream.readFully(bytes)
                    inputStream.close()
                }
                catch (IOException e) {
                    println("[error]  Could not read jar into bytes: ${e.getMessage().replace('\n', '')}")
                    failure = true
                    return false    // continue to next jar
                }
                try {
                    configuration.uploadJarFile(bytes)
                }
                catch (Exception e) {
                    println ("[error]  Could not upload jar bytes: ${e.getMessage()}")
                    failure = true
                    return false    // continue to next jar
                }
                println ('[ok]  Jar loaded.')
                try {
                    ALSBJarInfo jarInfo = configuration.getImportJarInfo()
                    ALSBImportPlan importPlan = jarInfo.getDefaultImportPlan()
                    importResult = configuration.importUploaded(importPlan)
                }
                catch (Exception e) {
                    println ("[error]  Could not import configuration: ${e.getMessage().replace('\n', '')}")
                    failure = true
                    return false    // continue to next jar
                }

                if (importResult.getImported().size() > 0) {
                    println ('[ok]  The following resources were imported:')
                    importResult.getImported().each { resource ->
                        println ("\t${resource}")
                    }
                }
                else {
                    println ('[error]  The following resources failed to import:')
                    Map<Ref, Diagnostics> failureMap = importResult.getFailed()
                    failureMap.keySet().each { key ->
                        println ("\t${key} - ${failureMap.get(key).toString()}")
                    }
                }
            }
            else {
                println ('[warning]  Aborting after first failure because \'Abort On First Failure\' was checked.')
            }
        }
        return !failure
    }

    /*
    *  @param custFilePath The path of the configuration XML file to apply
    *  @return True if the configuration is applied successfully, false otherwise
    */
    public static boolean loadCustomization(String custFilePath) {
        println ('[action]  Applying customization file to the server...')
        File custFile = new File(custFilePath)
        if (!custFile.isFile()) {
            println ("[error]  Customization file ${custFile} does not exist.")
            println ('[possible solution]  Please update the step configuration with a valid Customization XML File.')
            return false
        }
        FileInputStream is
        try {
            is = new FileInputStream(custFile)
        }
        catch (IOException e) {
            println ("[error]  Could not initialize file stream: ${e.getMessage()}")
            return false
        }
        catch (SecurityException e) {
            println ("[error]  Could not initialize file stream: ${e.getMessage()}")
            return false
        }
        List<Customization> custList
        try {
            custList = Customization.fromXML(is)
        }
        catch (IOException e) {
            println ("[error]  Could not create Customization from file: ${e.getMessage()}")
            return false
        }
        catch (XmlException e) {
            println ("[error]  Malformed XML: ${e.getMessage()}")
            println ('[possible solution]  Ensure the configuration file contains valid XML.')
            return false
        }
        custList.removeAll([null])
        try {
            configuration.customize(custList)
        }
        catch (Exception e) {
            println ("[error]  Could not apply customization: ${e.getMessage()}")
            return false
        }
        println ('[ok]  Customization applied.')
        return true
    }
}