<#
Licensed Materials - Property of IBM Corp.
IBM UrbanCode Deploy
(c) Copyright IBM Corporation 2015, 2016. All Rights Reserved.

U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
GSA ADP Schedule Contract with IBM Corp.
#>

# Sets path parameters for the input/output Java properties files that are passed through the command execution
Param (
  [string]$_IBMUCDPathToPluginInputPropsFile,
  [string]$_IBMUCDPathToPluginOutputPropsFile
)

$_IBMUCDErrorIterator = $Error.Count
$_IBMUCDOutputProperties = @{}

Write-Host ""

Function _IBMUCDWriteErrorToHost {
    if ($_IBMUCDErrorIterator -lt $Error.Count) {
        Write-Host $Args[0]
        While ($_IBMUCDErrorIterator -lt $Error.Count) {
            $global:_IBMUCDErrorIterator++
        }
        Write-Host ""
    }
}

# Accepts all valid single line PowerShell Code and passes output to UCD as string
# Builds and saves output Java properties file to the pass in path (OutputPath)
Function Set-StepOutputProperty () {
    <#
          .Synopsis
          Sets step output property in the Deploy server.

          .Description
          Use the funtion to pass information from your scripts to Deploy throught the creation
          of step output properties.

          .parameter Name
          The the name or key of the output property. If the property already exists then it Will
          be overwritten.

          .parameter Value
          The value of the output property.

          .Example
          Set-StepOutputProperty "Deploy" "Is good"

          Description
          -----------
          The function creates a step output property 'Deploy' with a value 'Is Good'
    #>
    [CmdletBinding()]
    param (
        [Parameter(
            Mandatory=$True,
            HelpMessage='Name of the Step output property.')]
        [String] $Name,
        [Parameter(
            Mandatory=$False,
            HelpMessage='Value of the Step output property.')]
        [String] $Value
    )
    $script:_IBMUCDOutputProperties.Set_Item($Name, $Value)
}

Function _IBMUCDStoreOutputProperties {
    $ArrayContainingOutputsToUCD = @()
    foreach($OutputProperty in $script:_IBMUCDOutputProperties.GetEnumerator()) {
        $ArrayContainingOutputsToUCD = $ArrayContainingOutputsToUCD + "$($OutputProperty.Name)=$($OutputProperty.Value)"
    }

    Set-Content $_IBMUCDPathToPluginOutputPropsFile $ArrayContainingOutputsToUCD
}

# Accepts all valid .properties files
# Grabs a Java props file (assumes ASCll encoding), and converting it to a hash-map object System.Collections.Hash-table
$_IBMUCDInputsHashTable = ConvertFrom-StringData (Out-String -InputObject (Get-Content $_IBMUCDPathToPluginInputPropsFile))
_IBMUCDWriteErrorToHost "ERROR  An Error has been encountered importing plug-in Properties. Please contact IBM UrbanCode support and ask for L3 support.`n"

# Accepts all valid PowerShell Script
# Executes PowerShell Script in Local Scope
Write-Host "ACTION  Executing PowerShell script.`n"
. ([ScriptBlock]::Create($_IBMUCDInputsHashTable.powerShellScript))
Write-Host ""
_IBMUCDWriteErrorToHost "ERROR  Your script encounter error(s). Please contact your script developer for assistance and relay this output.`n"
Write-Host "OK Script executed successfully.`n"

_IBMUCDStoreOutputProperties

# Will fail if with 1 (see post-processing) if any errors occurred (except if user cleared errors upon script exit)
# Error Handling
if ($_IBMUCDErrorIterator -eq 0) {
    Write-Host "OK  Plug-in executed successfully."
    exit $_IBMUCDInputsHashTable.successfulExitStatus
}
Write-Host "ERROR  Script executed with errors!"
