<#
Licensed Materials - Property of IBM* and/or HCL**
UrbanCode Deploy
(c) Copyright IBM Corporation 2015, 2017. All Rights Reserved.
(c) Copyright HCL Technologies Ltd. 2018. All Rights Reserved.

U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
GSA ADP Schedule Contract with IBM Corp.

* Trademark of International Business Machines
** Trademark of HCL Technologies Limited
#>

# Sets path parameters for the input/output Java properties files that are passed through the command execution
Param (
  [string]$_IBMUCDPathToPluginInputPropsFile,
  [string]$_IBMUCDPathToPluginOutputPropsFile
)

$_IBMUCDOutputProperties = @{}

Write-Host ""

# Check for an error and throw message if one occurred
Function _IBMUCDErrorCheck ($errorVar) {
    if ($errorVar.Count) {
        throw $errorVar
    }
}

# Accepts all valid single line PowerShell Code and passes output to UCD as string
# Builds and saves output Java properties file to the pass in path (OutputPath)
Function Set-StepOutputProperty () {
    <#
          .Synopsis
          Sets a step output property on the UrbanCode Deploy server.

          .Description
          Use this function to pass information from scripts to UrbanCode Deploy with step output properties.

          .parameter Name
          The name or key of the output property. If the property already exists, it will be overwritten.

          .parameter Value
          The value of the output property.

          .Example
          Set-StepOutputProperty "Deploy" "Is good"

          Description
          -----------
          The function creates a step output property 'Deploy' with a value 'Is Good'
    #>
    [CmdletBinding()]
    param (
        [Parameter(
            Mandatory=$True,
            HelpMessage='Name of the Step output property.')]
        [String] $Name,
        [Parameter(
            Mandatory=$False,
            HelpMessage='Value of the Step output property.')]
        [String] $Value
    )
    $script:_IBMUCDOutputProperties.Set_Item($Name, $Value)
    _IBMUCDStoreOutputProperties
}

Function _IBMUCDStoreOutputProperties {
    $ArrayContainingOutputsToUCD = @()
    foreach($OutputProperty in $script:_IBMUCDOutputProperties.GetEnumerator()) {
        $ArrayContainingOutputsToUCD = $ArrayContainingOutputsToUCD + "$($OutputProperty.Name)=$($OutputProperty.Value)"
    }

    Set-Content $_IBMUCDPathToPluginOutputPropsFile $ArrayContainingOutputsToUCD
}

# Accepts all valid .properties files
# Grabs a Java props file (assumes ASCll encoding), and converting it to a hash-map object System.Collections.Hash-table
$_IBMUCDInputsHashTable = ConvertFrom-StringData -ErrorVariable propError -StringData (Out-String -ErrorVariable propError -InputObject (Get-Content $_IBMUCDPathToPluginInputPropsFile))
_IBMUCDErrorCheck $propError

# Set successful exit status to be used in post processing script
$SuccessfulExitStatus = $_IBMUCDInputsHashTable.successfulExitStatus
Set-StepOutputProperty "successfulExitStatus" "$SuccessfulExitStatus"

# Accepts all valid PowerShell Scripts
$scriptBlock = [ScriptBlock]::Create($_IBMUCDInputsHashTable.powerShellScript)

# Executes PowerShell script block in Local Scope
Write-Host "[Action] Executing PowerShell script.`n"
try {
    Invoke-Command -ErrorVariable invokedError -ScriptBlock $scriptBlock

    # Only throws an error if the script block fails to execute
    _IBMUCDErrorCheck $invokedError
}
catch {
    throw "PowerShell script execution failed. Error Message: $($_.Exception.Message)`n"
}

# Won't be reached if an error is thrown or an exit code is returned
exit $successfulExitStatus