/*
 * Licensed Materials - Property of HCL
 * UrbanCode Deploy
 * (c) Copyright HCL Technologies Ltd. 2018. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * Trademark of HCL Technologies Limited
 */
package com.urbancode.air.plugin.sybase

import com.urbancode.air.CommandHelper
import com.urbancode.air.ExitCodeException

class ISQLCmdHelper {
    private String sybaseProfile, username, password, server, database, failure, defaultShell
    private List<String> additionalArgs
    private File isqlCommand

    public ISQLCmdHelper(
        File isqlCommand,
        String sybaseProfile,
        String username,
        String password,
        String server,
        String database,
        String failure,
        List<String> additionalArgs,
        String defaultShell)
    {
        this.isqlCommand = isqlCommand
        this.sybaseProfile = sybaseProfile
        this.username = username
        this.password = password
        this.server = server
        this.database = database
        this.failure = failure
        this.additionalArgs = additionalArgs
        this.defaultShell = defaultShell
    }

    public int runSQLFiles(List<File> sqlFiles) {
        CommandHelper commandHelper = new CommandHelper()
        int status = 0 // Return status of the command
        List<String> failures = [] // List of all SQL files that returned errors

        commandHelper.ignoreExitValue(true) // Causes non-zero exit codes to return

        for (File sqlFile in sqlFiles) {
            def cmdArgs = [
                isqlCommand.absolutePath,
                "-U",
                username,
                "-P",
                password,
                "-X",
                "-S",
                server,
                "-D",
                database,
                "-i",
                sqlFile.absolutePath,
                "--retserverror"
            ]

            if (sybaseProfile) {
                // handle Unix sourcing
                if (defaultShell) {
                    cmdArgs = [
                        defaultShell,
                        "-c",
                        "source ${sybaseProfile} && ${cmdArgs.join(" ")}"
                    ]

                }
                //handle Windows sourcing
                else {
                    // TO DO
                }
            }

            /* Add additional arguments if specified */
            for (String additionalArg in additionalArgs) {
                for (arg in additionalArg.split(' ', 2)) {
                    cmdArgs << arg
                }
            }

            try {
                status = commandHelper.runCommand("[Action] Executing SQL file: ${sqlFile.getName()}", cmdArgs)
                println()
            }
            catch (RuntimeException ex) {
                println("[Error] Empty argument or failed security check after command execution: "
                    + "${cmdArgs.join(' ')}.")
                throw ex
            }
            catch (IOException ex) {
                println("[Error] I/O operations have failed after command execution: "
                    + "${cmdArgs.join(' ')}.")
                throw ex
            }

            /* Determine which action to take based on the 'failure' property */
            if (status != 0) {
                if ("warn".equals(failure)) {
                    print("[Warning] ")
                }
                else if ("failAfter".equals(failure)) {
                    print("[Error] ")
                    failures << sqlFile.getName()
                }
                else {
                    print("[Error] SQL file '${sqlFile.getName()}' returned an error.")
                    return status // Return immediately without executing other files
                }

                println("SQL file '${sqlFile.getName()}' returned an error.")
            }
        }

        if ("failAfter".equals(failure) && !failures.isEmpty()) {
            println("\n[Error] The following SQL files returned errors:")
            for (String fileName in failures) {
                println("\t" + fileName)
            }

            return 1
        }

        /* Return success if no error occurred or if warning failure mode selected */
        return 0
    }
}