/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2016. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/

/*
 * ISAM REST API Documentation: https://www.ibm.com/support/knowledgecenter/SSPREK_8.0.1.3/com.ibm.isamw.doc/develop/api_web/index.html?lang=en
 */

package com.urbancode.air.plugin.isam

import groovy.json.JsonSlurper
import org.apache.http.client.methods.HttpPost
import org.apache.http.client.utils.URIBuilder
import org.apache.http.entity.StringEntity
import org.apache.http.HttpResponse
import org.apache.http.impl.client.DefaultHttpClient
import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder

/**
 * Helper class for the ISAM Plugin. Makes basic REST calls that assist in
 * ISAM actions. In addition, several functions are made to help format the correct
 * URI for the REST calls.
 */
public class ISAMHelper {

    /**
     * Empty constructor
     */
    public ISAMHelper() {
    }

    /**
     * Constructs and returns the formatted address for the specified REST call. ISAM REST calls
     * constructed in this function will have the following structure:
     * <host>://<port>/api/<resource>
     *
     * @param hostname The full URL of the ISAM hostname
     * @param resource  The data entity that will be defined by the target
     * @return          The fully constructed address for the REST call
     */
    public String createAPIAddress(String hostname, String resource) {
        return hostname + "/wga/" + resource
    }

    /**
     * Confirm that the Hostname is properly formatted. All URIs must have a Scheme (http/https),
     * Host (www.ibm.com), and Port (8080). Otherwise, crash with invalid Hostname error
     *
     * @param hostname The String representation of the ISAM's hostname/URI
     * @return         String that begins with http, otherwise https:// will be appended to the front. If hostname
     *                 and port cannot be found, then crash
     */
    public String checkHostname(String hostname) {
        URI uri
        try {
            uri = new URI(hostname)
        } catch (URISyntaxException ex) {
            throw new URISyntaxException("[Error]  Invalid syntax for the Hostname property. Please confirm Hostname follows this example: 'https://ISAM-server:8080'")
        }
        URIBuilder builder = new URIBuilder()
        // Confirm valid Scheme was given
        if (uri.getScheme() == "http" || uri.getScheme() == "https"){
            builder.setScheme(uri.getScheme())
        }
        else {
            builder.setScheme("https")
            println "[Warning]  Valid Scheme was not found. 'https' will be used as the default."
        }
        // Confirm valid Host was given
        if (uri.getHost()) {
            builder.setHost(uri.getHost())
        }
        else {
            throw new URISyntaxException("[Error]  No Host was given (ex. www.ibm.com:8080). Please confirm within the Hostname property.")
        }
        // Confirm valid Port was given
        if (uri.getPort()) {
            builder.setPort(uri.getPort())
        }
        else {
            throw new URISyntaxException("[Error]  No Port was given (ex. www.ibm.com:8080). Please confirm within the Hostname property.")
        }
        def result = builder.toString()

        // Return only the hostname, ignore all extra paths
        println "[Ok]  Hostname found: ${result}"
        return result
    }

    /**
     * Returns the DefaultHttpClient object that will connect and authenticate with the ISAM Server
     * Unless switching accounts, this object can be used for all REST calls. In addition, it is
     * set with Preemptive Authentication to send the authenticating message early to reduce overhead.
     * All certificates are accepted.
     *
     * @param username String representation of the username used to login into ISAM
     * @param password String representation of the password used to login into ISAM
     * @return         Object contains the authentication information to connect with ISAM
     * @see            DefaultHttpClient
     */
    public DefaultHttpClient createClient(String username, String password) {
        DefaultHttpClient client = new DefaultHttpClient()
        HttpClientBuilder clientBuilder = new HttpClientBuilder()
        clientBuilder.setUsername(username)
        clientBuilder.setPassword(password)
        clientBuilder.setPreemptiveAuthentication(true)
        com.urbancode.air.XTrustProvider.install()
        clientBuilder.setTrustAllCerts(true)

        client = clientBuilder.buildClient()
        return client
    }

     /**
      * Calls a REST POST request and returns the HTTPResponse
      *
      * @param client    The DefaultHttpClient that contains the necessary authentication information.
      * @param fullUrl   The complete URL address of the REST POST call.
      * @param jsonData   The data structure that contains JSON information for the post request.
      * @return          The HTTPResponse for the POST REST Call.
      */
     public HttpResponse doHttpPost(DefaultHttpClient client, String fullUrl, def jsonData){
        // Form JSON for the POST REST call
        StringEntity issueEntity
        if (jsonData) {
             issueEntity = new StringEntity(jsonData)
        }
        else {
            throw new IllegalArgumentException("[Error]  No JSON input was given.")
        }

        issueEntity.setContentType("application/json")

        // Create HttpPost Request and append JSON
        URIBuilder builder = new URIBuilder(fullUrl)
        builder.setParameter("Accept", "application/json")
        //builder.setParameter("Accept-Charset", "UTF-8")
        builder.setParameter("Content-Type", "application/json")
        URI uri = builder.build()
        HttpPost postRequest = new HttpPost(uri)
        postRequest.setEntity(issueEntity)

        // Execute the REST Call
        HttpResponse response
        try {
            response = client.execute(postRequest)
        }
        catch (IOException ex) {
            throw new IOException("[Error]  Unable to access the ISAM Server. Check the hostname: ${ex.getMessage()}")
        }
        return response
     }

    /**
     * Uses the REST Post request. A status code of 200 is expected when the add comment is successful.
     * The call will fail if a non-200s status code is returned.
     *
     * @param client    The DefaultHttpClient that contains the necessary authentication information.
     * @param fullAddress The full http address used to call the POST request
     * @param jsonData  The String data structure that contains the comment
     * @return          Return REST Status Code
     */
     public int updateConfigEntity(DefaultHttpClient client, String fullAddress, String jsonData){
        HttpResponse response = doHttpPost(client, fullAddress, jsonData)
        def statusLine = response.getStatusLine()
        def statusCode = statusLine.getStatusCode()
        println "[Ok]  StatusCode: " + statusCode

        // Analyze returned status code...
        if (statusCode == 200) {
           println "[Ok]  Reverse Proxy: Update Configuration Entity was successful."
        }
        else if (statusCode == 400) {
           println "[Error]  Skipping Update Configuration Entity: There is a problem with the request. View the below message for more information."
           println "[Error]  Status Code: ${statusCode} - Message:\n${response.entity?.content?.text}"
        }
        else if (statusCode == 404) {
           println "[Error]  Skipping Update Configuration Entity: The request service does not exist. View the below message for more information."
           println "[Error]  Status Code: ${statusCode} - Message:\n${response.entity?.content?.text}"
        }
        else if (statusCode == 500) {
           println "[Error]  Skipping Update Configuration Entity: An internal (ISAM) error occurred. View the below message for more information."
           println "[Error]  Status Code: ${statusCode} - Message:\n${response.entity?.content?.text}"
        }
        else {
           println "[Error]  Skipping Update Configuration Entity: Unexpected Status Code. View the below message for more information."
           println "[Error]  Status Code: ${statusCode} - Message:\n${response.entity?.content?.text}"
        }
        return statusCode
     }
}