/* REXX */
/*%STUB CALLCMD*/
/*********************************************************************/
/* Copyright:    Licensed Materials - Property of IBM and/or HCL     */
/*                                                                   */
/*        Copyright IBM Corporation. All rights reserved             */
/*        Copyright HCL Technologies limited. All rights reserved    */
/*                                                                   */
/*        US Government Users Restricted Rights -                    */
/*        Use, duplication or disclosure restricted by               */
/*        GSA ADP Schedule Contract with IBM Corp.                   */
/*                                                                   */
/*********************************************************************/

   Call syscalls('ON')

   parse arg '"'manifest'"' '"'restore'"',
                '"'traceOption'"' '"'TempDSN'"' '"'binloc'"',
                '"'sysoutValue'"' '"'tempUnit'"' '"'tempVolSerValue'"'.

   if traceOption == 'true' then interpret 'trace a'

   /* Set Sysout class and temp volser */
   sysoutClass = ''
   tempVolser  = ''
   If sysoutValue /= "" Then sysoutClass = 'SYSOUT('sysoutValue')'
   If tempVolSerValue /= "" Then tempVolser  = 'VOLUME('tempVolSerValue')'

   Say 'Manifest file :' manifest
   Say 'Unpacked location :' binloc
   Say 'Restore Mapping File :' restore

   call loadManifestIntoStem(manifest)
   call loadContainerMapperIntoStem(restore)
   call createSourceTargetStemMap
   Call processManifestContainers

Exit

loadManifestIntoStem: procedure expose manlist.
  parse arg manifest
     Address syscall "readfile (manifest) manlist."
   If retval < 0 Then
   Do
       Say "Problem reading manifest file : "manifest". " ||,
           "Errno : "errno" Reason : "right(errnojr,8,0)"."

      Call Exitproc(8)
   End
Return

loadContainerMapperIntoStem: procedure expose maplist.

  parse arg mapperfile
  Address syscall "readfile (mapperfile) maplist."
  If retval < 0 Then
  Do
    Say "Problem reading mapping file : "mapperfile". " ||,
      "Errno : "errno" Reason : "right(errnojr,8,0)"."

    Call Exitproc(8)
  End

  if maplist.0 = 0 then
  do
    Say 'No mapping found in the mapping file' restore
    Call Exitproc(8)
  end

return

createSourceTargetStemMap: procedure expose maplist. containerMap.

  drop containerMap.
  Do rest = 1 to maplist.0
    If Pos('<sourceContainer',maplist.rest) > 0 Then
    Do
      Parse var maplist.rest . ' name="' sourceDsn '"' .
      rest = rest + 1
      If Pos('<targetContainers>', maplist.rest) = 0 Then
      Do
        Say 'Target dataset not found for 'sourceDsn
        Call Exitproc(8)
      End

      rest = rest + 1
      targets = ''
      do while pos('</targetContainers>', maplist.rest) = 0
        if pos('<targetContainer ', maplist.rest) > 0 then
        do
          Parse var maplist.rest . ' name="' targetDsn '"' .
          targets = targets targetDsn
        end
        rest = rest + 1
      end
      sourceDsn = Strip(sourceDsn)
      targets = Strip(targets)
      containerMap.sourceDsn = targets
    End
  End

Return

processManifestContainers :

   isDeleted = 0
   Do i = 1 to manlist.0
     Select
       When (Pos('<deleted>',manlist.i) > 0) Then
         isDeleted = 1

       When (Pos('</deleted>',manlist.i) > 0) Then
         isDeleted = 0

       When (Pos('<container',manlist.i) > 0) Then
       Do
         Parse var manlist.i . ' name="' DataSet '" type="' type '"' .

         drop memberlist.
         memberlist.0 = 0
         memberindex = 0
         i = i + 1
         Do while Pos('<resource',manlist.i) > 0
            Parse var manlist.i . '<resource' . ' name="' member '"' .
            /* Need to change the XML format chars to real chars         */
            member = chkmem3(member)          
            /* change characters to wildcards only for IEBCOPY           */
            if (isDeleted == 0) then 
              member = chkmem2(member)

            memberindex = memberindex + 1
            memberlist.memberindex = member
            memberlist.0 = memberindex
            i = i + 1           
         End
         Call performDeployment dataset type isDeleted
         i = i - 1
       End
       Otherwise
         Nop
     End
   End

Return

performDeployment:
  parse arg sourceDsn type isDeleted

  if isDeleted Then
  Do
    targets = getTargetDatasetsFromContainerMap(sourceDsn)
    do g = 1 to words(targets)
      DeplDSN = word(targets, g)
      If type = 'sequential' then
        call deleteDataset(DeplDSN)
      else
        call deletePdsMembers(DeplDSN)
    End
  end
  else do
    if type = 'sequential' then
      call copySequentialDataset sourceDsn
    else
      call copyPdsMembers sourceDsn
  end

Return

copySequentialDataset:
  Parse arg sourceDsn

  binFilePath = binloc"/"sourceDsn".bin"
  call copyHFSBinaryToSequentialDataset tempUnit, tempVolser, binFilePath

  targets = getTargetDatasetsFromContainerMap(sourceDsn)
  do p = 1 to words(targets)
    DeplDSN = word(targets, p)
    Say 'Deploying sequential dataset to 'DeplDSN'.'
    call receiveSeqfileIntoDataset DeplDSN
  end

return

copyPDSMembers :

  Parse arg sourceDsn

  binFilePath = binloc"/"sourceDsn".bin"
  call copyHFSBinaryToSequentialDataset tempUnit, tempVolser, binFilePath

  call receiveSeqfileIntoDataset TempDSN

  call freeIebcopyDDs

  Address TSO "ALLOC F(SYSPRINT) NEW REUSE"

  Address TSO "ALLOC F(INDD)  DA('"TempDSN"') SHR REUSE"
  If (rc > 0) then
  Do
    say 'Problem allocating dataset 'TempDSN'. Return code : 'rc
    call ExitProc(8)
  End

  Address TSO "ALLOC F(SYSUT4) NEW" ||,
          " CYLINDERS UNIT("tempUnit") SPACE(5 10) "tempVolser

  Address TSO "ALLOC F(SYSIN) NEW REUSE"

  call writeSysinRecordsForIebcopy

  targets = getTargetDatasetsFromContainerMap(sourceDsn)
  do r = 1 to words(targets)
    DeplDSN = word(targets, r)
    call createTargetDatasetIfNotExist sourceDsn DeplDSN TempDSN
    call runIebcopyProgram DeplDSN
  end

  call freeIebcopyDDs

  x = Msg('off')
    Address TSO "DELETE '"TempDSN"'"
  x = msg('on')

Return

writeSysinRecordsForIebcopy:

  DROP SYSLINE.
  SYSLINE.1 = " COPYGROUP OUTDD=OUTDD,INDD=((INDD,R))"
  Do memberindex = 1 to memberlist.0
    sysIndex = memberindex + 1
    SYSLINE.sysIndex = "         SELECT MEMBER="memberlist.memberindex
  end
  SYSLINE.0 = memberlist.0 + 1
  Address TSO "EXECIO * DISKW SYSIN (STEM SYSLINE. FINIS)"

return

createTargetDatasetIfNotExist:procedure

  parse arg sourceDsn DeplDSN TempDSN

  Address ISPEXEC "DSINFO DATASET('"DeplDSN"')"
  select
    when rc = 0 then
      nop
    when rc = 8 then do /* Dataset does not exist */
      Say 'Dataset' DeplDSN 'does not exist. Allocating dataset like 'sourceDsn
      call allocateTargetDataset DeplDSN TempDSN
    end
    otherwise do
      say 'DSINFO failed on 'DeplDSN'. Return code : 'rc
      call ExitProc(8)
    end
  end

return

runIebcopyProgram:

  parse arg targetDsn

  Address TSO "ALLOC F(OUTDD) DA('"targetDsn"') SHR REUSE"
  If (rc > 0) then
  Do
    say 'Problem allocating dataset 'targetDsn'. Return code : 'rc
    call ExitProc(8)
  End

  Say 'Deploying members to 'targetDsn'.'

  say 'IEBCOPY control statement'
  do x = 1 to SYSLINE.0
    say SYSLINE.x
  end

  Address ISPEXEC "ISPEXEC SELECT PGM(IEBCOPY)"
  If rc <> 0 Then
  Do
    Address TSO "EXECIO * DISKR SYSPRINT (FINIS STEM sysprint."
    Do sys = 1 to sysprint.0
      Say Strip(sysprint.sys)
    End
    call freeIebcopyDDs
    Call ExitProc(8)
  End

return

freeIebcopyDDs:

  x = Msg('off')
    Address TSO "FREE F(SYSIN)"
    Address TSO "FREE F(SYSPRINT)"
    Address TSO "FREE F(SYSUT4)"
    Address TSO "FREE F(OUTDD)"
    Address TSO "FREE F(INDD)"
  x = Msg('on')

return

receiveSeqfileIntoDataset:
  parse arg dataset

  x = PROMPT("ON")

  /* Allocate a temporary throw away log so we don't fill up LOG.MISC */
  Address TSO "ALLOC F(LOGFILE) NEW" ||,
      " CYLINDERS UNIT("tempUnit") "tempVolser" DSORG(PS)" ||,
      " BLKSIZE(3120) LRECL(255) RECFM(V B) SPACE(1 1)"
  Address TSO "ALLOC F(SYSUT4) NEW" ||,
          " CYLINDERS UNIT("tempUnit") "tempVolser" SPACE(5 10)"
  Address ISPEXEC "QBASELIB LOGFILE ID(LOGFILE)"

  XX=OUTTRAP('trapOut.')
    Queue " DATASET('"dataset"')" sysoutClass
    Address TSO "RECEIVE INFILE(SEQFILE) NONAMES LOGDS("LOGFILE")"
    receive_rc = rc
  XX=OUTTRAP('OFF')

  if receive_rc <> 0 Then
  Do
    Do xmit = 1 to trapOut.0
        Say trapOut.xmit
    End
    call freeReceiveDDs
    Call ExitProc(8)
  End

  call freeReceiveDDs

return

freeReceiveDDs:

  x = msg('off')
    Address TSO "FREE F(LOGFILE)"
    Address TSO "FREE F(SEQFILE)"
    Address TSO "FREE F(SYSUT4)"
  x = msg('on')

return

allocateTargetDataset:procedure

  parse arg targetDsn TempDSN

  blksize = getDatasetBlockSize(TempDSN)

  Address TSO "ALLOC DA('"targetDsn"') LIKE('"TempDSN"') BLKSIZE("blksize")"
  If (rc > 0) then
  Do
    say 'Problem allocating dataset 'targetDsn'. Return code : 'rc
    call ExitProc(8)
  End

return

getDatasetBlockSize:procedure

  parse arg dataset

  Address ISPEXEC "DSINFO DATASET('"dataset"')"
  If (rc > 0) then
  Do
    say 'DSINFO failed on 'dataset'. Return code : 'rc
    call ExitProc(8)
  End

  blockSize = Strip(zdsblk)

return blockSize

copyHFSBinaryToSequentialDataset:procedure

  parse arg tempUnit, tempVolser, binFilePath

  trk = getFileSizeInTracks(binFilePath)
  if trk < 26471 then
    dstype = "BASIC"
  else
    dstype = "LARGE"

  x = msg('off')
    Address TSO "FREE F(HFSFILE)"
    Address TSO "FREE F(SEQFILE)"
    Address TSO "FREE F(SYSPRINT)"
  x = msg('on')

  Address TSO "ALLOC F(SYSPRINT) DUMMY"
  Address TSO "ALLOC F(SEQFILE) NEW" ||,
          " TRACKS UNIT("tempUnit") "tempVolser" DSORG(PS)" ||,
          " BLKSIZE(3120) LRECL(80) RECFM(F B)" ||,
          " SPACE("trk" "trk") DSNTYPE("dstype")"
  Address TSO "ALLOC F(HFSFILE) PATH('"binFilePath"')"
  Address TSO "OCOPY INDD(HFSFILE) OUTDD(SEQFILE) BINARY"

  Address TSO "FREE F(HFSFILE)"

return

getFileSizeInTracks: procedure 

  parse arg filePath 

  shellcmd="ls -REgoa '"filePath"'"
  
  sh_rc = bpxwunix(shellcmd,,stdout.,stderr.)
  If sh_rc /= 0 Then do
    say "Command :" shellcmd "failed with rc :" sh_rc
    Call ExitProgram(sh_rc)
  end

  If stderr.0 > 0 Then
  Do
    Say '---STDERR---'
    Do e = 1 to stderr.0
        Say stderr.e
    End
  End
  If stdout.0 > 0 Then
  Do
    parse var stdout.1 stuff 21 sizeInBytes .
    if sizeInBytes <= 56664 Then
      return 1
    else
      return  Format(sizeInBytes/56664,,0)
  End
  else do 
    say "Unable to get size of the file: " filePath
    call ExitProc(8)
  end 
return

deleteDataset: procedure
  parse arg datasetName 
  var = "'"datasetName"'"
  ListdsiRC = Listdsi(var directory)
  If ListdsiRC <= 4 Then
  Do
    Say 'Deleting dataset 'datasetName
    Address TSO "DELETE '"datasetName"'"
  End
  else        
    say 'The dataset 'datasetName' was not deleted because it did not exist.'
Return

deletePdsMembers: procedure expose memberlist.

  parse arg dataset
  Address ISPEXEC "LMINIT DATAID(DID) DATASET('"dataset"') ENQ(SHRW)"
  select
    when rc = 0 then
      nop
    when rc = 8 then do
      Say 'Target dataset 'dataset' does not exist. Skipping member deletion.'
      return
    end
    otherwise do
      say 'The LMINIT function failed for dataset 'dataset'. Return code : 'rc
      call ExitProc(8)
    end
  end

  Address ISPEXEC "LMOPEN DATAID(&DID) OPTION(OUTPUT)"
  If (rc > 0) then
  Do
    Call freeDataset
    say 'The LMOPEN function failed for dataset 'dataset'. Return code : 'rc
    call ExitProc(8)
  End
 
  Do m = 1 to memberlist.0   
    call deletePdsMember dataset memberlist.m
  End

  Call closeDataset
  Call freeDataset

Return

deletePdsMember:

  parse arg dataset member
  Say 'Deleting the member 'member' from 'dataset'.'
  Address ISPEXEC "LMMDEL DATAID(&DID) MEMBER("member")"
  Select
    When (rc = 0) Then
        Nop
    When (rc = 8) Then
    Do
        Say 'The deletion of the member 'member' failed ' ||,
        'because member did not exist in the dataset' dataset
    End
    Otherwise
    Do
      Call closeDataset
      Call freeDataset
      say 'The deletion of the member 'member' failed. Return code : 'rc
      call ExitProc(8)
    End
  End

return

closeDataset:
  Address ISPEXEC "LMCLOSE DATAID(&DID)"
  If (rc <> 0) then
  Do
    say 'The LMCLOSE function failed with return code : 'rc
    call ExitProc(8)
  End
Return 

freeDataset:
  Address ISPEXEC "LMFREE DATAID(&DID)"
  If (rc <> 0) then
  Do
    say 'The LMFREE function failed with return code : 'rc
    call ExitProc(8)
  End
Return

chkmem2: Procedure

  /* change characters to wildcards for IEBCOPY           */

  Parse arg member
  Validchars = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ@#$'
  newMemb = ''

  Do mm = 1 to Length(member)
    char = Substr(member,mm,1)
    If Pos(char,Validchars) = 0 Then
      newMemb = newMemb||'%'
    Else
      newMemb = newMemb||char
  End

Return newMemb

chkmem3: Procedure

  /* change characters that may cause problems in the xml */

  Parse arg member
  nonValid = 'HexValueIs'
  newMemb = ''

  Do mm = 1 to Length(member)
    char = Substr(member,mm,10)
    If char = nonValid Then
    Do
       newMemb = newMemb || D2C(Substr(member,mm+10,03))
       mm = mm + 12
    End
    Else
      newMemb = newMemb||Substr(member,mm,1)
  End

Return newMemb

getTargetDatasetsFromContainerMap: procedure expose containerMap.
  parse arg sourceDataset
  if symbol('containerMap.sourceDataset') = 'VAR' then
    return containerMap.sourceDataset

  Say "No target dataset found for source dataset:- "sourceDataset
  call ExitProc(8)
Return

ExitProc :

   Parse arg exit_rc

   ZISPFRC = exit_rc
   Address ISPEXEC "VPUT (ZISPFRC) SHARED"

Exit exit_rc