#pragma nomargins nosequence
/*
 * Licensed Materials - Property of IBM and/or HCL Corp.
 * UrbanCode Deploy
 * (c) Copyright IBM Corporation 2002, 2016. All Rights Reserved.
 * (c) Copyright HCL Technologies Ltd. 2020. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <dynit.h>

#include "com_ibm_teamz_build_binder_jni_BinderInfo.h"

#define _IEW_TARGET_RELEASE _IEW_ZOSV2R4_

/********************************/
/* Binder's C/C++ Include Files */
/********************************/
#include <__iew_api.h>

char END_OF_STRING = '\0';
char *DD_NAME = "BUZTOOL";
const int LIST_SIZE = 1;

void throwBinderException(JNIEnv *env, char *errorMsg);

JNIEXPORT jbyteArray JNICALL Java_com_ibm_teamz_build_binder_jni_BinderInfo_getBinderInfo
(JNIEnv *env, jobject obj, jbyteArray dsArray, jbyteArray memArray, jbyteArray tempfArray)
{
  char buffer[200];
  size_t bufferSize = sizeof(buffer);

  struct program_inputs
  {
    char *dataset;
    char *member;
    char *filePath;
  } input;

  input.dataset = (*env)->GetByteArrayElements(env, dsArray, 0);
  int dsnLength = (*env)->GetArrayLength(env, dsArray);
  input.dataset[dsnLength] = END_OF_STRING;

  input.member = (*env)->GetByteArrayElements(env, memArray, 0);
  int memberLength = (*env)->GetArrayLength(env, memArray);
  input.member[memberLength] = END_OF_STRING;

  input.filePath = (*env)->GetByteArrayElements(env, tempfArray, 0);
  int fileLength = (*env)->GetArrayLength(env, tempfArray);
  input.filePath[fileLength] = END_OF_STRING;

  // Allocate Dataset and Member to a DD name
  __dyn_t ip;
  dyninit(&ip);
  ip.__ddname = DD_NAME;
  dynfree(&ip);
  ip.__dsname = input.dataset;
  ip.__status = __DISP_SHR;
  if (dynalloc(&ip) != 0)
  {
    snprintf(buffer, bufferSize, "Dynalloc error: Error code %d, info code %d. Dataset %s",
             ip.__errcode, ip.__infocode, ip.__dsname);
    throwBinderException(env, buffer);
  }

  // create file list for __iew_openW()
  char *files_keys[1] = {"PRINT"};
  void *files_values[1];
  files_values[0] = input.filePath;
  _IEWList *files_list = __iew_create_list(LIST_SIZE, files_keys, files_values);
  if (files_list == NULL)
  {
    snprintf(buffer, bufferSize, " create_list error: files list is null.");
    dynfree(&ip);
    throwBinderException(env, buffer);
  }

  // create exits list for __iew_openW()
  char *exits_keys[1] = {"SAVE"};
  void *exits_values[1];
  struct _exit_address
  {
    void *entry_point;
    void *user_data;
    void *severity_level;
  } exitsA = {NULL, NULL, 0};
  exits_values[0] = &exitsA;
  _IEWList *exits_list = __iew_create_list(LIST_SIZE, exits_keys, exits_values);
  if (exits_list == NULL)
  {
    snprintf(buffer, bufferSize, " create_list error: exits list is null.");
    dynfree(&ip);
    throwBinderException(env, buffer);
  }

  // open workmod session, load BINDER, create api context with
  // target release = <see above: #define _IEW_TARGET_RELEASE>, intent = access
  const char *parms = "MAP=Y,XREF=Y,CASE=MIXED,TERM=Y,LIST=ALL";
  unsigned int rc, reason;
  _IEWAPIContext *apiCnxt = __iew_openW(_IEW_TARGET_RELEASE, _IEW_ACCESS, files_list, exits_list, parms, &rc, &reason);
  if (apiCnxt == NULL)
  {
    snprintf(buffer, bufferSize, " openW error: apiCnxt is null.");
    dynfree(&ip);
    throwBinderException(env, buffer);
  }

  // initialize api flags
  _IEWAPIFlags apiflags = {0, 0, 0, 0, 0, 0, 0, 0, 0};

  // set api flags for __iew_includeName()
  apiflags.__imports = 1;
  apiflags.__aliases = 1;
  apiflags.__attrib = 1;
  // read in program object via __iew_includeName()
  int includeRc = __iew_includeName(apiCnxt, DD_NAME, input.member, apiflags);
  if (includeRc != 0)
  {
    int msgLength = snprintf(buffer, bufferSize,
                             " includeName error: rc=%u, rs=<0X%.8X>.",
                             includeRc, __iew_get_reason_code(apiCnxt));
    // set api flags for __iew_closeW()
    apiflags.__protect = 1;
    // close workmod session, delete api context
    _IEWAPIContext *retCnxt = __iew_closeW(apiCnxt, apiflags, &rc, &reason);
    if (retCnxt != NULL)
    {
      snprintf(buffer + msgLength, bufferSize - msgLength,
               " closeW error: context is not NULL. return code = %X, rs =<0X%.8X>",
               rc, reason);
    }
    dynfree(&ip);
    throwBinderException(env, buffer);
  }

  // class B_IDRB: Binder identification record, indicating the binder version, size, and
  // how and when the program object was created.
  _IEWBinderIDEntry *binderEntry;
  int count = __iew_getD(apiCnxt, "B_IDRB", NULL, NULL, (void **)&binderEntry);
  if (count == 0)
  {
    int msgLength = snprintf(buffer, bufferSize, " getD error: no data items returned.");
    // set api flags for __iew_closeW()
    apiflags.__protect = 1;
    // close workmod session, delete api context
    _IEWAPIContext *retCnxt = __iew_closeW(apiCnxt, apiflags, &rc, &reason);
    if (retCnxt != NULL)
    {
      snprintf(buffer + msgLength, bufferSize - msgLength,
               " closeW error: context is not NULL. return code = %X, rs =<0X%.8X>", rc, reason);
    }
    dynfree(&ip);
    throwBinderException(env, buffer);
  }

  // set api flags for __iew_closeW()
  apiflags.__protect = 1;
  // close workmod session, delete api context
  _IEWAPIContext *retCnxt = __iew_closeW(apiCnxt, apiflags, &rc, &reason);
  if (retCnxt != NULL)
  {
    snprintf(buffer, bufferSize,
             " closeW error: context is not NULL. return code = %X, rs =<0X%.8X>", rc, reason);
    dynfree(&ip);
    throwBinderException(env, buffer);
  }

  dynfree(&ip);

  // Module Bound (slash /) Module text size
  int msgLength = snprintf(buffer, bufferSize, "%s/%d", binderEntry->__idb_data_bound, binderEntry->__idb_module_size);
  jbyteArray msgByteArray = (*env)->NewByteArray(env, msgLength);
  (*env)->SetByteArrayRegion(env, msgByteArray, 0, msgLength, (jbyte *)buffer);
  return msgByteArray;
}

void throwBinderException(JNIEnv *env, char *errorMsg)
{
  char exceptionPath[] = "com/ibm/teamz/build/binder/jni/BinderException";
  __etoa(exceptionPath);
  jclass exceptionClass = (*env)->FindClass(env, exceptionPath);
  __etoa(errorMsg);
  (*env)->ThrowNew(env, exceptionClass, errorMsg);
}
