#!/usr/bin/env groovy
/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Release
 * (c) Copyright IBM Corporation 2014. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
import com.urbancode.air.*

import java.net.URI
import java.net.URL
import java.nio.charset.Charset
import java.util.Date
import java.text.SimpleDateFormat

import static com.urbancode.release.rest.framework.Clients.*
import com.urbancode.release.rest.models.Change
import com.urbancode.release.rest.models.Initiative
import com.urbancode.release.rest.models.Release
import com.urbancode.release.rest.models.Application
import com.urbancode.release.rest.models.ChangeType
import com.urbancode.release.rest.models.internal.PluginIntegrationProvider
import com.urbancode.release.rest.framework.Clients
import com.urbancode.release.rest.models.Change.Status
import com.urbancode.release.rest.models.Change.Severity

import com.rallydev.rest.RallyRestApi
import com.rallydev.rest.request.QueryRequest
import com.rallydev.rest.response.QueryResponse
import com.rallydev.rest.util.QueryFilter

final def workDir = new File('.').canonicalFile
def apTool = new AirPluginTool(this.args[0], this.args[1])
def props = apTool.getStepProperties()

//We launch the integration here
def integration = new RallyIntegration (props)
integration.releaseAuthentication()
integration.runIntegration ()

public class RallyIntegration {

    def integrationProviderId
    def provider
    def ucrClient
    def releaseToken
    def serverUrl
    def rallyUrl
    def rallyApiKey

    def mappingStatuses
    def mappingRelease
    def mappingTypes
    def mappingApplication
    def mappingSeverities
    def initiatives
    def slurper
    def rallyHelper
    def ucrChangeType
    int lastUpdated

    def changesByExternalId // Map of UCD ids to UCR Changes

    def ucrChangeTypeId
    def rallyInitiativeKey
    def rallyNewInitiative
    def rallyApplicationKey
    def rallyReleaseKey
    def allInitiativesNameId
    def allReleasesNameId
    def allApplicationsNameId

    def rallyType

    def externalIdField = "FormattedID"

    // Main Constructor for Rally Plugin
    RallyIntegration (props) {
        this.rallyType = props['rallyType']
        this.rallyInitiativeKey = props['rallyInitiativeKey']
        this.rallyApplicationKey = props['rallyApplicationKey']
        this.rallyNewInitiative = props['rallyNewInitiative']
        this.rallyReleaseKey = props['rallyReleaseKey']
        this.ucrChangeTypeId = ucrChangeTypeId = props['ucrChangeType']
        if(props['lastUpdated']) {
            try {
                this.lastUpdated = Integer.parseInt(props['lastUpdated'])
            }
            catch (NumberFormatException e) {
                println("Import Range must be an Integer or left blank!")
                System.exit(1)
            }
        }
        this.integrationProviderId = props['releaseIntegrationProvider']
        this.releaseToken = props['releaseToken']
        this.serverUrl = props['releaseServerUrl']
        this.rallyUrl = props['rallyUrl']
        if(rallyUrl.charAt(rallyUrl.length() -1) != '/') {
            this.rallyUrl = this.rallyUrl + '/'
        }
        this.rallyApiKey = props['rallyApiKey']

        this.mappingStatuses = props['mappingStatuses']
        this.mappingRelease = props['mappingRelease']
        this.mappingApplication = props['mappingApplication']
        this.mappingSeverities = props['mappingSeverities']
        this.mappingTypes = props['mappingTypes']
        this.initiatives = props['initiatives']
        this.slurper = new groovy.json.JsonSlurper()
    }

    //--------------------------------------------------------------
    //Authentication with Release
    def releaseAuthentication () {
        Clients.loginWithToken(serverUrl, releaseToken)
    }

    //--------------------------------------------------------------
    def init () {
        // Basic Rally Client Object
        this.rallyHelper = new RallyRestApi(new URI(rallyUrl), rallyApiKey)

        //We retrieve the full provider object
        this.provider = new PluginIntegrationProvider().id(integrationProviderId).get()

        //We need to make sure that all changes or initiatives imported from that integration won't be editable
        this.provider.disabledFields("change", "type", "name", "status", "release", "application", "description", "severity", "initiative")
        this.provider.disabledFields("initiative", "description", "name")

        this.provider.save()
    }

    //--------------------------------------------------------------
    def runIntegration () {
        //We load the integration provider and the rallyHelper library
        init()

        //We create or update each of them
        def countCreatedItem = 0
        def countWarning = 0

        //We set the last execution date
        def lastExecution = provider.getProperty("lastExecution")
        def lastExecutionDate = null
        if (lastExecution != null) {
            lastExecutionDate = new Date(lastExecution.toLong())
            println "Last Execution Date: "+lastExecutionDate
        }


        // Retrieve all Initiatives Name and Id from UCR
        def initEntity = new Initiative()
        initEntity.format("name")
        this.allInitiativesNameId = initEntity.getAll()

        // Retrieve all Applications Name and Id from UCR
        def appEntity = new Application()
        appEntity.format("name")
        this.allApplicationsNameId = appEntity.getAll()

        // Retrieve all Releases Name and Id from UCR
        def relEntity = new Release()
        relEntity.format("name")
        this.allReleasesNameId = relEntity.getAll()

        println "--------------------------------------------------------------"
        //We retrieve all changes for that provider from UCRelease
        def allChanges = change().getAllForIntegrationProvider(integrationProviderId)
        println ("Existing Changes for that Integration: "+allChanges.size())

        changesByExternalId = allChanges.collectEntries({ change -> [(change.externalId):change]})


        println("--------------------------------------------------------------")

        if(ucrChangeTypeId == null || ucrChangeTypeId.equals("")) {

            println("--------------------------------------------------------------")
            println("Please Select a UCR Change Type.")
            println("This will be the change type that is used for imported values")
            println("--------------------------------------------------------------")
            throw RunTimeException("No UCR Change Type was specified")
        }

        //Here we need to gather all Change Types created in Release in order to do the Mapping of Types
        ucrChangeType= changeType().id(ucrChangeTypeId).get()


        //List of changes from RALLY
        QueryRequest defectCount = new QueryRequest(rallyType)

        //If a last updated date is specified, filter the query to only import those dates
        if(lastUpdated) {
            Calendar calendar = GregorianCalendar.getInstance()
            calendar.add(Calendar.DAY_OF_YEAR, -lastUpdated)
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd")
            String filterSinceDate = dateFormat.format(calendar.getTime())
            QueryFilter filter = new QueryFilter("LastUpdateDate", ">=", filterSinceDate)

            defectCount.setQueryFilter(filter)
            println("Filtered Rally items to those updated since " + calendar.getTime())
        }

        int pageSize = 200
        defectCount.setPageSize(pageSize)
        int pageCount = 0
        boolean continueRunning = true
        int totalRallyChanges
        int numberCreated = 0
        int numberUpdated = 0

        //Loop through each page of changes, until all query results have been processed
        while (continueRunning) {
            //We need to save or create in bulk
            def bulkCreateChangeList = [] as List
            def bulkUpdateChangeList = [] as List

            defectCount.setStart(pageSize*pageCount + 1)
            pageCount++
            QueryResponse defectCountResponse = rallyHelper.query(defectCount)
            def responseErrors = defectCountResponse.getErrors()
            if(responseErrors.length > 0) {
                println(responseErrors)
                throw RunTimeException("Error in retrieving Rally entities")
            }

            def importedChangeList = slurper.parseText(defectCountResponse.getResults().toString())

            totalRallyChanges = defectCountResponse.getTotalResultCount()

            // Iterate through Rally work items
            importedChangeList.each{ // Check to see if it exists in UCR
                item ->
                def existingChange = doesChangeAlreadyExists (item.get(externalIdField), allChanges)

                //We keep a reference to the created or updated item
                def createdOrUpdatedChange = null

                if (existingChange == null) {
                    // NEW CHANGE - Just give the item (Json data for Rally Entry)
                    createdOrUpdatedChange = setChangeData (item, null)
                    bulkCreateChangeList.add(createdOrUpdatedChange)
                    countCreatedItem++
                }
                else {
                    // Existing Change - Pass the json, the existingChange, and other stuff for now
                    createdOrUpdatedChange = setChangeData (item, existingChange)


                    // Check for differing Integration Provider Ids
                    if (existingChange.integrationProvider.id != null && existingChange.integrationProvider.id != provider.id) {
                        // If the change is managed by another integration provider, add to the count
                        // that will eventually be displayed
                        countWarning++
                    }
                    else {
                        bulkUpdateChangeList.add(createdOrUpdatedChange)
                    }
                }
            }


            //Some changes will be created
            if (bulkCreateChangeList.size() > 0) {
                def changeToCreateArray = bulkCreateChangeList as Change[]
                change().post(changeToCreateArray)
                numberCreated = numberCreated + bulkCreateChangeList.size()
            }
            //Some changes will be updated
            if (bulkUpdateChangeList.size() > 0) {
                def changeToUpdateArray = bulkUpdateChangeList as Change[]
                change().put(changeToUpdateArray)
                numberUpdated = numberUpdated + bulkUpdateChangeList.size()
            }
            if (countWarning > 0) {
                println (countWarning+" items are already managed by an other Integration Provider")
            }

            //If page index has surpassed all of the changes, exit the loop
            if (totalRallyChanges <= (pageSize*pageCount + 1)) {
                continueRunning = false
            }

        }

        println("--------------------------------------------------------------")
        println ("Number of Changes that will be created: "+numberCreated)
        println ("Number of Changes that will be updated: "+numberUpdated)
        println ("Number of WorkItems to import from Rally: "+totalRallyChanges)

        def currentTime = new Date()
        println ("Last execution set to: "+currentTime)
        provider.property("lastExecution", currentTime.time.toString()).save()
        rallyHelper.close()
    }

    //--------------------------------------------------------------
    def Change doesChangeAlreadyExists (externalId, allChanges) {

        Change existingChange = changesByExternalId.get(externalId)

        return existingChange
    }

    //--------------------------------------------------------------
    /**
     * This populates the UCR client model with the needed data to save it.
     */
    def Change setChangeData (rallyEntry, existingChange) {
        def ucrChange = null

        if (existingChange != null) {
            // We will be updating this already existing UCR change
            ucrChange = existingChange
        }
        else {
            // We will be creating a new UCR change
            ucrChange = change()
        }

        ucrChange.name(rallyEntry.get("Name"))

        // The URL used for generating the iFrame on the Changes page in UCR
        def rallyObjectURL
        if(rallyEntry.get("FormattedID") != null && !rallyEntry.get("FormattedID").isEmpty()) {
            rallyObjectURL = this.rallyUrl + "#/search?keywords=" + rallyEntry.get("FormattedID")
        }
        else {
            rallyObjectURL = rallyEntry.get("_ref")
        }

        // Setting that value here
        ucrChange.externalUrl(rallyObjectURL)
        ucrChange.externalId(rallyEntry.get(externalIdField))

        // Setting the integration provider ID on the change in UCR
        ucrChange.setIntegrationProvider(new PluginIntegrationProvider().id(integrationProviderId))

        // Setting the Description
        def description =rallyEntry.get("Description")
        if( description != null ) {
            ucrChange.description(description)
        }

        //------------------------MAPPING RELEASE <=> rallyReleaseKey ------------------------------
        def rallyReleaseValue = rallyEntry.get(rallyReleaseKey)
        def rallyReleaseValueString = extractAssociatedEntity (rallyReleaseValue, ucrChange)

        allReleasesNameId.each{ item ->
            if (item.name.equals(rallyReleaseValueString)) ucrChange.release(item)
        }

        //------------------------MAPPING INIATIVE <=> rallyInitiativeKey --------------------------
        def rallyInitiativeValue = rallyEntry.get(rallyInitiativeKey)
        def rallyInitiativeValueString = extractAssociatedEntity (rallyInitiativeValue, ucrChange)

        allInitiativesNameId.each{ item ->
            if (item.name.equals(rallyInitiativeValueString)) ucrChange.initiative(item)
        }

        if (rallyNewInitiative && ucrChange.initiative == null && rallyInitiativeValueString != null && !rallyInitiativeValueString.equals("")) {
            //create the new initiative
            Initiative initiative = initiative().name(rallyInitiativeValueString).setIntegrationProvider(new PluginIntegrationProvider().id(integrationProviderId))
            initiative.save()
            ucrChange.initiative(initiative)
            // Retrieve all Initiatives Name and Id from UCR
            def initEntity = new Initiative()
            initEntity.format("name")
            this.allInitiativesNameId = initEntity.getAll()
            System.out.println("Created Initiative: " + rallyInitiativeValueString)
        }

        //------------------------MAPPING APPLICATION <=> rallyApplicationKey-----------------------
        def rallyApplicationValue = rallyEntry.get(rallyApplicationKey)
        def rallyApplicationValueString = extractAssociatedEntity (rallyApplicationValue, ucrChange)

        allApplicationsNameId.each{ item ->
            if (item.name.equals(rallyApplicationValueString)) ucrChange.application(item)
        }


        /**********************************
         Severity and Status Hard-Coded Mapping
         ***********************************/
        def rallyStatusField
        if (rallyType == "Milestone" || rallyType == "Change" || rallyType == "Blocker") {
            rallyStatusField = "" //No Status field for these types
        }
        else if (rallyType == "Defect" || rallyType == "DefectSuite" || rallyType == "HierarchicalRequirement") {
            rallyStatusField = "ScheduleState"
        }
        else if (rallyType == "Task") {
            rallyStatusField = "State"
        }
        else if (rallyType == "TestCase") {
            rallyStatusField = "Defect Status"
        }

        def rallySeverityField

        if (rallyType == "HierarchicalRequirement" || rallyType == "DefectSuite" || rallyType == "Task" ||
        rallyType == "Milestone" || rallyType == "Blocker" || rallyType == "Change") {
            rallySeverityField = "" //No Severity field for these types
        }
        else if (rallyType == "Defect") {
            rallySeverityField = "Severity"
        }
        else if (rallyType == "TestCase") {
            rallySeverityField = "Risk"
        }

        //------------------------MAPPING SEVERITIES-----------------------------------------------
        def rallySeverity = rallyEntry.get(rallySeverityField)
        def ucrSeverity = getMappingSeverities (rallySeverity)
        if (ucrSeverity) {
            def ucrSeveritiesEnum = Change.Severity.values()

            for (s in ucrSeveritiesEnum) {
                if (s.toString().equals(ucrSeverity)) {
                    ucrChange.severity(s)
                }
            }
        }

        //------------------------MAPPING TYPES-----------------------------------------------
        ucrChange.type(ucrChangeType)

        //------------------------MAPPING STATUSES-----------------------------------------------
        //We map the UCR statuses to Rally statuses
        def rallyStatusValue = rallyEntry.get(rallyStatusField)
        def ucrStatus = getMappingStatus(rallyStatusValue)
        if (ucrStatus) {
            def ucrStatusesEnum = Change.Status.values()

            for (s in ucrStatusesEnum) {
                if (s.toString() == ucrStatus) {
                    ucrChange.status(s)
                }
            }
        }

        return ucrChange
    }

    //--------------------------------------------------------------
    def extractAssociatedEntity (rallyDataValue, ucrChange) {
        def rallyDataValueString

        // Is the value a json object or a string?
        if(rallyDataValue instanceof HashMap) {
            // Does the object have a "Name" value?
            if(rallyDataValue.containsKey("Name")) {
                rallyDataValueString = rallyDataValue.get("Name")
            }
            // Does the object have a "name" value
            else if(rallyDataValue.containsKey("name")) {
                rallyDataValueString = rallyDataValue.get("name")
            }
            // Does the object have a "_refObjectName" value
            else if(rallyDataValue.containsKey("_refObjectName")) {
                rallyDataValueString = rallyDataValue.get("_refObjectName")
            }
        }
        else {
            rallyDataValueString = rallyDataValue
        }

        return rallyDataValueString
    }

    //--------------------------------------------------------------
    def getMappingStatus (status) {
        def mappedStatus = null
        if (mappingStatuses != null) {
            def fullMapping = slurper.parseText(mappingStatuses)
            fullMapping.each {
                def ucrStatus = it.ucrStatuses
                def rallyStatus = it.rallyStatuses
                if (rallyStatus.toString() == status) {
                    mappedStatus = ucrStatus
                }
            }
        }
        else {
            log ("No mapping for statuses set")
        }
        return mappedStatus
    }

    //--------------------------------------------------------------
    def getMappingSeverities (severity) {
        def mappedSeverity = null
        if (mappingSeverities != null) {
            def fullMapping = slurper.parseText(mappingSeverities)
            fullMapping.each {
                def rallySeverity = it.rallySeverities
                def ucrSeverity = it.ucrSeverities
                if (severity.toString() == rallySeverity) {
                    mappedSeverity = ucrSeverity
                }
            }
        }
        else {
            log ("No mapping for severities set")
        }
        return mappedSeverity
    }

}