#!/usr/bin/env groovy
/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Release
 * (c) Copyright IBM Corporation 2016. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
import com.urbancode.air.AirPluginTool
import com.urbancode.air.PropsParser
import com.urbancode.air.XTrustProvider
import com.urbancode.release.rest.framework.Clients
import com.urbancode.release.rest.models.internal.Comment
import com.urbancode.release.rest.models.internal.TaskExecution
import com.urbancode.release.rest.models.internal.ScheduledDeployment
import com.urbancode.release.rest.models.internal.ApplicationEnvironment
import com.urbancode.release.rest.models.Version;
import com.urbancode.release.rest.models.Inventory;
import com.urbancode.urelease.integration.nolio.NolioRestAPIHelper
import com.nolio.platform.server.dataservices.api.model.OpenAPIServiceStub
import com.nolio.platform.server.dataservices.api.model.OpenAPIServiceStub.GetJobStatus
import org.apache.log4j.*
import groovy.util.logging.*
import groovy.json.JsonSlurper

//Setup log4j
//def currentDirectory = new File(getClass().protectionDomain.codeSource.location.path).parent
//def config = new ConfigSlurper().parse(new File(currentDirectory+'/log4jSetup.groovy').toURL())
//PropertyConfigurator.configure(config.toProperties())
//Logger log = Logger.getInstance(getClass())

def apTool = new AirPluginTool(args[0], args[1])

XTrustProvider.install()

//We launch the Executor class here
def executor = new GenericProcessExecutor (apTool)

executor.executeTask ()

//That class is responsible for executing generic processes and deployment processes
//Because it handles both and the logic between running a simple process and a deployment process is handled here
//we need a way to tell if this a deployment task or a simple process (configuration task)
//The following parameter "deploymentProcess=true" must then be set on the task if that task should be run as a deployment task

public class GenericProcessExecutor {

    def releaseToken
    def serverUrl
    def nolioURL
	def nolioRestApiURL
    def nolioUsername
    def nolioPassword
    def integrationProviderVersion
    def integrationProviderId
    def factory
    def provider
	def extraProperties
    def apTool
	def NolioRestAPIHelper restHelper
    def projectName = ''
	def buildName = ''
	def deploymentName = ''
	def propsParser = new PropsParser()

    //Main constructor
    GenericProcessExecutor (apTool) {
        this.apTool = apTool
        def props = apTool.getStepProperties()
        this.releaseToken = props['releaseToken']
        this.serverUrl = props['releaseServerUrl']
        //The properties defined in the UI on the integration are automatically available
        this.nolioURL = props['nolioURL']
		if (!this.nolioURL.endsWith("/")) {
            this.nolioURL += "/"
        }
		this.nolioRestApiURL = this.nolioURL + 'datamanagement/a/'
        this.nolioUsername = props['nolioUsername']
        this.nolioPassword = props['nolioPassword']
                // Project Name
        this.integrationProviderId = props['releaseIntegrationProvider']

        //this is the context passed by UCR
        //It contains the version, application, scheduledDeployment, targets
        this.extraProperties = props['extraProperties']

        //Lets authenticate with UCR
        Clients.loginWithToken(serverUrl, releaseToken)

        projectName = props['projectName'] ?: "UCR Deployments"
		buildName = props['buildPattern']
		deploymentName = props['deploymentPattern']
    }

    //--------------------------------------------------------------
    def executeTask () {
		this.restHelper = new NolioRestAPIHelper(this.nolioRestApiURL, this.nolioUsername, this.nolioPassword)

        def slurper = new JsonSlurper();
        def jsonBuilder = new groovy.json.JsonBuilder();

        //We load the context variables
        def contextProperties = slurper.parseText(extraProperties)

        //We need the ucr task if so we can handle that UCR object later
        def urcTaskId = contextProperties.task.id
		def sdId = contextProperties.task.scheduledDeploymentId

        def taskExecution = new TaskExecution()
        taskExecution.id(urcTaskId)
		taskExecution = taskExecution.get()

		ScheduledDeployment sd = new ScheduledDeployment()
		sd.id(sdId)
		sd.format("detail")
		sd = sd.get()

		def parsedProjectName = propsParser.evaluatePropsToReplace(projectName, sd, taskExecution)
		def parsedBuild = propsParser.evaluatePropsToReplace(buildName, sd, taskExecution)
		def parsedDeployment = propsParser.evaluatePropsToReplace(deploymentName, sd, taskExecution)

        //We will need the user to create comments
        def userId = contextProperties.userId

        //We need to create one process for each target
        def targets = contextProperties.task.targets

        if (targets.size() == 0) {
             new Comment().userId(userId).task(taskExecution).comment("No Targets to deploy to!").post()
			 return
        }
        else {
            println "The process will be run for "+targets.size()+" target(s)"
        }

		def targIds = "";
		def targetIds = [] as List
		Map<String,String> targetIdToName = new HashMap<String,String>();
		def targetCount = 0;
		targets.each {
			target ->
			targetIds.add(target.externalId);
			targetIdToName.put(target.externalId, target.name);
			if(targetCount > 0) {
				targIds += ",";
			}
			targIds += target.externalId;
			targetCount++;
		}

        //The task will be successful if all the process for each target is successful
        //It at least one fails then the task fails
        def atLeastOneProcessFailed = false;
		//Application Id
		def applicationId = contextProperties.task.application.externalId;
		//Process Name
		def processName = contextProperties.task.properties.processName;

		def artifactPackage = null;
		def artifactPackageId = null;
		if (!(contextProperties.task.version instanceof Integer)) {
			//Artifact Package Name
			artifactPackage = contextProperties.task.version.name;
			//Artifact Package Name
			artifactPackageId = contextProperties.task.version.externalId;
		}
		//Template Category Id
		def templateCategoryId = contextProperties.task.properties.categoryId;
		//UpdatePipeline
		def updatePipeline = contextProperties.task.properties.updatePipeline;
		//Properties list is an extra parameter input in the task dialog
        def propMap = [:]
		def deployProps = contextProperties.task.properties.deployProps;

        if (deployProps) {
            def splitProps = deployProps.split(";|\n")*.trim();
            for (def prop in splitProps) {
                def (key, val) = prop.tokenize("=")*.trim();
                propMap.put(key, val);
            }
        }

		//How long to wait for the deployment to be created
		def timeoutMinutes = 100;
		def timeout = timeoutMinutes*60*1000

		def deploymentRequest = restHelper.runDeploymentPlan(
            processName,
            applicationId,
            targetIds,
            artifactPackage,
            parsedProjectName,
            templateCategoryId,
            parsedBuild,
            parsedDeployment,
            propMap,
            timeout
        )
		println deploymentRequest
		def deploymentIdsList = [] as List
		def incompleteDeployments = [] as List
		Map<String,String> deploymentIdToEnvName = new HashMap<String,String>();
		def deploymentIds = "";
		def deploymentCount = 0;
		deploymentRequest.deploymentResults.each {
			deployment ->
			if(deploymentCount > 0) {
				deploymentIds += ",";
			}
			deploymentIdsList.add(deployment.id);
			incompleteDeployments.add(deployment.id);
			def envName = targetIdToName.get(deployment.envId);
			deploymentIdToEnvName.put(deployment.id, envName);
			deploymentIds += deployment.id
			deploymentCount++;
		}
		def secondsToWait = 30;
		def endTime =  (secondsToWait*1000) + System.currentTimeMillis();
		def waitInterval = 5000;

		taskExecution.updateProperty("deploymentIds", deploymentIds);
		if (incompleteDeployments.size() > 0 && artifactPackageId != null) {
			taskExecution.updateProperty("artifactPackageId", artifactPackageId);
			taskExecution.updateProperty("targets", targIds);
		}

		//TODO: we don't know any statuses or what to look for in nolio yet
		def completeDeploymentIds = [] as List
		def done = false
		while ((endTime > System.currentTimeMillis()) && !done) {
			deploymentIdsList.each {
				deploymentId ->
				if(!completeDeploymentIds.contains(deploymentId)) {
					def deploymentResponse = restHelper.getDeploymentState(deploymentId)
					def status = deploymentResponse.deploymentState
					def description = deploymentResponse.description
					def targetName = deploymentIdToEnvName.get(deploymentId);

					if (status.indexOf("Succeeded") > -1) {
						new Comment().userId(userId).task(taskExecution).comment("Job Execution Completed for "+targetName).post()
						completeDeploymentIds.add(deploymentId);
						incompleteDeployments.remove(deploymentId);
					}
					else if (status.indexOf("Failed") > -1) {
						new Comment().userId(userId).task(taskExecution).comment("Job Execution Failed for "+targetName+" with status: $status").post()
						completeDeploymentIds.add(deploymentId);
						incompleteDeployments.remove(deploymentId);
						atLeastOneProcessFailed = true;
					}
					else if (status.indexOf("Active") > -1) {
						if (description.indexOf("(Waiting for user input)") > -1) {
							new Comment().userId(userId).task(taskExecution).comment("Job Execution for "+targetName+" is waiting on manual intervention").post()
						}
					}
					else {
						new Comment().task(task).comment("Job Execution returned an unknown status of: $status").post()
					}
				}
			}
			if(completeDeploymentIds.size() != deploymentIdsList.size()) {
				sleep waitInterval
			}
			else {
				done = true
			}
		}

        //--------------------------------------------------------------------------------------
        //If at least one of the target process failed then we fail the task
        if (atLeastOneProcessFailed) {
            taskExecution.fail();
        }
        else if(done) {
            taskExecution.complete();
			if (updatePipeline && artifactPackageId != null) {
				def inventories = [] as List
				targetIds.each {
					targetId ->
					def inventory = new Inventory();
					inventory.applicationTarget(new ApplicationEnvironment().externalId(targetId));
					inventory.version(new Version().externalId(artifactPackageId));
					inventories.add(inventory);
				}
				new Inventory().sync(inventories);
			}
        }
    }
}

