package com.urbancode.urelease.integration.nolio

import groovy.json.JsonSlurper
import java.text.SimpleDateFormat
import com.google.gson.Gson
import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParser
import org.apache.commons.httpclient.NameValuePair

/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/

class NolioRestAPIHelper {

    private RestHelper restHelper
    private File logFile;
    private currentDirectory = new File(getClass().protectionDomain.codeSource.location.path).parent
    private SimpleDateFormat sdf = new SimpleDateFormat("MM-dd-yy-hh-mm-ss");

    public NolioRestAPIHelper(String url, String username, String password) {
        logFile = new File(currentDirectory+"/TracerNolioRestApiHelper.txt")
        addLog ("Rest API Nolio URL => "+url)
        restHelper = new RestHelper(url, username, password)
    }

	public JsonArray getApplications() {
		def jsonString = restHelper.getOnUrl("api/v4/applications")
		logFile << "------------------------------"
		logFile << "/applications/"
		logFile << "Get Applications response "+jsonString+"\n"
		println jsonString
		JsonParser parser = new JsonParser()
		JsonElement element = parser.parse(jsonString)
		JsonArray result = element.getAsJsonArray()

        return result
    }

	public JsonArray getEnvironments(String appId) {
		def JsonArray result = new JsonArray();
		if (appId != null && appId.length() > 0) {
			def jsonString = restHelper.getOnUrl("api/v4/applications/$appId/environments")
			logFile << "------------------------------"
			logFile << "/applications/$appId/environments"
			logFile << "Get Environments response "+jsonString+"\n"
			JsonParser parser = new JsonParser()
			JsonElement element = parser.parse(jsonString)
			result = element.getAsJsonArray()
		}

        return result
    }

	public JsonArray getTemplateCategories(String appId) {
		def JsonArray result = new JsonArray();
		if (appId != null && appId.length() > 0) {
			def jsonString = restHelper.getOnUrl("api/v4/applications/$appId/categories")
			logFile << "------------------------------"
			logFile << "/applications/$appId/categories"
			logFile << "Get Environments response "+jsonString+"\n"
			JsonParser parser = new JsonParser()
			JsonElement element = parser.parse(jsonString)
			JsonObject json = element.getAsJsonObject()
			result = json.get("list").getAsJsonArray()
		}

        return result
    }

	public JsonArray getDeploymentTemplates(String appId, String categoryId) {
		def JsonArray result = new JsonArray();
		if (appId != null && appId.length() > 0) {
			def jsonString = restHelper.getOnUrl("api/v4/applications/$appId/categories/$categoryId/templates")
			logFile << "------------------------------"
			logFile << "/applications/$appId/categories/$categoryId/templates"
			logFile << "Get Templates response "+jsonString+"\n"
			JsonParser parser = new JsonParser()
			JsonElement element = parser.parse(jsonString)
			JsonObject json = element.getAsJsonObject()
			result = json.get("list").getAsJsonArray()
		}

        return result
    }

    public List getApplicationProjects(String appId) {
        def result = []
        if (appId != null && appId.length() > 0) {
            def jsonString = restHelper.getOnUrl("api/v4/applications/$appId/projects")
            logFile << "------------------------------"
            logFile << "applications/$appId/projects"
            logFile << "Get Application Project response "+jsonString+"\n"
            new JsonSlurper().parseText(jsonString).each { project ->
                result.add(project.id)
            }
        }

        return result
    }

    public Object getDeploymentPlans(String appId, String projectId) {
        def result
        if (appId != null && appId.length() > 0) {
            addLog ("Request: applications/$appId/projects/$projectId/deployment-plans")
            def jsonString = restHelper.getOnUrl("api/v4/applications/$appId/projects/$projectId/deployment-plans")
            addLog ("Response: "+jsonString)
            result = new JsonSlurper().parseText(jsonString)
        }

        return result
    }

	public JsonObject getArtifactPackageContent(String artifactId) {
        JsonObject result = new JsonObject();
        if (artifactId != null) {
            addLog ("Request: /artifact_packages/$artifactId/artifact_version_full_paths")
			def jsonString = restHelper.getOnUrl("artifact_packages/$artifactId/artifact_version_full_paths")
            addLog ("Response: "+jsonString)
            JsonParser parser = new JsonParser()
			JsonElement element = parser.parse(jsonString)
			result = element.getAsJsonObject()
        }

        return result
    }

	public Object getArtifactVersion(String appName, String artifactType, String artifactDefinition) {
        def result
        if (appName != null && artifactType != null && artifactDefinition != null) {
            addLog ("Request: /artifact-version-details")
			JsonObject body = new JsonObject();
			body.addProperty("application", appName);
			body.addProperty("artifactDefinition", artifactDefinition);
			body.addProperty("artifactType", artifactType);
            def jsonString = restHelper.postOnUrl(body.toString(), "api/v4/artifact-version-details")
            addLog ("Response: "+jsonString)
            result = new JsonSlurper().parseText(jsonString)
        }

        return result
    }

    public String getLatestDeployment(String appId, String envId) {
        def result
        if (appId != null && appId.length() > 0 && envId != null && envId.length() > 0) {
            addLog ("Request: releases-reports?application=$appId&environement=$envId")
            def jsonString = restHelper.getOnUrl("api/v4/releases-reports?application=$appId&environement=$envId")
            addLog ("Response: "+jsonString)
            def report = new JsonSlurper().parseText(jsonString)
            if (report.size() > 0) {
                result = report[-1].id
            }
        }

        return result
    }

	public JsonArray getArtifactPackages(String appId) {
        def result
        if (appId != null && appId.length() > 0 ) {
            addLog ("Request: applications/$appId/artifact_packages?filter=&orderBy=id-false&pageSize=20")
            def jsonString = restHelper.getOnUrl("applications/$appId/artifact_packages?filter=&orderBy=id-false&pageSize=20")
            addLog ("Response: "+jsonString)
            JsonParser parser = new JsonParser()
			JsonElement element = parser.parse(jsonString)
			JsonObject json = element.getAsJsonObject()
			result = json.get("list").getAsJsonArray()
        }

        return result
    }

	//TODO it may make sense to have the project name be the name of the release
	public Object runDeploymentPlan(
        String templateName,
        String appId,
        List envIds,
        String artifactPackage,
        String projectName,
        String templateCategory,
        String buildName,
        String deploymentName,
        Map properties,
        Integer timeout)
    {
        def result
        if (templateName != null && appId != null && envIds != null && templateCategory != null) {
			Gson gson = new Gson();
			JsonArray envs = gson.toJsonTree(envIds).getAsJsonArray()
            addLog ("Request: /run-deployment-plan")
			JsonObject body = new JsonObject();
			body.addProperty("deployment", deploymentName);
			body.add("environmentIds", envs);
			body.addProperty("applicationId", appId);
			if (artifactPackage != null) {
				body.addProperty("artifactPackage", artifactPackage);
			}
			//TODO does this make sense or should it be packageName?
			body.addProperty("build", buildName);
			body.addProperty("deploymentPlan", deploymentName);
			body.addProperty("deploymentTemplate", templateName);
			//We can be smart and parse the properties to possibly override any of these fields
			if(properties != null) {
                JsonObject jsonProperties = gson.toJsonTree(properties);
				body.add("properties", jsonProperties);
			}
			body.addProperty("project", projectName);
			body.addProperty("templateCategoryId", templateCategory);
			body.addProperty("timeout", timeout);
            def jsonString = restHelper.postOnUrl(body.toString(), "api/v4/run-deployment-plan")
            addLog ("Response: "+jsonString)
			result = new JsonSlurper().parseText(jsonString)
        }

        return result
    }

	public Object getDeploymentState(String deploymentId) {
        def result
        if (deploymentId != null && deploymentId.length() > 0) {
            def jsonString = restHelper.getOnUrl("api/v4/deployment-state/$deploymentId")
            logFile << "------------------------------"
            logFile << "deployment-state/$deploymentId"
            logFile << "Get Deployment State response "+jsonString+"\n"
            result = new JsonSlurper().parseText(jsonString)
        }

        return result
    }

    private void addLog (String log) {
        def date = sdf.format(new Date());
        logFile << date+" - "+log+" \n"
    }
}
