package com.urbancode.urelease.integration.nolio

import com.ibm.icu.text.IDNA
import com.ibm.icu.text.StringPrepParseException

import java.io.*

/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
import org.apache.commons.httpclient.HttpClient
import org.apache.commons.httpclient.HttpMethod
import org.apache.commons.httpclient.UsernamePasswordCredentials
import org.apache.commons.httpclient.auth.AuthScope
import org.apache.commons.httpclient.methods.GetMethod
import org.apache.commons.httpclient.methods.PostMethod
import org.apache.commons.httpclient.params.HttpParams
import org.apache.http.client.utils.URIBuilder
import org.apache.log4j.Logger

public class RestHelper {

    //**********************************************************************************************
    // CLASS
    //**********************************************************************************************
    static private final Logger log = Logger.getLogger(RestHelper.class)
    static final private int DEFAULT_TIMEOUT = 600000
    //**********************************************************************************************
    // INSTANCE
    //**********************************************************************************************
    private HttpClient client
    private String url

    //----------------------------------------------------------------------------------------------
    public RestHelper(String baseURL, String username, String password) {
        url = baseURL
        initRestHelper(username, password)
    }

    //----------------------------------------------------------------------------------------------
    public String getUrl() {
        return url
    }

    //----------------------------------------------------------------------------------------------
    public boolean isConnected() {
        boolean result = false

        try {
            GetMethod method = new GetMethod(url)
            doOnUrl(method)
            result = true
        }
        catch (Exception e) {
        }
        return result
    }

    //----------------------------------------------------------------------------------------------
    private void initRestHelper(String username, String password) {
        url = addSlash(url)
        //We need to make sure URLs with special characters will be converted
        this.url = globalize(this.url)

        URIBuilder builder = null
        try {
            builder = new URIBuilder(this.url)
        }
        catch (URISyntaxException e) {
            log.fatal("Problem with RestHelper's URL: ", e)
            throw new RuntimeException("Problem with Rester's URL:", e)
        }

        if (builder != null) {

            client = new HttpClient()
            client.getParams().setAuthenticationPreemptive(true)
            client.getState().setCredentials(
                    AuthScope.ANY,
                    new UsernamePasswordCredentials(username, password)
            )
        }
    }

    //----------------------------------------------------------------------------------------------
    /**
     * Ensures that the given string ends with a /
     *
     * @param baseUrl
     * @return
     */
    private String addSlash(String baseUrl) {
        String result
        if (!baseUrl.endsWith("/")) {
            result = baseUrl + "/"
        }
        else {
            result = baseUrl
        }

        return result
    }

    private String globalize(String url) {
        try {
            URL urlToGlobalize = new URL(url)
            //Here we convert the host name
            String udUrlAscii = IDNA.convertIDNToASCII(urlToGlobalize.getAuthority(), IDNA.DEFAULT).toString()
            //And we rebuild the URL with the new converted host name
            url = urlToGlobalize.getProtocol()+"://"+udUrlAscii+urlToGlobalize.getPath()
        }
        catch (MalformedURLException ex) {
            throw new RuntimeException("Malformed URL $url for Integration Provider", ex)
        }
        catch (StringPrepParseException ex) {
            throw new RuntimeException("Can not convert the URL $url from IDN to ASCII", ex)
        }

        return url
    }

    //----------------------------------------------------------------------------------------------
    public void setClientTimeout (int timeout) {
        HttpParams params = client.getParams()
        params.setSoTimeout(timeout)
        params.setConnectionManagerTimeout(timeout)
    }
    
    //----------------------------------------------------------------------------------------------
    public void resetClientTimeout () {
        setClientTimeout (DEFAULT_TIMEOUT)
    }
    
    //----------------------------------------------------------------------------------------------
    public String getOnUrl(String suffix)
    throws IOException {
        String url = this.url + suffix
        String result = null
        resetClientTimeout()
        GetMethod request = new GetMethod(url)
        request.setDoAuthentication(true)
        log.debug("GET on url " + url)
        result = doOnUrl(request)

        return result
    }
	
	    //----------------------------------------------------------------------------------------------
    public String postOnUrl(String body, String suffix)
    throws IOException {
        String url = this.url + suffix
        String result = null
        resetClientTimeout()
        PostMethod request = new PostMethod(url)
        request.setDoAuthentication(true)
		request.setRequestBody(body)
		request.setRequestHeader("Content-type", "application/json")
        log.debug("GET on url " + url)
        result = doOnUrl(request)

        return result
    }

    //----------------------------------------------------------------------------------------------
    /**
     * Performs a particular Http request based on the request passed through to the function.
     * @param request The particular Http request we woudl like to execute
     * @return A string representation of the response or empty string if the response had no content
     * @throws IOException If EntityUtils or HttpClient#execute fails to perform the task
     * @throws IllegalStateException If the returned response is in an illegal state
     */
    synchronized private String doOnUrl(HttpMethod request)
    throws IOException, IllegalStateException {
        String result = ""
        String url = request.getURI().getPath()

        try {
            def statusCode = client.executeMethod(request)
            result = request.getResponseBodyAsString()

            if (result != null) {

                String message = "Received reponse ${statusCode}:${request.getStatusText()} from ${request.getName()}@$url"
                if (!(statusCode >= 200 && statusCode < 300)) {
                    log.error(message)
                    log.error(result)
                    throw new RuntimeException("Error: $message")
                }
            }
        }
        catch (IllegalStateException e) {
            log.error("Response content in illegal state", e)
            throw new IllegalStateException("IllegalStateException while trying to ${request.getName()} on url: $url", e)
        }
        catch (SocketTimeoutException e) {
            log.error("Could not get a response in a reasonable amount of time", e)
            throw new SocketTimeoutException("Timeout Exception while trying to ${request.getName()} on url: $url")
        }
        catch (IOException e) {
            log.error("Could not read response", e)
            throw new IOException("IOException while trying to ${request.getName()} on url: $url", e)
        }
        catch (Exception e) {
            log.error("Problem with doOnUrl: ", e)
            throw new RuntimeException("Problem with doOnUrl:", e)
        }
        finally {
            if (request != null) {
                request.releaseConnection()
            }
        }

        return result
    }
    
    
    //----------------------------------------------------------------------------------------------
    public List<String> getHelpOnConnectionException(Exception ex){
        List<String> helpList = new ArrayList<String>()

        if (ex.getCause() != null) {
            String msg = ex.getCause().toString()
            if (msg.contains("Unauthorized")) {
                helpList.add("The wrong username and/or password")
            }
            if (msg.contains("UnknownHostException") || msg.contains("HttpHostConnectException")) {
                helpList.add("The CA Release Automation instance seems not accessible from the IBM UrbanCode Release server")
                helpList.add("The CA Release Automation server URL is not correct")
                helpList.add("A firewall might be blocking the connection with CA Release Automation")
            }
            if (msg.contains("MalformedURLException")) {
                helpList.add("The URL provided contains special characters that are not allowed")
                helpList.add("The Port number might not be a valid number")
            }
        }

        return helpList
    }
}
