#!/usr/bin/env groovy
/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Release
 * (c) Copyright IBM Corporation 2016. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
import com.urbancode.air.AirPluginTool
import com.urbancode.air.XTrustProvider
import com.urbancode.release.rest.framework.Clients
import com.urbancode.release.rest.models.internal.Comment
import com.urbancode.release.rest.models.internal.TaskExecution
import com.nolio.platform.server.dataservices.api.model.OpenAPIServiceStub
import com.nolio.platform.server.dataservices.api.model.OpenAPIServiceStub.GetJobStatus
import org.apache.log4j.*
import groovy.util.logging.*

import groovy.json.JsonSlurper

//Setup log4j
def currentDirectory = new File(getClass().protectionDomain.codeSource.location.path).parent
def config = new ConfigSlurper().parse(new File(currentDirectory+'/log4jSetup.groovy').toURL())
PropertyConfigurator.configure(config.toProperties())
Logger log = Logger.getInstance(getClass())

def apTool = new AirPluginTool(args[0], args[1])

XTrustProvider.install()

//We launch the Executor class here
def executor = new GenericProcessExecutor (apTool)

executor.executeTask ()

public class GenericProcessExecutor {

    def releaseToken
    def serverUrl
    def nolioURL
    def nolioUsername
    def nolioPassword
    def integrationProviderVersion
    def integrationProviderId
    def factory
    def provider
    def extraProperties
    def apTool

    //Main constructor
    GenericProcessExecutor (apTool) {
        this.apTool = apTool
        def props = apTool.getStepProperties()
        this.releaseToken = props['releaseToken']
        this.serverUrl = props['releaseServerUrl']
        //The properties defined in the UI on the integration are automatically available
        this.nolioURL = props['nolioURL']
        this.nolioUsername = props['nolioUsername']
        this.nolioPassword = props['nolioPassword']
        this.integrationProviderId = props['releaseIntegrationProvider']

        //this is the context passed by UCR
        //It contains the version, application, scheduledDeployment
        this.extraProperties = props['extraProperties']

        //Lets authenticate with UCR
        Clients.loginWithToken(serverUrl, releaseToken)
    }

    //--------------------------------------------------------------
    def executeTask () {

        def slurper = new JsonSlurper();
        def jsonBuilder = new groovy.json.JsonBuilder();

        //We load the context variables
        def contextProperties = slurper.parseText(extraProperties)

        //We need the ucr task if so we can handle that UCR object later
        def urcTaskId = contextProperties.task.id

        def taskExecution = new TaskExecution()
        taskExecution.id(urcTaskId)
        //We will need the user to create comments
        def userId = contextProperties.userId

        //The task will be successful if all the process for each target is successful
        //It at least one fails then the task fails
        def atLeastOneProcessFailed = false;

            //Here all plugin properties will be used
            //Application Name
            def applicationName;
            //Process Name
            def processName;
            //Environment name
            def environmentName;
            //Server list is an extra parameter input in the task dialog
            def serverList;
            //Parameter list is an extra parameter input in the task dialog
            def parameters;
            //Could be set as a plugin property later
            def timeout = 100;

            if (contextProperties.task.pluginProperties != null) {

                if (contextProperties.task.pluginProperties.processName != null) {
                    processName = contextProperties.task.pluginProperties.processName;
                }
                else {
                    new Comment().userId(userId).task(taskExecution).comment("Required processName plugin property!").post()
                    taskExecution.fail();
                }

                if (contextProperties.task.pluginProperties.environmentName != null) {
                    environmentName = contextProperties.task.pluginProperties.environmentName;
                }
                else {
                    new Comment().userId(userId).task(taskExecution).comment("Required environmentName plugin property!").post()
                    taskExecution.fail();
                }

                if (contextProperties.task.pluginProperties.applicationName != null) {
                    applicationName = contextProperties.task.pluginProperties.applicationName;
                }
                else {
                    new Comment().userId(userId).task(taskExecution).comment("Required applicationName plugin property!").post()
                    taskExecution.fail();
                }

                if (contextProperties.task.pluginProperties.serverList != null) {
                    serverList = contextProperties.task.pluginProperties.serverList;
                    serverList = serverList.trim().split(",|\n|\r\n");
                }
                else {
                    new Comment().userId(userId).task(taskExecution).comment("Required serverList plugin property!").post()
                    taskExecution.fail();
                }

                if (contextProperties.task.pluginProperties.parameters != null) {
                    parameters = contextProperties.task.pluginProperties.parameters;
                    parameters = parameters.trim().split("\n|\r\n");
                }
                else {
                    new Comment().userId(userId).task(taskExecution).comment("Required parameters plugin property!").post()
                    taskExecution.fail();
                }

            }

            if (!nolioURL.endsWith("/")) {
                nolioURL += "/"
            }
            nolioURL += 'datamanagement/ws/OpenAPIService'

            //Create the process for that target
            def stub = new OpenAPIServiceStub(nolioURL)
            def runProcess = new OpenAPIServiceStub.RunProcess()

            if (nolioUsername && nolioUsername.length() > 0) {
                runProcess.setUsername(nolioUsername)
            }
            if (nolioPassword && nolioPassword.length() > 0) {
                runProcess.setPassword(nolioPassword)
            }
            if (applicationName && applicationName.length() > 0) {
                runProcess.setAppName(applicationName)
            }
            if (processName && processName.length() > 0) {
                runProcess.setProcessName(processName)
            }
            if (environmentName && environmentName.length() > 0) {
                runProcess.setEnvironmentName(environmentName)
            }

            if (serverList && serverList.size() > 0) {
                def serverArray = new OpenAPIServiceStub.ArrayOfString()
                serverList.each {
                    if (it && it.trim().length() > 0) {
                        serverArray.addString(it.trim())
                    }
                }
                runProcess.setServers(serverArray)
            }

            if (parameters && parameters.size() > 0) {
                def parameterArray = new OpenAPIServiceStub.String2StringMap()
                parameters.each {
                    if (it && it.trim().length() > 0 && it.indexOf("=") > -1){
                        def key = it.substring(0, it.indexOf("="))?.trim()
                        def value = it.substring(it.indexOf("=") + 1)?.trim()
                        def nameValuePair = new OpenAPIServiceStub.Entry_type0()
                        nameValuePair.setKey(key)
                        nameValuePair.setValue(value)
                        parameterArray.addEntry(nameValuePair)
                    }
                }
                runProcess.setParameters(parameterArray)
            }

            //--------------------------------------------------------------------------------------
            println "WS URL: $url"

            def runProcessE = new OpenAPIServiceStub.RunProcessE()
            runProcessE.setRunProcess(runProcess)
            def OpenAPIServiceStub.RunProcessResponseE response = stub.runProcess(runProcessE)

            def jobId = response.getRunProcessResponse().get_return()

            println "Received Job ID: $jobId"

            def endTime = System.currentTimeMillis() + timeout*60*1000
            def done = false

            def GetJobStatus jobStatus = new GetJobStatus()

            if (nolioUsername && nolioUsername.length() > 0) {
                jobStatus.setUsername(nolioUsername)
            }
            if (nolioPassword && nolioPassword.length() > 0) {
                jobStatus.setPassword(nolioPassword)
            }
            jobStatus.setJobId(jobId)

            def jobStatusE = new OpenAPIServiceStub.GetJobStatusE()
            jobStatusE.setGetJobStatus(jobStatus)

            while ((endTime > System.currentTimeMillis()) && !done) {
                println "Checking job status: "
                def OpenAPIServiceStub.GetJobStatusResponseE jobStatusResponse = stub.getJobStatus(jobStatusE)
                def status = jobStatusResponse.getGetJobStatusResponse().get_return().getJobState()
                println "Current status: $status"
                if (status.indexOf("FINISH") > -1) {
                    done = true
                    new Comment().userId(userId).task(taskExecution).comment("Job Execution Completed for "+environmentName).post()
                    taskExecution.complete();
                    break
                }
                else if (status.indexOf("FAIL") > -1 || status.indexOf("STOP") > -1) {
                    new Comment().userId(userId).task(taskExecution).comment("Job Execution Failed for "+environmentName+" with status: $status, ${jobStatusResponse.getGetJobStatusResponse().get_return().toString()}").post()
                    taskExecution.fail();
                }
                else {
                    sleep 10000
                }
            }

            if (!done) {
                new Comment().userId(userId).task(taskExecution).comment("Timed out waiting for process execution for "+environmentName+" to complete").post()
                taskExecution.fail();
            }

    }
}

