package com.urbancode.urelease.integration.nolio

/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Release
 * (c) Copyright IBM Corporation 2016. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
import com.nolio.platform.server.dataservices.api.model.OpenAPIServiceStub.GetAssignedProcessesForEnvironmentResponse
import com.urbancode.commons.util.environment.Environment
import com.urbancode.release.rest.models.Application
import com.urbancode.release.rest.models.Component
import com.urbancode.release.rest.models.ComponentVersion
import com.urbancode.release.rest.models.Inventory
import com.urbancode.release.rest.models.Version
import com.urbancode.release.rest.models.internal.Comment
import com.urbancode.release.rest.models.internal.TaskExecution
import com.urbancode.release.rest.models.internal.ApplicationEnvironment
import com.urbancode.release.rest.models.internal.PluginIntegrationProvider

import com.nolio.platform.server.dataservices.api.model.OpenAPIServiceStub
import com.urbancode.air.*
import com.urbancode.release.rest.framework.Clients
import com.urbancode.release.rest.models.internal.Task.TaskType
import com.urbancode.release.rest.models.internal.TaskPlan
import com.urbancode.urelease.integration.nolio.NolioRestAPIHelper
import com.urbancode.urelease.integration.nolio.sync.FullSyncClient
import com.urbancode.urelease.integration.nolio.sync.SyncClient
import com.urbancode.urelease.integration.nolio.sync.TaskPlanFullSyncClient
import org.apache.log4j.*
import groovy.util.logging.*
import java.util.concurrent.TimeUnit
import java.util.Map.Entry
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonArray

public class NolioIntegration {
    final long DEFAULT_FULL_SYNC = 60 // Default full sync interval when one isn't provided
    long syncInterval // Interval between full syncs
    def releaseToken
    def serverUrl
    def nolioSOAPURL
    def nolioRESTURL
    def nolioUsername
    def nolioPassword
    def integrationProviderVersion
    def integrationProviderId
    def OpenAPIServiceStub soapStub
    def NolioRestAPIHelper restHelper
    def provider

    // We paginate the updating of entities to UCR.  We limit it to this number
    // for all entities
    def limitPerQuery = 500
    def plan2DeploymentMap = [:] as Map


    //Main constructor
    NolioIntegration(props) {
        //Release Token used for Authentication
        this.releaseToken = props['releaseToken']
        //Release Server URL
        this.serverUrl = props['releaseServerUrl']

        //Integration Provider currently running that Integration
        this.integrationProviderId = props['releaseIntegrationProvider']

        //Nolio properties
        def url = props['nolioURL']
        if (!url.endsWith("/")) {
            url += "/"
        }
        this.nolioSOAPURL = url + 'datamanagement/ws/OpenAPIService'
        this.nolioRESTURL = url + 'datamanagement/a/'
        this.nolioUsername = props['nolioUsername']
        this.nolioPassword = props['nolioPassword']

        /* Use default interval if one isn't provided or is invalid */
        if (props['syncInterval']) {
            try {
                this.syncInterval = Long.parseLong(props['syncInterval'])
            }
            catch (NumberFormatException ex) {
                println("[WARN] Sync interval ${props['syncInterval']} is invalid. " +
                    "Defaulting to ${DEFAULT_FULL_SYNC} minutes.")
                this.syncInterval = DEFAULT_FULL_SYNC
            }
        }
        else {
            this.syncInterval = DEFAULT_FULL_SYNC
        }
    }

    /* Constructor used for unit tests. */
    NolioIntegration(NolioRestAPIHelper restHelper) {
        this.restHelper = restHelper
    }

    //--------------------------------------------------------------
    //Authentication with Release
    def releaseAuthentication () {
        Clients.loginWithToken(serverUrl, releaseToken)
    }

    //--------------------------------------------------------------
    def runIntegration () {
        println "------------------------------------------------------------------------------------------------------------------------------"

        //We load the Integration Provider Info
        provider = new PluginIntegrationProvider().id(integrationProviderId).get()

        boolean fullSync = false
        /* Never execute a full sync when the sync interval is set to -1 */
        if (syncInterval != -1) {
            String lastFullSync = provider.getProperty("lastFullSync")

            if (lastFullSync != null) {
                /* Check if the syncInterval has passed since last full sync */
                long lastSyncMillis  = Long.parseLong(lastFullSync)
                long millisDiff = System.currentTimeMillis() - lastSyncMillis
                long minuteDiff = TimeUnit.MILLISECONDS.toMinutes(millisDiff)

                if (minuteDiff >= syncInterval) {
                    fullSync = true
                    println("Executing full sync since ${minuteDiff} minutes have elapsed, " +
                        "exceeding the ${syncInterval} minute full sync interval.")
                }
                else {
                    fullSync = false
                    println("A full sync will not occur because ${minuteDiff} hours have elapsed, "
                        + "and the full sync interval is set to ${syncInterval} hours.")
                }
            }
            else {
                fullSync = true
                println("A full sync has never been executed. Running full sync now.")
            }
        }
        else {
            println("A full sync will not be executed since the full sync interval is set to -1.")
        }

        XTrustProvider.install()
        this.restHelper = new NolioRestAPIHelper(nolioRESTURL, nolioUsername, nolioPassword)

        /* When a full sync occurs delete any removed Nolio applications/processes from UCR */
        SyncClient<Application> appSyncClient
        SyncClient<TaskPlan> procSyncClient
        if (fullSync) {
            appSyncClient = new FullSyncClient<Application>(new Application(),
                limitPerQuery, integrationProviderId)
            procSyncClient = new TaskPlanFullSyncClient(new TaskPlan(), limitPerQuery,
                integrationProviderId)
        }
        else {
            appSyncClient = new SyncClient<Application>(new Application(), limitPerQuery)
            procSyncClient = new SyncClient<TaskPlan>(new TaskPlan(), limitPerQuery)
        }

        println "---------------------------------------------------------------------------------------------------------------"
        //Import Applications
        List<Application> bulkApplicationList = importApplications(appSyncClient)

        println "---------------------------------------------------------------------------------------------------------------"
        //Import Environments
        importEnvironments(bulkApplicationList)

        println "---------------------------------------------------------------------------------------------------------------"
        //Import Processes
        importProcesses(bulkApplicationList, procSyncClient)

        println "---------------------------------------------------------------------------------------------------------------"
        //Import Versions, Components, ComponentVersions, and re-sync Applications with children
        SyncClient<Application> appResyncClient = new SyncClient<Application>(new Application(),
            limitPerQuery)
        for (Application application : bulkApplicationList) {
            SyncClient<Version> versionSyncClient = new SyncClient<Version>(new Version(),limitPerQuery)
            importArtifactPackages(application, versionSyncClient)
            appResyncClient.addImport(application)
        }
        appResyncClient.syncImports() // Sync any remaining imports.
        bulkApplicationList.clear()
        println "---------------------------------------------------------------------------------------------------------------"
        //Update Executing Tasks
        updateExecutingTasks()

        if (fullSync) {
            long currTime = System.currentTimeMillis()
            provider.property("lastFullSync", currTime.toString()).save()
        }
    }
        //--------------------------------------------------------------
    List<Application> importApplications (SyncClient<Application> appSyncClient) {
        println "<b><u>>>>> Import Applications</u></b>"
        //List of all applications from CA that will be turned into UCR Applications
        //We get all UCD application
        List<Application> bulkApplicationList = new ArrayList<Application>()
        long startImportApplication = System.currentTimeMillis()
        def nolioApps = restHelper.getApplications()
        long endImportApplication = System.currentTimeMillis()
        println "# CA Release Automation Applications imported: ${nolioApps.size()} " +
                "in ${(endImportApplication-startImportApplication)/1000} seconds"

        nolioApps.each {
            Object item ->
                Application application = new Application()
                    .name(item.get("name").getAsString())
                    .automated(true)
                    .property("objectType", "APPLICATION")
                    .description(item.has("description") && item.get("description") != null ? item.get("description").getAsString() : "")
                    .integrationProvider(provider)
                    .externalId(item.get("id").getAsString())

                bulkApplicationList.add(application)
                appSyncClient.addImport(application)
        }

        appSyncClient.syncImports() // Sync any remaining imports.
        println("Synced ${appSyncClient.importCount} applications.")
        appSyncClient.syncRemovals() // Sync objects that have been removed (paginates internally)
        println("Removed ${appSyncClient.removalCount} applications no longer in Nolio.")
        return bulkApplicationList
    }

    //--------------------------------------------------------------
    void importArtifactPackages(Application application, SyncClient<Version> versionSyncClient) {
        println "<b><u>>>>> Import Versions</u></b>"

        if (application != null) {
            String externalAppId = application.externalId;
            Map<String, Component> appChildrenMap = new HashMap<String, Component>()
            def nolioArtifactPackages = restHelper.getArtifactPackages(application.externalId)
            //println "# processing versions for application: ${app.name}"
            nolioArtifactPackages.each { Object artifactPackage ->

                def artifactPackageId = artifactPackage.get("id").getAsString()
                Version version = new Version()
                    .name(artifactPackage.get("name").getAsString())
                    .automated(true)
                    .createdDate(System.currentTimeMillis())
                    .description(artifactPackage.has("description") && artifactPackage.get("description") != null ? artifactPackage.get("description").getAsString() : "")
                    .integrationProvider(provider)
                    .externalId(artifactPackageId)
                    .application(application)

                SyncClient<Application> compSyncClient = new SyncClient<Component>(new Component(),
                    limitPerQuery)
                SyncClient<Version> compVersionSyncClient = new SyncClient<ComponentVersion>(
                    new ComponentVersion(), limitPerQuery)
                importComponents(compSyncClient, compVersionSyncClient, artifactPackageId,
                    version, appChildrenMap)
                versionSyncClient.addImport(version)
            }


            application.children(appChildrenMap.values().toArray(new Component[0]))
        }

        versionSyncClient.syncImports() // Sync any remaining imports.
        println("Synced ${versionSyncClient.importCount} versions.")
        versionSyncClient.syncRemovals() // Sync objects that have been removed (paginates internally)
        println("Removed ${versionSyncClient.removalCount} versions no longer in Nolio.")
    }

    void importComponents(
        SyncClient<Application> compSyncClient,
        SyncClient<Version> compVersionSyncClient,
        String artifactPackageId,
        Version ucrVersion,
        Map<String, Component> appChildrenMap)
    {
        JsonObject artifactContent = restHelper.getArtifactPackageContent(artifactPackageId)
        List<Component> bulkComponents = new ArrayList<Component>()

        if (artifactContent.has('types')) {
            Map<String, ComponentVersion> versionChildren = new HashMap<String, ComponentVersion>()
            JsonObject types = artifactContent.getAsJsonObject('types')
            Set<Entry<String, JsonElement>> typeEntries = types.entrySet()
            for (Map.Entry<String,JsonElement> typeEntry : typeEntries) {
                String typeName = typeEntry.getKey()
                JsonObject typeObject = types.get(typeName)
                JsonObject definitions = typeObject.getAsJsonObject('definitions')
                Set<Entry<String, JsonElement>> defEntries = definitions.entrySet()
                for (Map.Entry<String,JsonElement> defEntry : defEntries) {
                    String defName = defEntry.getKey()
                    JsonObject defObject = definitions.get(defName)
                    String defExternalId = defObject.get("id").getAsString()

                    Component component = new Component()
                        .name(defName)
                        .automated(true)
                        .property("objectType", "COMPONENT")
                        .integrationProvider(provider)
                        .externalId(defExternalId)

                    if (!appChildrenMap.containsKey(defExternalId)) {
                        compSyncClient.addImport(component)
                        appChildrenMap.put(defExternalId, component)
                    }

                    for (JsonObject version : defObject.getAsJsonArray('versions')) {
                        String versionExternalId = version.get("id").getAsString()

                        if (!versionChildren.containsKey(versionExternalId)) {
                            ComponentVersion componentVersion = new ComponentVersion()
                                .component(component)
                                .name(version.get("name").getAsString())
                                .automated(true)
                                .integrationProvider(provider)
                                .externalId(versionExternalId)

                            versionChildren.put(versionExternalId, componentVersion)

                            if (compVersionSyncClient.importList.size() + 1 >= limitPerQuery) {
                                compSyncClient.syncImports() // Component must sync before its version
                            }

                            compVersionSyncClient.addImport(componentVersion)
                        }
                    }
                }
            }

            ucrVersion.children(versionChildren.values().toArray(new ComponentVersion[0]))

            compSyncClient.syncImports() // Sync any remaining imports.
            println("Synced ${compSyncClient.importCount} components.")
            compVersionSyncClient.syncImports() // Sync any remaining imports.
            println("Synced ${compVersionSyncClient.importCount} component versions.")
        }
    }

    //--------------------------------------------------------------
    void importEnvironments(bulkApplicationList) {
        println "<b><u>>>>> Import Environments</u></b>"
        //List of all environments from CA that will be turned into UCR Application Targets
        def bulkEnvironmentList = [] as List
        def syncedEnvs = false
        long startImport = System.currentTimeMillis()

        bulkApplicationList.each { Application app ->
            if (app != null) {
                def nolioEnvironments = restHelper.getEnvironments(app.externalId)
                //println "# processing environments for application: ${app.name}"
                nolioEnvironments.each { Object env ->
                    def environment = new ApplicationEnvironment()
                        .name(env.get("name").getAsString())
                        .description(env.has("description") && env.get("description") != null ? env.get("description").getAsString() : "")
                        .integrationProvider(provider)
                        .externalId(env.get("id").getAsString())
                    if (app.getExternalId() != null) {
                        def parent = new Application().externalId(app.getExternalId())
                        environment.application(parent)
                    }
                    bulkEnvironmentList.add(environment)

                    if (bulkEnvironmentList.size() >= limitPerQuery) {
                        long startImportIntoUCR = System.currentTimeMillis()
                        def environmentsUpdated = new ApplicationEnvironment().sync(bulkEnvironmentList)
                        long endImportIntoUCR = System.currentTimeMillis()
                        println "# UCR Environments updated: " + environmentsUpdated.size() + " in " +
                                ((endImportIntoUCR - startImportIntoUCR) / 1000) + "s"
                        bulkEnvironmentList.clear()
                        syncedEnvs = true
                    }
                }
            }
        }

        if (bulkEnvironmentList.size() > 0) {
            long startImportIntoUCR = System.currentTimeMillis()
            def environmentsUpdated = new ApplicationEnvironment().sync(bulkEnvironmentList)
            long endImportIntoUCR = System.currentTimeMillis()
            println "# UCR Environments updated: ${environmentsUpdated.size()} in " +
                    "${(endImportIntoUCR - startImportIntoUCR) / 1000}s"
        }
        else if(!syncedEnvs) {
            println "No Environments to sync"
        }
    }

    //--------------------------------------------------------------
    void importProcesses(List<Application> bulkApplicationList, SyncClient<TaskPlan> procSyncClient) {
        println "<b><u>>>>> Import Processes</u></b>"

        bulkApplicationList.each { Application app ->
            if (app != null) {
                def nolioCategories = restHelper.getTemplateCategories(app.externalId)
                nolioCategories.each { Object category ->
                    def categoryId = category.get("id").getAsString()
                    def nolioDeploymentTemplates = restHelper.getDeploymentTemplates(app.externalId, categoryId)
                    //println "# processing deployment templates for application: ${app.name}"
                    nolioDeploymentTemplates.each { Object template ->
                        def process = new TaskPlan()
                            .name(template.get("name").getAsString())
                            .description(template.has("description") && template.get("description") != null ? template.get("description").getAsString() : "")
                            .integrationProvider(provider)
                            .externalId(template.get("id").getAsString())
                            .property("processId", template.get("id").getAsString())
                            .property("processName", template.get("name").getAsString())
                            .property("categoryId", categoryId)
                            .executionStep("ExecuteTask")
                            .taskType(TaskType.PluginTask)
                            .active(true)
                            .automated(true)
                            .application(app)

                        procSyncClient.addImport(process)
                    }
                }
            }
        }

        procSyncClient.syncImports() // Sync any remaining imports.
        println("Synced ${procSyncClient.importCount} processes.")
        procSyncClient.syncRemovals() // Sync objects that have been removed (paginates internally)
        println("Removed ${procSyncClient.removalCount} processes no longer in Nolio.")
    }

    //--------------------------------------------------------------
    def importArtifactPackagesViaDeploymentPlans(List<Application> bulkApplicationList) {
        println "<b><u>>>>> Import Versions</u></b>"
        //List of all artifact packages from CA that will be turned into UCR versions
        def bulkVersionList = [] as List
        def syncedVersions = false
        long startImport = System.currentTimeMillis()

        bulkApplicationList.each { Application app ->
            if (app != null) {
                def nolioProjects = restHelper.getApplicationProjects(app.externalId)
                nolioProjects.each { Object projectId ->
                    def nolioDeploymentPlans = restHelper.getDeploymentPlans(app.externalId, projectId)
                    nolioDeploymentPlans.each { Object deploymentPlan ->
                        if(deploymentPlan != null && deploymentPlan.hasProperty('artifactPackage') && deploymentPlan.artifactPackage
                        && deploymentPlan.artifactPackage != null && deploymentPlan.artifactPackage != '') {
                            def snapshot = new Version()
                                    .name(handleTrailingWhiteSpace(deploymentPlan.artifactPackage))
                                    .automated(true)
                                    .integrationProvider(provider)
                                    .createdDate(System.currentTimeMillis())
                                    //TODO: how do we get this???? can this just be the appid + artifact package name?
                                    .externalId(deploymentPlan.deploymentPlanId)
                                    .application(app)

                            bulkVersionList.add(snapshot)

                            if (bulkVersionList.size() > limitPerQuery) {
                                long startImportIntoUCR = System.currentTimeMillis()
                                def versionsUpdated = new Version().sync(bulkVersionList)
                                long endImportIntoUCR = System.currentTimeMillis()
                                println "# Versions updated: ${versionsUpdated.size()} in ${(endImportIntoUCR - startImportIntoUCR) / 1000}s"
                                bulkVersionList.clear()
                                syncedVersions = true
                            }
                        }
                    }
                }
            }
        }

        if (bulkVersionList.size() > 0) {
            long startImportIntoUCR = System.currentTimeMillis()
            def versionsUpdated = new Version().sync(bulkVersionList)
            long endImportIntoUCR = System.currentTimeMillis()
            println "# Versions updated: ${versionsUpdated.size()} in ${(endImportIntoUCR - startImportIntoUCR) / 1000}s"
        }
        else if(!syncedVersions) {
            println "No Versions to sync"
        }
    }

    //--------------------------------------------------------------
    def updateExecutingTasks() {
        println "<b><u>>>>> Update Executing Tasks</u></b>"
        long startImport = System.currentTimeMillis()
        def inventories = [] as List
        //List of all processes from UCD that will be turned into UCR Automated Tasks
        def executingTasks = new TaskExecution().getAllExecutingForProvider(provider.id.toString())

        if (executingTasks.size() > 0) {
            executingTasks.each {
               task ->
               if (task.propertyValues.deploymentIds != null) {
                    def deploymentIdArray = task.propertyValues.deploymentIds.split(',')
                    def incompleteDeployments = [] as List
                    def atLeastOneProcessFailed = false;
                    deploymentIdArray.each {
                        deploymentId ->
                            def deploymentResponse = restHelper.getDeploymentState(deploymentId)
                            def status = deploymentResponse.deploymentState
                            def description = deploymentResponse.description
                            def deploymentErrors = deploymentResponse.deploymentErrors

                            if(status == null){
                                task.fail()
                            }

                            if (status.indexOf("Succeeded") > -1) {
                                new Comment().task(task).comment("Job Execution Completed").save()
                            }
                            else if (status.indexOf("Failed") > -1) {
                                new Comment().task(task).comment("Job Execution Failed with status: $status").save()
                                atLeastOneProcessFailed = true;
                            }
                            else if (status.indexOf("Active") > -1) {
                                def matches = (description =~ /([0-9]+%)/)
                                def percent = matches[0][0]
                                if (description.indexOf("(Waiting for user input)") > -1) {
                                    new Comment().task(task).comment("$percent: Job Execution is waiting on manual intervention").save()
                                }else if(description.indexOf("(With errors)") > -1){
                                    new Comment().task(task).comment("$percent: Paused due to failure:: $deploymentErrors").save()
                                }else{
                                    new Comment().task(task).comment("$percent").save()
                                }
                                incompleteDeployments.add(deploymentId)
                            }
                            else {
                                new Comment().task(task).comment("Job Execution returned an unknown status of: $status").save()
                                incompleteDeployments.add(deploymentId)
                            }
                    }
                    if (atLeastOneProcessFailed) {
                        task.fail();
                    }
                    else if(incompleteDeployments.size() == 0) {
                        task.complete();
                        def updatePipeline = task.propertyValues.updatePipeline;
                        if (updatePipeline) {
                            def artifactPackageId = task.propertyValues.artifactPackageId;
                            def targets = task.propertyValues.targets.split(',')
                            targets.each {
                                targetId ->
                                def inventory = new Inventory();
                                inventory.applicationTarget(new ApplicationEnvironment().externalId(targetId));
                                inventory.version(new Version().externalId(artifactPackageId));
                                inventories.add(inventory);
                            }
                        }
                    }
                }
            }
        }

        if (inventories.size() > 0) {
            new Inventory().sync(inventories);
        }

        long endImport = System.currentTimeMillis()

        println "# Executing tasks updated: ${executingTasks.size()} " +
                            "in ${(startImport-endImport)/1000}s"
    }

    //--------------------------------------------------------------
    //Method that replaces the white spaces on versions name with an underscore
    def handleTrailingWhiteSpace(s) {
        if(s.endsWith(" ") || s.endsWith("_")) {
            //println "Trailing whitespace in name: $s"
            s = s + "_"
        }
        return s
    }


    //--------------------------------------------------------------
    //Print logs
    def printLog(type, message) {
        println "<span class=\""+type+"\">"+message+"</span>"
    }
}
