﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace test04
{
    /// <summary>
    /// 今回はBitmapクラスのマーシャリングをテーマにしたので、
    /// 以下のようにBitmapクラスで画像配列を管理しています。
    /// 
    /// しかし、マルチスレッドなどを導入した場合、安全ではありません。
    /// あるスレッドが、Bitmapクラスをロックしている時に、他のスレッドがさらにロックしようとするとエラーになります。
    /// 
    /// 実際のプログラムでは、少し面倒ですが、byte配列などで管理することをお勧めします。
    /// 
    /// 本プログラムは、あくまで参考程度のプログラムと考えてください。
    /// （引数チェックもしていない(-_-;)）
    /// 
    /// あと、Debugのプロパティ → デバッグ → ネイティブコードのデバッグを有効にする
    /// を有効にしていますので、NativeFunc内でブレークポイントを設定できます。
    /// </summary>
    public partial class Form1 : Form
    {
        private Bitmap dsp = null;
        private Wrapper.WrapperClass wr = new Wrapper.WrapperClass();

        public Form1()
        {
            InitializeComponent();

            // dspの初期化
            dsp = new Bitmap(300, 200, System.Drawing.Imaging.PixelFormat.Format8bppIndexed);
            grayScalePalette(dsp);

            // コントロールの初期化
            pictureBox1.Location = new Point(0, 0);
        }

        /// <summary>
        /// 描画する部分
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void pictureBox1_Paint(object sender, PaintEventArgs e)
        {
            pictureBox1.Width = dsp.Width;
            pictureBox1.Height = dsp.Height;

            e.Graphics.DrawImage(dsp, 0, 0, dsp.Width, dsp.Height);
        }

        /// <summary>
        /// ファイルを開く
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void openToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ssl.Text = "***";

            OpenFileDialog ofd = new OpenFileDialog();
            ofd.Filter = "Jpeg Files|*.jpg|Bitmap Files|*.bmp|All Files|*.*";


            if (ofd.ShowDialog() == DialogResult.OK)
            {
                if (dsp != null)
                {
                    dsp.Dispose(); // Bitmapクラスは、ちゃんとDisposeしましょう。リーク対策。
                }

                dsp = new Bitmap(ofd.FileName);

                if (dsp.PixelFormat != System.Drawing.Imaging.PixelFormat.Format8bppIndexed)
                {
                    ssl.Text = "Format8bppIndexed ではありません";
                    return;
                }
                else
                {
                    ssl.Text = "OK : Open";
                }

                grayScalePalette(dsp);

                pictureBox1.Invalidate(true); // これを実行しないと表示変更されません
            }
        }

        /// <summary>
        /// 反転処理実行
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void reverseToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ssl.Text = "***";

            int w = dsp.Width;
            int h = dsp.Height;
            Bitmap buf = new Bitmap(w, h, System.Drawing.Imaging.PixelFormat.Format8bppIndexed);
            grayScalePalette(buf);

            wr.Reverse(dsp, buf, w * h);

            dsp.Dispose(); // dspをいったん破棄してから、
            dsp = buf;     // bufと入れ替えます

            pictureBox1.Invalidate(true); // これを実行しないと表示変更されません
        }

        /// <summary>
        /// パレットをグレイスケールにする（定番のやり方）
        /// </summary>
        /// <param name="srcdst"></param>
        private void grayScalePalette(Bitmap srcdst)
        {
            System.Drawing.Imaging.ColorPalette pal = srcdst.Palette;

            for (int i = 0; i < 256; ++i)
            {
                pal.Entries[i] = Color.FromArgb(i, i, i);
            }

            srcdst.Palette = pal;
        }
    }
}
