#ifndef CatalystAdaptor_h
#define CatalystAdaptor_h

#include <catalyst.h>
#include <stdio.h>
#include <string.h>

//-----------------------------------------------------------------------------
/**
 * Initialize Catalyst.
 */
//-----------------------------------------------------------------------------
void do_catalyst_initialization(int argc, char* argv[]) {
  conduit_node* catalyst_init_params = conduit_node_create();
  for (int cc = 1; cc < argc; ++cc) {
    if (strcmp(argv[cc], "--output") == 0 && (cc + 1) < argc) {
      conduit_node_set_path_char8_str(catalyst_init_params, "catalyst/pipelines/0/type", "io");
      conduit_node_set_path_char8_str(catalyst_init_params, "catalyst/pipelines/0/filename", argv[cc + 1]);
      conduit_node_set_path_char8_str(catalyst_init_params, "catalyst/pipelines/0/channel", "grid");
      ++cc;
    }
    else {
      // pass scripts on command line
      char buf[256];
      snprintf(buf, 256, "catalyst/scripts/script%d", (cc - 1));
      conduit_node_set_path_char8_str(catalyst_init_params, buf, argv[cc]);
    }
  }
  conduit_node_set_path_char8_str(catalyst_init_params, "catalyst_load/implementation", "paraview");
  conduit_node_set_path_char8_str(catalyst_init_params, "catalyst_load/search_paths/paraview", PARAVIEW_IMPL_DIR);
  enum catalyst_status err = catalyst_initialize(catalyst_init_params);
  conduit_node_destroy(catalyst_init_params);
  if (err != catalyst_status_ok)
    printf("Failed to initialize Catalyst: %d\n", err);
}

//-----------------------------------------------------------------------------
/**
 * Execute per cycle
 */
//-----------------------------------------------------------------------------
void do_catalyst_execute(int cycle, double time, Grid* grid, Attributes* attribs) {
  conduit_node* catalyst_exec_params = conduit_node_create();
  conduit_node_set_path_int64(catalyst_exec_params, "catalyst/state/timestep", cycle);
  conduit_node_set_path_float64(catalyst_exec_params, "catalyst/state/time", time);

  // add channels; we only have 1 channel here; let's name it "grid"
  // declare the type of the channel; we're using Conduit Mesh Blueprint to describe the mesh and fields
  conduit_node_set_path_char8_str(catalyst_exec_params, "catalyst/channels/grid/type", "mesh");

  // create the mesh
  conduit_node* mesh = conduit_node_create();

  // add coordsets
  conduit_node_set_path_char8_str(mesh, "coordsets/coords/type", "uniform");

  conduit_node_set_path_int64(mesh, "coordsets/coords/dims/i", grid->Extent[1] - grid->Extent[0]);
  conduit_node_set_path_int64(mesh, "coordsets/coords/dims/j", grid->Extent[3] - grid->Extent[2]);
  conduit_node_set_path_int64(mesh, "coordsets/coords/dims/k", grid->Extent[5] - grid->Extent[4]);

  conduit_node_set_path_float64(mesh, "coordsets/coords/origin/x", grid->Extent[0]);
  conduit_node_set_path_float64(mesh, "coordsets/coords/origin/y", grid->Extent[2]);
  conduit_node_set_path_float64(mesh, "coordsets/coords/origin/z", grid->Extent[4]);

  conduit_node_set_path_float64(mesh, "coordsets/coords/spacing/x", grid->Spacing[0]);
  conduit_node_set_path_float64(mesh, "coordsets/coords/spacing/y", grid->Spacing[1]);
  conduit_node_set_path_float64(mesh, "coordsets/coords/spacing/z", grid->Spacing[2]);

  // add topology
  conduit_node_set_path_char8_str(mesh, "topologies/mesh/type", "uniform");
  conduit_node_set_path_char8_str(mesh, "topologies/mesh/coordset", "coords");

  // add cell-based density
  conduit_node_set_path_char8_str(mesh, "fields/density/association", "element");
  conduit_node_set_path_char8_str(mesh, "fields/density/topology", "mesh");
  conduit_node_set_path_char8_str(mesh, "fields/density/volume_dependent", "false");
  conduit_node_set_path_external_float32_ptr(mesh, "fields/density/values", attribs->Density, grid->NumberOfCells);
  conduit_node_set_path_external_node(catalyst_exec_params, "catalyst/channels/grid/data", mesh);

  enum catalyst_status err = catalyst_execute(catalyst_exec_params);
  if (err != catalyst_status_ok)
    printf("Failed to execute Catalyst: %d\n", err);
  conduit_node_destroy(catalyst_exec_params);
  conduit_node_destroy(mesh);
}

//-----------------------------------------------------------------------------
/**
 * Finalize Catalyst.
 */
//-----------------------------------------------------------------------------
void do_catalyst_finalization() {
  conduit_node* catalyst_fini_params = conduit_node_create();
  enum catalyst_status err = catalyst_finalize(catalyst_fini_params);
  if (err != catalyst_status_ok)
    printf("Failed to execute Catalyst: %d\n", err);
  conduit_node_destroy(catalyst_fini_params);
}

#endif
