import vtk

def readLongestParagraphs(filename, size, numberParagraphs=None):
    """
    Put numberParagraphs longest paragraphs from a file into a list.
    Each paragraph will be a separate element in this list.
    If numberParagraphs is missing or is too long, all paragraphs from the file will be used.
    """
    
    f = open(filename, 'r')
    data = f.read()

    # split text into paragraphs
    paragraphs = data.split("\n\n")

    # only keep paragraphs longer than 10 words
    paragraphs = [p for p in paragraphs if len(p)>10]

    # pick numberParagraphs longest paragraphs for analysis
    longest = sorted(paragraphs, key=len, reverse=True)
    actualNumberParagraphs = min(len(longest), numberParagraphs)
    print("read", actualNumberParagraphs, "paragraphs from", filename)
    size.append(actualNumberParagraphs)
    return longest[0:numberParagraphs]

def writeNodesEdges(nodeCoords,
                 scalar = [], name = [], power = [1,1],
                 nodeLabel = [],
                 edges = [],
                 method = 'vtkPolyData',
                 fileout = 'test'):
    """
    Store points and/or graphs as vtkPolyData or vtkUnstructuredGrid.
    Required argument:
    - nodeCoords is an array of node coordinates (nnodes,3)
    Optional arguments:
    - edges is a list of edges in the format [nodeID1,nodeID2]
    - scalar is the list of attributes, each is the list of scalars for all nodes
    - name is the list of scalars' names
    - power is the scaling list for attributes: 1 for r~scalars, 0.333 for V~scalars
    - nodeLabel is a list of node labels
    - method = 'vtkPolyData' or 'vtkUnstructuredGrid'
    - fileout is the output file name (will be given .vtp or .vtu extension)
    """

    points = vtk.vtkPoints()
    for node in nodeCoords:
        points.InsertNextPoint(node)

    if edges:
        line = vtk.vtkCellArray()
        line.Allocate(len(edges))
        for edge in edges:
            line.InsertNextCell(2)
            line.InsertCellPoint(edge[0])
            line.InsertCellPoint(edge[1])   # line from point edge[0] to point edge[1]

    if scalar:
        attribute0 = vtk.vtkFloatArray()
        attribute0.SetNumberOfComponents(1)
        attribute0.SetName(name[0])
        attribute0.SetNumberOfTuples(len(scalar[0]))
        for i, j in enumerate(scalar[0]):   # i becomes 0,1,2,..., and j runs through scalar[0]
            attribute0.SetValue(i,j**power[0])
        if len(scalar) > 1:
            attribute1 = vtk.vtkFloatArray()
            attribute1.SetNumberOfComponents(1)
            attribute1.SetName(name[1])
            attribute1.SetNumberOfTuples(len(scalar[1]))
            for i, j in enumerate(scalar[1]):   # i becomes 0,1,2,..., and j runs through scalar[1]
                attribute1.SetValue(i,j**power[1])

    if nodeLabel:
        label = vtk.vtkStringArray()
        label.SetName('tag')
        label.SetNumberOfValues(len(nodeLabel))
        for i, j in enumerate(nodeLabel):   # i becomes 0,1,2,..., and j runs through scalar
            label.SetValue(i,j)

    if method == 'vtkPolyData':
        polydata = vtk.vtkPolyData()
        polydata.SetPoints(points)
        if edges:
            polydata.SetLines(line)
        if scalar:
            polydata.GetPointData().AddArray(attribute0)
            if len(scalar) > 1:
                polydata.GetPointData().AddArray(attribute1)
        if nodeLabel:
            polydata.GetPointData().AddArray(label)
        writer = vtk.vtkXMLPolyDataWriter()
        writer.SetFileName(fileout+'.vtp')
        writer.SetInputData(polydata)
        writer.Write()
        print('wrote into', fileout+'.vtp')
    elif method == 'vtkUnstructuredGrid':
        # caution: ParaView's Tube filter does not work on vtkUnstructuredGrid
        grid = vtk.vtkUnstructuredGrid()
        grid.SetPoints(points)
        if edges:
            grid.SetCells(vtk.VTK_LINE, line)
        if scalar:
            grid.GetPointData().AddArray(attribute0)
            if len(scalar) > 1:
                grid.GetPointData().AddArray(attribute1)
        if nodeLabel:
            grid.GetPointData().AddArray(label)
        writer = vtk.vtkXMLUnstructuredGridWriter()
        writer.SetFileName(fileout+'.vtu')
        writer.SetInputData(grid)
        writer.Write()
        print('wrote into', fileout+'.vtu')
