import string
from gensim import corpora
from collections import defaultdict
from scipy.spatial import distance
import numpy as np
from sklearn import manifold
from nodesAndEdges import writeNodesEdges, readLongestParagraphs
from tqdm import tqdm

npar = 30   # target number of paragraphs per text (30 for a simple demo)
size = []   # actual number of paragraphs per text

documents = readLongestParagraphs('timeMachine.txt', size, npar)    # Herbert Wells
documents += readLongestParagraphs('oliverTwist.txt', size, npar)   # Charles Dickens
documents += readLongestParagraphs('adventuresOfHuckleberryFinn.txt', size, npar)   # Mark Twain
documents += readLongestParagraphs('theWarOfTheWorlds.txt', size, npar)   # Herbert Wells
documents += readLongestParagraphs('astro.txt', size, npar)         # astrophysics paper
documents += readLongestParagraphs('brothersKaramazov.txt', size, npar)   # Fyodor Dostoevsky

authorTags = [1, 2, 3, 1, 4, 5]
novelPerAuthorTags = [1, 1, 1, 2, 1, 1]

# convert line breaks and dashes to spaces, and remove punctuation
for i, p in enumerate(documents):
    tmp = p.replace('\n', ' ').replace('-',' ')
    for c in string.punctuation:
        tmp = tmp.replace(c,'')
    documents[i] = tmp

# remove common words and tokenize (break into words)
stoplist = set('for from a of the and to in at through'.split())
texts = [[word for word in document.lower().split() if word not in stoplist] for document in documents]

# count words across all paragraphs
frequency = defaultdict(int)
for text in texts:
    for token in text:
        frequency[token] += 1

# remove words that appear only once across all paragraphs
texts = [[token for token in text if frequency[token] > 1] for text in texts]

# build a dictionary of words from scratch (not related to above word count)
dictionary = corpora.Dictionary(texts)
# print(dictionary.token2id)   # print IDs of all words in the dictionary
nwords = len(dictionary.token2id)
print('built a global dictionary with', nwords, 'words')

# convert documents to sparse vectors containing tuples (wordID, wordCount);
# corpus is a list of paragraphs, each is a list of tuples
corpus = [dictionary.doc2bow(text) for text in texts]

# convert sparse vectors to full vectors of length nwords
n = sum(size)
fullCorpus = np.zeros((n,nwords), dtype=np.int32)
for i, d in enumerate(corpus):
    for word in d:
        id, count = word
        fullCorpus[i,id] = count

# connect pairs with at least 5 words in common
n, i = len(fullCorpus), -1
edges = []
for d1 in tqdm(fullCorpus):
    i += 1
    row = []
    for j, d2 in enumerate(fullCorpus):
        if i < j:
            if sum((d1!=0) * (d2!=0)) >=5:
                edges.append([i,j])

# break edges into two groups
edges1, edges2 = [], []
n1 = sum(size[:5])
for edge in edges:
    dostoevsky = False
    for node in range(sum(size[:5]),n): # one of the nodes is from Dostoevsky
        if node in edge:
            dostoevsky = True
    if dostoevsky:
        newEdge = [edge[0]-n1, edge[1]-n1]
        edges2.append(newEdge)
    else:
        edges1.append(edge)

import networkx as nx

# first 5 works
H1 = nx.Graph()
n1 = sum(size[:5])
H1.add_nodes_from(range(n1))
H1.add_edges_from(edges1)
pos = nx.spring_layout(H1, k=0.02, dim=3)   # dictionary (nodeID,array([x,y,z]))
xyz1 = np.zeros((n1,3))
for i in pos:
    xyz1[i,:] = pos[i]

# Dostoevsky
H2 = nx.Graph()
n2 = n - n1
H2.add_nodes_from(range(n2))
H2.add_edges_from(edges2)
pos = nx.spring_layout(H2, k=1, dim=3)   # dictionary (nodeID,array([x,y,z]))
xyz2 = np.zeros((n2,3))
for i in pos:
    xyz2[i,:] = pos[i] + 1.5

print(nx.number_of_nodes(H1)+nx.number_of_nodes(H2), 'nodes and', len(edges), 'edges')
print('Two groups of edges:', len(edges1), len(edges2))

author, novelPerAuthor = [], []
for i, s in enumerate(size[:5]):
    author += [authorTags[i]] * s
    novelPerAuthor += [novelPerAuthorTags[i]] * s

writeNodesEdges(xyz1, edges=edges1, scalar=[author,novelPerAuthor],
             name=['author','novel per author'], fileout='network7a')

author, novelPerAuthor = [], []
i, s = 5, size[5]
author += [authorTags[i]] * s
novelPerAuthor += [novelPerAuthorTags[i]] * s

writeNodesEdges(xyz2, edges=edges2, scalar=[author,novelPerAuthor],
             name=['author','novel per author'], fileout='network7b')
