import glob from 'glob-promise'
import * as path from 'path'
import fs from 'fs'
import fse from 'fs-extra'
import { fileURLToPath } from 'url'
import chalk from 'chalk'
const __filename = fileURLToPath(import.meta.url)
const __dirname = path.dirname(__filename)
const relpath = path.join.bind(path, __dirname)
import { spawn } from 'cross-spawn'
import moment from 'moment'
import { rm } from 'fs/promises'

const currentTime = () => moment().format('DD/MM/YYYY hh:MM:ss:mmm')

const argv = (str) => {
  const idx = process.argv.findIndex((a) => a.startsWith(str))
  if (idx > -1) {
    return process.argv[idx]
  }
  return null
}

const resolve = (pathParam) => path.resolve(__dirname, pathParam)

const createBuild = (libraryName, widgetFile, widgetConfigPath) => {
  if (!fs.existsSync(resolve(`../build`))) {
    fs.mkdirSync(resolve(`../build`), { recursive: true })
  }

  const f = fs.readFileSync(resolve('../webpack.config.js'), 'utf-8')
  const regexPathHtmlWebpackPluginTemplate = /\.\..*index\.html/;
  const matchedPathHtmlWebpackPluginTemplate = f.match(regexPathHtmlWebpackPluginTemplate)
  const pathHtmlWebpackPluginTemplate = matchedPathHtmlWebpackPluginTemplate && matchedPathHtmlWebpackPluginTemplate[0]

  const webpackConfig = argv('serve')
    ? f
      .replace(pathHtmlWebpackPluginTemplate,  pathHtmlWebpackPluginTemplate.replace('../', ''))
      .replaceAll('___LIBRARY_NAME___', libraryName)
      .replaceAll('___WIDGET___', '')
      .replace(/externals:\s*{[\s\S]*?},/g, '')
    : f
      .replace(pathHtmlWebpackPluginTemplate,  pathHtmlWebpackPluginTemplate.replace('../', ''))
      .replaceAll('___LIBRARY_NAME___', libraryName)
      .replaceAll('___WIDGET___', '.widget')

  const webpackConfigPath = resolve(`../build/webpack_${libraryName}.config.js`)

  fs.writeFileSync(webpackConfigPath, webpackConfig)

  return webpackConfigPath
}

const runWidgetNpm = (widgetFile, libraryName, widgetConfigPath) => {
  if (!fs.existsSync(resolve(`../${libraryName}`))) {
    fs.mkdirSync(resolve(`../${libraryName}`))
  }
  const webpackConfigPath = createBuild(libraryName, widgetFile, widgetConfigPath)
  console.log(chalk.blue(`[${currentTime()}]`), chalk.yellow(`[Compiling ${libraryName}]`))

  const args = ['--color', '--watch', '--config', webpackConfigPath];

  if (argv('serve')) {
    args.unshift('serve')
  }
  const npm = spawn('./node_modules/.bin/webpack', args)
  npm.stdout.on('data', (data) => {
    console.log(chalk.blue(`[${currentTime()}]`), chalk.yellow(`[${libraryName}]`))
    console.log(chalk.whiteBright(data))
  })
  npm.stderr.on('data', (data) => {
    if (`${data}`.startsWith('[BABEL] Note: ')) {
      console.log(chalk.gray(data))
    } else if (`${data}`.startsWith('<i> [webpack-dev-server]')) {
      console.log(chalk.green(data))
    } else {
      console.log(chalk.red(data))
    }
  });
  ['beforeExit', 'exit'].forEach(eventName => process.on(eventName, () => npm.kill(-9)))

}

const compile = async () => {
  const files = await glob('**/config.json', {cwd: relpath('../'), ignore: ['**/node_modules/**', '**/build/**', '**/dist_sdk/**', '**/dist/**']} )
  return await Promise.all(files.map(async file => {
    const widgetConfig = JSON.parse(fs.readFileSync(resolve(`../${file}`), 'utf-8'))
    const libraryName = widgetConfig.uniqueIdentifier
    const widgetFile = (await glob(`**/${libraryName}/**/index.widget.js`, {cwd: relpath('../'), ignore: ['**/node_modules/**', '**/build/**', '**/dist_sdk/**', '**/dist/**']}))[0]
    return  {
      widgetFile,
      widgetConfigPath: file,
      libraryName
    }
  }))
}

const deleteBuildLibrary = async (path) => {
  console.log(chalk.green('\tRemoving build library'))
  await rm(path, { recursive: true, force: true })
}

const main = async () => {
  if (!fs.existsSync(resolve('../build'))) {
    fs.mkdirSync(resolve('../build'))
  } else {
    await deleteBuildLibrary(resolve('../build'))
    fs.mkdirSync(resolve('../build'))
  }
  const widgetsList = await compile()

  for (const widget of widgetsList) {
    const {libraryName, widgetFile, widgetConfigPath} = widget
    runWidgetNpm(widgetFile, libraryName, widgetConfigPath)
  }
}

main()