import webpack from 'webpack'
import archiver from 'archiver'
import glob from 'glob-promise'
import * as path from 'path'
import fs from 'fs'
import {fileURLToPath} from 'url'
import prompts from 'prompts'
const __filename = fileURLToPath(import.meta.url)
const __dirname = path.dirname(__filename)
const relpath = path.join.bind(path, __dirname)
import { rm } from 'fs/promises'
import chalk from 'chalk'

const argv = (str) => {
  const idx = process.argv.findIndex((a) => a.startsWith(str))
  if (idx > -1) {
    return process.argv[idx]
  }
  return null
}

const webpackAsync = async (webpackConfig) => {
  return new Promise((resolve, reject) => {
    webpack(webpackConfig, (err, stats) => {
      if (stats.hasErrors() || err) {
        reject(err ?? stats.compilation.errors)
        return
      }
      resolve()
    });
  })
}

const changeWebpackConfigForProduction = async (webpackConfigText, libraryName, widgetFilePath) => {
  const parsedWebpackConfigText = webpackConfigText.replaceAll('___LIBRARY_NAME___', libraryName)
  fs.writeFileSync(path.resolve(__dirname, `../dist/${libraryName}/webpack_${libraryName}.config.js`), parsedWebpackConfigText)

  const webpackConfigObject = await import(`../dist/${libraryName}/webpack_${libraryName}.config.js`).then(m => m.default)

  webpackConfigObject.entry = widgetFilePath
  webpackConfigObject.mode = 'production'
  webpackConfigObject.output = {
    filename: `main.min.js`,
    path: `${path.resolve(__dirname, '../dist')}/${libraryName}`,
    library: libraryName,
    assetModuleFilename: `assets/[hash][ext][query]`
  }

  fs.unlinkSync(path.resolve(__dirname, `../dist/${libraryName}/webpack_${libraryName}.config.js`))

  return webpackConfigObject
}

const validateWidgetConfig = async (widgetConfig) => {
  const compareArrays = (arr1, arr2) => arr1.every(a => arr2.some(b => a === b));
  const requiredFields = ['engineVersion', 'uStoreVersion', 'uniqueIdentifier', 'displayName', 'description', 'configurationHelpText', 'defaultConfiguration', 'widgetType']
  const configKeys = Object.keys(widgetConfig)
  const keysValid = compareArrays(requiredFields, configKeys) && configKeys.length === requiredFields.length
  const engineVersionValid = /^([0-9.]+)$/g.test(widgetConfig.engineVersion)
  const uStoreVersionValid = /^([0-9.]+)$/g.test(widgetConfig.uStoreVersion)
  const uniqueIdentifierValid = /^([a-zA-Z0-9_]+)$/g.test(widgetConfig.uniqueIdentifier)
  return keysValid && engineVersionValid && uStoreVersionValid && uniqueIdentifierValid
}

const buildLibrary = async (widgetFilePath, libraryName, widgetConfigPath) => {
  console.log(chalk.green('\tCreating dist folder'))
  if (!fs.existsSync(path.resolve(__dirname, `../dist`))) {
    fs.mkdirSync(path.resolve(__dirname, `../dist`))
  }
  if (!fs.existsSync(path.resolve(__dirname, `../dist/${libraryName}`))) {
    fs.mkdirSync(path.resolve(__dirname, `../dist/${libraryName}`))
  }

  console.log(chalk.green('\tCopying widget configuration'))
  fs.copyFileSync(widgetConfigPath, relpath(`../dist/${libraryName}/config.json`))
  const jsonConfig = JSON.parse(fs.readFileSync(relpath(`../dist/${libraryName}/config.json`), 'utf-8'))
  const configValid = await validateWidgetConfig(jsonConfig)
  if (!configValid) {
    console.log(chalk.red('Widget configuration is invalid'))
    process.exit(1)
  }
  console.log(chalk.green('\tCopying widget thumbnail'))
  fs.copyFileSync(`${libraryName}/thumbnail.png`, relpath(`../dist/${libraryName}/thumbnail.png`))

  console.log(chalk.green('\tConfiguring webpack'))
  const webpackConfigText = fs.readFileSync(path.resolve(__dirname, '../webpack.config.js'), 'utf-8')
  const webpackConfig = await changeWebpackConfigForProduction(webpackConfigText, libraryName, widgetFilePath)

  console.log(chalk.green('\tCompiling widget'))
  try {
    await webpackAsync(webpackConfig)
  } catch (err) {
    console.error(err)
  }
}

const archiveLibrary = async (libraryName) => {
  console.log(chalk.green('\tCreating ZIP'))
  const output = fs.createWriteStream( `${relpath('../dist')}/${libraryName}.zip` );
  const archive = archiver('zip', {
    zlib: { level: 9 } // Sets the compression level.
  });
  archive.on('warning', function(err) {
    if (err.code === 'ENOENT') {
      console.warn(err)
    } else {
      throw err;
    }
  });

  archive.on('error', function(err) {
    throw err;
  });
  archive.pipe(output);
  archive.glob('**', {cwd:`${relpath('../dist')}/${libraryName}` });

  await archive.finalize();
}

const deleteBuildLibrary = async (libraryName) => {
  console.log(chalk.green('\tRemoving build library'))
  await rm(`${relpath('../dist')}/${libraryName}`, { recursive: true, force: true })
}

const getWidgetsData = async () => {
  const widgetsPaths = await glob('**/index.widget.js', {cwd: relpath('../'), ignore: ['**/node_modules/**', '**/build/**', '**/dist_sdk/**', '**/dist/**']})
  return await Promise.all(widgetsPaths.map(async (widgetPath) => {
    const widgetFilePath = relpath('../', widgetPath)
    const widgetConfigPath = relpath('..', widgetPath, '../../config.json')
    const widgetConfigExists = fs.existsSync(widgetConfigPath)
    const widgetConfig = widgetConfigExists && !argv('config') ? JSON.parse(fs.readFileSync(widgetConfigPath, 'utf-8')) : await getWidgetConfig()
    const libraryName = widgetConfig.uniqueIdentifier
    return  {
      widgetConfigPath,
      widgetFilePath,
      libraryName
    }
  }))
}

const createWidgetLibrary = async ({ widgetConfigPath, widgetFilePath, libraryName }) => {
  await buildLibrary(widgetFilePath, libraryName, widgetConfigPath)
  await archiveLibrary(libraryName)
  await deleteBuildLibrary(libraryName)
}

const getWidgetConfig = async () => {
  const questions = [
    {
      type: 'number',
      name: 'engineVersion',
      message: 'Engine version(numeric only)',
      initial: '1',
    },
    {
      type: 'text',
      name: 'uStoreVersion',
      message: 'uStore version(only numbers and .)',
      initial: '15.1.10259',
    },
    {
      type: 'text',
      name: 'uniqueIdentifier',
      message: 'Unique identifier(only letters, numbers and _)',
      initial: 'xmpie_Sample_1',
    },
    {
      type: 'text',
      name: 'displayName',
      message: 'Display name',
      initial: 'Sample Widget',
    },
    {
      type: 'text',
      name: 'description',
      message: 'Description',
      initial: 'Sample Widget',
    },
    {
      type: 'text',
      name: 'configurationHelpText',
      message: 'Configuration help text',
      initial: 'How to configure your widget',
    },
    {
      type: 'text',
      name: 'defaultConfiguration',
      message: 'Default configuration(JSON format)',
      initial: '{}',
    }
  ]
  return prompts(questions)
}

const main = async () => {
  const widgets = await getWidgetsData()
  for (const widget of widgets) {
    console.log(chalk.yellow('Building Widget', widget.libraryName))
    await createWidgetLibrary(widget)
  }
  process.exit(0)
}

main()
