import React, { useEffect, useState } from 'react'

import { PRODUCTS_PAGE_SIZE } from './FeaturedProducts/constants.js'
import {
  FeaturedProductsContainer,
} from './FeaturedProducts/containers/FeaturedProductsContainer.js'
import RefreshIcon from './assets/icons/refresh.svg'

import '../styles/index.widget.css'
import './common/styles/styles.css'

const IndexWidget = ( { config, uStoreProvider } ) => {
  const [id] = useState(() => Date.now().toString(36) + Math.random().toString(36).slice(2, 6))
  const [configState, setConfigState] = useState( null )
  const [featuredProducts, setFeaturedProducts] = useState( [] )
  const [isLoading, setIsLoading] = useState( false )
  const isPlaceholderCase = !configState?.productGroupID
    && ( configState?.productsIDs?.length === 0 || configState?.productsIDs === undefined )

  useEffect( () => {
    const controller = new AbortController()
    const { signal } = controller
    const { securityToken, languageCode, apiUrl } = uStoreProvider.contextService.context

    const headers = {
      Accept: 'application/json',
      'Content-Type': 'application/json',
      'Accept-Language': languageCode,
      Authorization: `uStore ${securityToken}`,
    }

    const getProductIDByFriendlyID = async ( friendlyID ) => {
      const productIdResponse = await fetch( `${apiUrl}/v1/store/products/id?friendlyID=${friendlyID}`, {
        method: 'GET',
        headers,
        signal,
      } )

      if ( !productIdResponse.ok ) {
        const errorMsg = await productIdResponse.text()
        throw new Error( errorMsg )
      }

      return productIdResponse.json()
    }

    const getProductsByIDs = async ( productsIDs ) => {
      const promisesProductIDs = productsIDs.map( getProductIDByFriendlyID )
      const productIDs = await Promise.allSettled( promisesProductIDs )
      const productIDsFiltered = productIDs.reduce( ( arrProductsIDs, productID ) => (
        productID.value ? [...arrProductsIDs, productID.value] : arrProductsIDs
      ), [] )
      if ( productIDsFiltered.length === 0 ) {
        return []
      }
      const products = await uStoreProvider.api.products.getProductsByIDs( productIDsFiltered )
      return products
    }

    const getCategoryIDByFriendlyID = async ( productGroupID ) => {
      const categoryIdResponse = await fetch( `${apiUrl}/v1/store/categories/id?friendlyID=${productGroupID}`, {
        method: 'GET',
        headers,
        signal,
      } )

      if ( !categoryIdResponse.ok ) {
        const errorMsg = await categoryIdResponse.text()
        throw new Error( errorMsg )
      }

      return categoryIdResponse.json()
    }

    const getProductsByProductGroupID = async ( productGroupID ) => {
      const newCategoryID = await getCategoryIDByFriendlyID( Number( productGroupID ) )
      const { Products } = await uStoreProvider.api.products.getProducts( newCategoryID, 1, PRODUCTS_PAGE_SIZE )
      return Products
    }

    const getProducts = async ( { productGroupID, productsIDs } ) => {
      const products = productGroupID
        ? await getProductsByProductGroupID( productGroupID )
        : productsIDs?.length > 0
          ? await getProductsByIDs( productsIDs )
          : []
      if ( products.length === 0 ) {
        return []
      }
      uStoreProvider.state.customState.set( `featuredProductsWidget_${id}`, products )
      setFeaturedProducts( products )
      return products
    };

    ( async () => {
      const { productGroupID, productsIDs } = JSON.parse( config )
      try {
        if ( productGroupID || productsIDs?.length > 0 ) {
          setIsLoading( true )
          await getProducts( { productGroupID, productsIDs } )
        }
      } catch ( err ) {
        console.error( err )
      } finally {
        setIsLoading( false )
      }
    } )()

    return () => {
      controller.abort()
    }
  }, [] )

  useEffect( () => {
    if ( config ) {
      try {
        const parsedConfig = JSON.parse( config )
        setConfigState( parsedConfig )
      } catch ( e ) {
        console.error( e )
      }
    }
  }, [config] )

  if ( isLoading ) {
    return (
      <div className="xw-featured-products-loading">
        <RefreshIcon className="xw-featured-products-refresh-icon xw-featured-products-rotating-icon" />
      </div>
    )
  }

  return (
    <>
      {
        !!configState
        && (
          ( ( configState.productGroupID || configState.productsIDs?.length >= 0 )
            && featuredProducts.length > 0 )
          || isPlaceholderCase
        )
        && (
          <FeaturedProductsContainer
            configState={configState}
            isPlaceholderCase={isPlaceholderCase}
            uStoreProvider={uStoreProvider}
            id={id}
          />
        )
      }
    </>
  )
}

export default IndexWidget
