import React from 'react'
import classNames from 'classnames'
import { Swiper, SwiperSlide } from 'swiper/react'
import { Keyboard, Navigation } from 'swiper/modules'

import { MAX_FEATURED_PRODUCTS, SWIPER_BREAKPOINTS } from '../constants.js'
import { TextPacksUnitsOfMeasure } from './TextPacksUnitsOfMeasure.js'
import { TextItemsUnitsOfMeasure } from './TextItemsUnitsOfMeasure.js'
import { Inventory } from './Inventory.js'
import ArrowLeftIcon from '../../assets/icons/left_arrow.svg'
import ArrowRightIcon from '../../assets/icons/right_arrow.svg'
import PlaceholderIcon from '../../assets/icons/placeholder.svg'

import 'swiper/css'
import 'swiper/css/pagination'
import 'swiper/css/navigation'
import '../../common/styles/SwiperStyleFeaturedProducts.css'
import '../FeaturedProducts.css'

export const FeaturedProducts = React.forwardRef( ( props, swiperRef ) => {
  const {
    configState,
    onInitSlider,
    isSliderFilled,
    isSliderOverFilled,
    isTopSeparator,
    isBothSeparator,
    isBottomSeparator,
    currency,
    featuredProductsWidget,
    isPlaceholderCase,
    placeholderProductsWidget,
    isLeftAlignedSwiperWrapper,
    isRightAlignedSwiperWrapper,
    isCenterAlignedSwiperWrapper,
    navigateToDetailedProductPage,
  } = props

  const featuredProductsWidgetLength = featuredProductsWidget?.length
  const configImageBoxHeight = {
    '--wg1-desktop-image-box-height': configState.imageBoxHeightDesktop && `${configState.imageBoxHeightDesktop}px`,
    '--wg1-mobile-image-box-height': configState.imageBoxHeightMobile && `${configState.imageBoxHeightMobile}px`,
  }

  return (
    <div
      className="xw-featured-products-container"
      style={{ marginBlock: configState.margins }}
    >

      {
        ( isTopSeparator || isBothSeparator ) && (
          <hr className="xw-featured-products-container-divider-top" />
        )
      }

      <h2
        className="xw-featured-products-title"
        style={{
          textAlign: configState.titleAlignment?.trim().toLowerCase(),
          color: configState.titleColor?.trim().toLowerCase(),
          fontSize: configState.titleSize,
        }}
      >
        {configState.title}
      </h2>
      <Swiper
        ref={swiperRef}
        allowTouchMove={isSliderOverFilled}
        breakpoints={{
          // => 320px
          [SWIPER_BREAKPOINTS.XS.SIZE]: {
            ...SWIPER_BREAKPOINTS.XS.SETTINGS,
            slidesPerView: configState.slidesPerViewSizeXS
              ? configState.slidesPerViewSizeXS
              : featuredProductsWidgetLength > 3 && placeholderProductsWidget.length > 3
                ? 2
                : 2.4,
            centeredSlides: ( isSliderFilled && featuredProductsWidgetLength > 3 )
                && ( isSliderFilled && placeholderProductsWidget.length > 3 ),
          },
          // => 768px
          [SWIPER_BREAKPOINTS.M.SIZE]: configState.slidesPerViewSizeM ? {
            ...SWIPER_BREAKPOINTS.M.SETTINGS,
            slidesPerView: configState.slidesPerViewSizeM,
            centeredSlides: false
          } : SWIPER_BREAKPOINTS.M.SETTINGS,
          // => 1200px
          [SWIPER_BREAKPOINTS.L.SIZE]: configState.slidesPerViewSizeL ? {
            ...SWIPER_BREAKPOINTS.L.SETTINGS,
            slidesPerView: configState.slidesPerViewSizeL,
            centeredSlides: false
          } : SWIPER_BREAKPOINTS.L.SETTINGS,
        }}
        className="xw-featured-products-swiper"
        keyboard={{
          enabled: true,
          onlyInViewport: false,
        }}
        loop={isSliderOverFilled}
        modules={[Navigation, Keyboard]}
        navigation={{
          nextEl: '.xw-featured-products-swiper-button-next',
          prevEl: '.xw-featured-products-swiper-button-prev',
        }}
        onInit={onInitSlider}
        wrapperClass={classNames( 'swiper-wrapper', {
          'xw-featured-products-swiper-wrapper-left': isLeftAlignedSwiperWrapper,
          'xw-featured-products-swiper-wrapper-right': isRightAlignedSwiperWrapper,
          'xw-featured-products-swiper-wrapper-center': isCenterAlignedSwiperWrapper,
        } )}
      >
        {( isPlaceholderCase
          ? placeholderProductsWidget
          : ( featuredProductsWidget || [] ) ).slice( 0, MAX_FEATURED_PRODUCTS )
          .map( ( slide, index ) => {
            const isItemsUnit = slide.MinimumQuantity > 1
                && !slide.Unit.PackType && slide.Unit.ItemType === 'Item'

            return (
              <SwiperSlide key={slide.FriendlyID || index}>
                <a
                  className="xw-featured-products-link"
                  onClick={!isPlaceholderCase ? navigateToDetailedProductPage( slide ) : null}
                >
                  <div
                    className={classNames(
                      'xw-featured-products-image-block',
                      {
                        'xw-featured-products-placeholder-wrapper': !slide.ImageUrl,
                      },
                    )}
                    style={{
                      backgroundColor: configState.imageBoxBackgroundColor,
                      padding: configState.imageBoxPadding,
                      ...configImageBoxHeight,
                    }}
                  >
                    {
                      slide.ImageUrl
                        ? <img alt={slide.Name} src={slide.ImageUrl} />
                        : (
                          <PlaceholderIcon
                            className="xw-featured-products-placeholder"
                            height="100%"
                            width="100%"
                          />
                        )
                    }

                  </div>

                  <div className="xw-featured-products-text-block">
                    <p className="xw-featured-products-name">{slide.Name}</p>
                    {slide.Unit?.PackType && <TextPacksUnitsOfMeasure slide={slide} />}
                    {isItemsUnit && <TextItemsUnitsOfMeasure slide={slide} />}
                    {slide.MinimumPrice?.Price
                        && (
                          <p className="xw-featured-products-price xw-featured-products-reset-margin">
                            From
                            {' '}
                            {currency.Symbol}
                            {slide.MinimumPrice?.Price}
                            {' '}
                            {currency.Code}
                          </p>
                        )}
                    {
                      slide.Inventory && (
                        <Inventory
                          inventory={slide.Inventory}
                          minQuantity={slide.Configuration.Quantity.Minimum}
                        />
                      )
                    }
                  </div>
                </a>
              </SwiperSlide>
            )
          } )}

        {
          isSliderOverFilled && (
            <ArrowLeftIcon
              className="xw-featured-products-swiper-button-prev"
              role="button"
              style={configImageBoxHeight}
            />
          )
        }

        {
          isSliderOverFilled && (
            <ArrowRightIcon
              className="xw-featured-products-swiper-button-next"
              role="button"
              style={configImageBoxHeight}
            />
          )
        }

      </Swiper>

      {
        ( isBottomSeparator || isBothSeparator ) && (
          <hr className="xw-featured-products-container-divider-bottom" />
        )
      }

    </div>
  )
} )
