import React, {
  Fragment, useCallback, useEffect, useRef, useState,
} from 'react'
import classNames from 'classnames'

import { HeaderWidget } from '../../common/components/HeaderWidget/HeaderWidget.js'
import { CarouselItemSlideTopMiddle } from '../components/CarouselItemSlideTopMiddle.js'
import { boxAnimations } from '../constants.js'
import ArrowRightIcon from '../../assets/icons/right_arrow.svg'
import ArrowLeftIcon from '../../assets/icons/left_arrow.svg'
import { CarouselItemSlideUnder } from '../components/CarouselItemSlideUnder.js'

import '../styles/CarouselContainer.css'

export function CarouselContainer( { configState } ) {
  const {
    boxCtaText,
    boxCtaTextColor,
    boxCtaBackgroundColor,
    sectionTitle,
    sectionSubtitle,
    sectionTitlesColor,
    boxes,
  } = configState
  const [activeIndex, setActiveIndex] = useState( 0 )
  const boxTextLocation = configState.boxTextLocation.trim().toLowerCase()
  const firstElements = activeIndex === 0
  const lastElement = boxes.length - 1
  const isTopOrMiddleLocation = ['top', 'middle'].includes( boxTextLocation )
  const isUnderLocation = ['under'].includes( boxTextLocation )
  const [isLastSlideVisible, setIsLastSlideVisible] = useState( false )
  const refLastSlide = useRef( null )

  const goToPrev = useCallback( ( evt ) => {
    evt.preventDefault()

    if ( activeIndex > 0 ) {
      setActiveIndex( activeIndex - 1 )
    }
  }, [activeIndex] )

  const goToNext = useCallback( ( evt ) => {
    evt.preventDefault()

    if ( activeIndex < boxes.length - 1 ) {
      setActiveIndex( activeIndex + 1 )
    }
  }, [activeIndex, boxes] )

  useEffect( () => {
    const refLastSlideCurrent = refLastSlide.current

    if ( refLastSlideCurrent ) {
      const observer = new IntersectionObserver(
        ( entries ) => {
          const [entry] = entries
          setIsLastSlideVisible( entry.isIntersecting )
        },
        {
          root: null,
          threshold: 0.1,
        },
      )

      observer.observe( refLastSlideCurrent )

      return () => {
        observer.unobserve( refLastSlideCurrent )
      }
    }

    return () => {}
  }, [] )

  return (
    <section
      className={`xw-featured-carousel-container xw-featured-anim-${configState.entranceAnimation}`}
      style={{ marginBlock: configState.margins }}
    >
      <HeaderWidget
        sectionSubtitle={sectionSubtitle}
        sectionTitle={sectionTitle}
        sectionTitlesColor={sectionTitlesColor}
      />
      <div className="xw-featured-carousel-wrapper">

        <ArrowLeftIcon
          className={classNames( {
            'xw-featured-carousel-arrow-prev-top-middle': isTopOrMiddleLocation,
            'xw-featured-carousel-arrow-prev-under': isUnderLocation,
            'xw-featured-carousel-arrow-disabled': firstElements,
          } )}
          height={30}
          onClick={goToPrev}
          role="button"
          width={30}
        />

        <ArrowRightIcon
          className={classNames( {
            'xw-featured-carousel-arrow-next-top-middle': isTopOrMiddleLocation,
            'xw-featured-carousel-arrow-next-under': isUnderLocation,
            'xw-featured-carousel-arrow-disabled': isLastSlideVisible,
          } )}
          height={30}
          onClick={goToNext}
          role="button"
          width={30}
        />

        <div className="xw-featured-carousel-content-wrapper">
          <ul
            className="xw-featured-carousel-content"
            style={{ '--active-featured-carousel-index': activeIndex }}
          >
            {boxes.map( ( slide, index ) => (
              <Fragment key={index}>

                {
                  isTopOrMiddleLocation && (
                    <CarouselItemSlideTopMiddle
                      ref={refLastSlide}
                      boxAnimation={boxAnimations[configState.boxAnimation]}
                      boxCtaBackgroundColor={boxCtaBackgroundColor}
                      boxCtaText={boxCtaText}
                      boxCtaTextColor={boxCtaTextColor}
                      boxTextLocation={boxTextLocation}
                      index={index}
                      lastElement={lastElement}
                      slide={slide}
                    />
                  )
                }

                {
                  isUnderLocation && (
                    <CarouselItemSlideUnder
                      ref={refLastSlide}
                      boxAnimation={boxAnimations[configState.boxAnimation]}
                      boxCtaBackgroundColor={boxCtaBackgroundColor}
                      boxCtaText={boxCtaText}
                      boxCtaTextColor={boxCtaTextColor}
                      boxTextLocation={boxTextLocation}
                      index={index}
                      lastElement={lastElement}
                      slide={slide}
                    />
                  )
                }

              </Fragment>
            ) )}
          </ul>
        </div>
      </div>
    </section>
  )
}
