import os
from requests import Session
from urllib.parse import urljoin
from .peewee import SqliteDatabase, Model, IntegerField, TextField, chunked
from warnings import simplefilter

simplefilter("ignore")
db = SqliteDatabase(None, pragmas={"foreign_keys": 1})


class BaseModel(Model):
    class Meta:
        database = db


class Game(BaseModel):
    game_pk = IntegerField(primary_key=True)
    official_date = TextField()
    game_date = TextField()
    away_team = TextField()
    home_team = TextField()
    content_link = TextField()


class LazyMLB:
    def __init__(self, cache_dir, config):
        self.mlb_api = "https://statsapi.mlb.com/api/v1/"
        self.lazy_api = "https://freesports.ddns.net/getM3U8.php"
        self.key_server = "freesports.ddns.net"
        self.s = Session()
        if "http_proxy" in config:
            self.s.proxies.update(
                {
                    "http": config["http_proxy"],
                    "https": config["http_proxy"],
                }
            )
        DB = os.path.join(cache_dir, "lazymlb0.db")
        db.init(DB)
        db.connect()
        db.create_tables(
            [Game],
            safe=True,
        )

    def fetch_teams(self):
        return

    def fetch_games_date(self, date):
        schedule_url = urljoin(self.mlb_api, "schedule")
        params = {"language": "en", "sportId": "1", "date": date}
        r = self.s.get(schedule_url, params=params)
        r.raise_for_status()
        res = r.json()

        def games(res):
            for date in res["dates"]:
                for game in date["games"]:
                    yield {
                        "game_pk": game["gamePk"],
                        "official_date": date["date"],
                        "game_date": game["gameDate"],
                        "away_team": game["teams"]["away"]["team"]["name"],
                        "home_team": game["teams"]["home"]["team"]["name"],
                        "content_link": game["content"]["link"],
                    }

        with db.atomic():
            for batch in chunked(games(res), 79):
                Game.replace_many(batch).execute()

    def get_games_date(self, date):
        if Game.select().where(Game.official_date == date).count() == 0:
            self.fetch_games_date(date)
        return Game.select().where(Game.official_date == date).order_by(Game.game_date)

    def get_game(self, game_pk):
        return Game.get(Game.game_pk == game_pk)

    def get_game_content(self, game_pk):
        game = Game.get(Game.game_pk == game_pk)
        content_url = urljoin(self.mlb_api, game.content_link)
        r = self.s.get(content_url)
        r.raise_for_status()
        return r.json()

    def get_media(self, date, media_id):
        params = {
            "league": "MLB",
            "date": date,
            "id": media_id,
            "cdn": "akc",
        }
        r = self.s.get(self.lazy_api, params=params, verify=False)
        r.raise_for_status()
        return r.text

    def get_media66(self, media_id):
        headers = {
            "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/99.0.0.0 Safari/537.36",
            "Accept": "application/json, text/plain, */*",
            "Accept-Language": "en-US,en;q=0.9",
        }
        r = self.s.get("https://api.mlb66.ir/api/sport/schedule", headers=headers)
        r.raise_for_status()
        for game in r.json()["games"]:
            for stream in game["streams"]:
                if stream["mediaid"] == str(media_id):
                    return (stream["url"], stream["keys"])
