import sys
import xbmc
import xbmcgui
from xbmc import Monitor, Player
from requests import Session
from requests.exceptions import ReadTimeout
from threading import Thread, Event
import traceback


class AcestreamInfo(object):
    def __init__(self):
        self.enabled = False
        self.window = xbmcgui.Window(10116)

        _zero = self.window.getWidth() // 2
        cl = (_zero, _zero + 60, _zero + 200, _zero + 260)
        self._title = xbmcgui.ControlLabel(
            _zero, 10, 565, 25, "Acestream Information", "font13", "0xff0084ff", "", 0x00000004
        )
        self._peers = xbmcgui.ControlLabel(cl[0], 40, 100, 25, "Peers:", "font12", "0x7fffffff", "", 0x00000004)
        self._peers_info = xbmcgui.ControlLabel(cl[1], 40, 100, 25, "0", "font12", "0xffffffff", "", 0x00000004)

        self._speed_down = xbmcgui.ControlLabel(cl[0], 65, 50, 25, "DL:", "font12", "0x7fffffff", "", 0x00000004)
        self._speed_down_info = xbmcgui.ControlLabel(cl[1], 65, 120, 25, "0", "font12", "0xffffffff", "", 0x00000004)
        self._downloaded = xbmcgui.ControlLabel(cl[2], 65, 50, 25, "DLT:", "font12", "0x7fffffff", "", 0x00000004)
        self._downloaded_info = xbmcgui.ControlLabel(cl[3], 65, 120, 25, "0", "font12", "0xffffffff", "", 0x00000004)

        self._speed_up = xbmcgui.ControlLabel(cl[0], 90, 50, 25, "UP:", "font12", "0x7fffffff", "", 0x00000004)
        self._speed_up_info = xbmcgui.ControlLabel(cl[1], 90, 120, 25, "0", "font12", "0xffffffff", "", 0x00000004)
        self._uploaded = xbmcgui.ControlLabel(cl[2], 90, 50, 25, "UPT:", "font12", "0x7fffffff", "", 0x00000004)
        self._uploaded_info = xbmcgui.ControlLabel(cl[3], 90, 100, 25, "0", "font12", "0xffffffff", "", 0x00000004)

    def update(self, info):
        self._peers_info.setLabel(str(info["peers"]))
        self._speed_down_info.setLabel(str(info["speed_down"]))
        self._downloaded_info.setLabel(str(info["downloaded"]))
        self._speed_up_info.setLabel(str(info["speed_up"]))
        self._uploaded_info.setLabel(str(info["uploaded"]))

    def enable(self):
        self.window.addControl(self._title)
        self.window.addControl(self._peers)
        self.window.addControl(self._peers_info)
        self.window.addControl(self._speed_down)
        self.window.addControl(self._speed_down_info)
        self.window.addControl(self._downloaded)
        self.window.addControl(self._downloaded_info)
        self.window.addControl(self._speed_up)
        self.window.addControl(self._speed_up_info)
        self.window.addControl(self._uploaded)
        self.window.addControl(self._uploaded_info)
        self.enabled = True

    def disable(self):
        self.window.removeControl(self._title)
        self.window.removeControl(self._peers)
        self.window.removeControl(self._peers_info)
        self.window.removeControl(self._speed_down)
        self.window.removeControl(self._speed_down_info)
        self.window.removeControl(self._downloaded)
        self.window.removeControl(self._downloaded_info)
        self.window.removeControl(self._speed_up)
        self.window.removeControl(self._speed_up_info)
        self.window.removeControl(self._uploaded)
        self.window.removeControl(self._uploaded_info)
        self.enabled = False


class EventThread(Thread):
    def __init__(self, group=None, target=None, name=None, abort=None, event_url=None):
        self.abort = abort
        self.event_url = event_url
        self.s = Session()
        super(EventThread, self).__init__(group=group, target=target, name=name)

    def run(self):
        while not self.abort.is_set():
            res = None
            try:
                r = self.s.get(self.event_url, timeout=3)
                res = r.json()["response"]
            except ReadTimeout:
                continue
            except Exception:
                traceback.print_exc()
            if res:
                print(res)
                for event in res["events"]:
                    if event["name"] == "download_stopped":
                        if "params" in event:
                            dialog = xbmcgui.Dialog()
                            dialog.notification(
                                "Acestream Engine",
                                f'{event["name"]}|{event["params"]["reason"]}',
                                xbmcgui.NOTIFICATION_INFO,
                                3000,
                            )
                        self.abort.set()


class AcePlayer(Player):
    def __init__(self):
        self.ended = False
        self.info = AcestreamInfo()
        Player.__init__(self)

    def onPlayBackError(self):
        self.ended = True

    def onPlayBackEnded(self):
        self.ended = True

    def onPlayBackStopped(self):
        self.ended = True


class AceMonitor(Monitor):
    def __init__(self, event_url, stat_url, command_url):
        self.player = AcePlayer()
        self.session = Session()
        self.event_url = event_url
        self.stat_url = stat_url
        self.command_url = command_url
        self.event_monitor = EventThread(abort=Event(), event_url=self.event_url)
        Monitor.__init__(self)

    def stat_request(self):
        r = self.session.get(self.stat_url, timeout=1)
        return r.json()["response"]

    def stop_request(self):
        params = {"method": "stop"}
        r = self.session.get(self.command_url, params=params, timeout=1)
        return r.json()

    def run(self):
        self.event_monitor.start()
        while not self.abortRequested():
            if xbmc.getCondVisibility("Window.IsActive(playerprocessinfo)"):
                if not self.player.info.enabled:
                    self.player.info.enable()
                self.player.info.update(self.stat_request())
            if self.waitForAbort(2) or self.player.ended:
                if self.player.info.enabled:
                    self.player.info.disable()
                break
        self.stop_request()
        self.event_monitor.abort.set()
        self.event_monitor.join()


if __name__ == "__main__":
    event_url = sys.argv[1]
    stat_url = sys.argv[2]
    command_url = sys.argv[3]
    AceMonitor(event_url, stat_url, command_url).run()
