import os
import requests
import random
from itertools import product
from .peewee import SqliteDatabase, Model, BlobField, TextField, IntegerField, DoesNotExist
from hashlib import sha1
from base64 import b64encode, b64decode
from binascii import a2b_hex, b2a_hex
from urllib.parse import urlencode
from bs4 import BeautifulSoup
from Cryptodome.Cipher import AES
from Cryptodome.Util.Padding import unpad
import bencodepy
import time

db = SqliteDatabase(None, pragmas={"foreign_keys": 1})


class BaseModel(Model):
    class Meta:
        database = db


class Stream(BaseModel):
    name = TextField()
    content_id = TextField(unique=True)
    checksum = TextField(unique=True)
    infohash = TextField()
    torrent = BlobField()
    last_played = IntegerField(default=0)


class AceTV:
    headers = {
        "Accept-Language": "en-US,en;q=0.5",
        "Accept-Encoding": "gzip",
        "Accept": "text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
        "User-Agent": "Mozilla/5.0 (Linux; Android 6.0.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.50 Safari/537.36 AceStream/3.1.67",
        "Connection": "close",
    }
    domains = ("torrentstream.org", "torrentstream.net", "torrentstream.info")
    server = ("s1", "s2", "s3")

    def __init__(self, cache_dir):
        DB = os.path.join(cache_dir, "acetv.db")
        db.init(DB)
        db.connect()
        db.create_tables([Stream])
        self.s = requests.Session()
        self.s.headers.update(self.headers)
        self.host = random.choice([f"{s}.{d}" for s, d in product(self.server, self.domains)])

    def __del__(self):
        db.close()
        self.s.close()

    @staticmethod
    def get_ace_api_sig(params):
        salt = a2b_hex(
            "48212b3a48314e6e7676585c7830625327283b302f415c6e527b247b5c"
            "6e2f3325315c7830622a5b72306f3e517a4e474b6b585440765c783062"
            "33444e3b67785f36364c32207b6046302c5c744b6d3e586f477e695928"
            "5c783062755d36457d5c747e303726483b39714531643f642d41375328"
        )
        data = urlencode(params).replace("&", "#")
        return sha1(data.encode("utf-8") + salt).hexdigest()

    def decrypt_acelive(self, data):
        acelive_key = a2b_hex("a50c4e33a2f48cc50ce275c9ff3a31bf")
        acelive_iv = a2b_hex("74e9cdd6391bcbd565f99503313329a3")
        aes = AES.new(acelive_key, AES.MODE_CBC, acelive_iv)
        return unpad(aes.decrypt(data[20:]), 16)

    def decode_acelive(self, data):
        bc = bencodepy.Bencode(encoding="utf-8", encoding_fallback="value")
        return bc.decode(self.decrypt_acelive(data))

    def get_stream_from_acelive(self, acelive_data):
        api_url = f"http://{self.host}/checktorrent"
        checksum = sha1(acelive_data).hexdigest()
        decoded = self.decode_acelive(acelive_data)
        params = {
            "_ap": "org.acestream.core",
            "_av": "301670312",
            "_n": "3.1.67",
            "_p": "android",
            "_r": random.randrange(2000000, 20000000),
            "_v": "3016700",
            "a": 0,
            "checksum": checksum,
            "d": 0,
            "z": 0,
        }
        params["_s"] = self.get_ace_api_sig(params)
        r = self.s.get(api_url, params=params)
        r.raise_for_status()
        soup = BeautifulSoup(r.text, "html.parser")
        if soup.id:
            stream = Stream(
                name=decoded.get("name", ""),
                content_id=soup.id.get_text(strip=True),
                checksum=checksum,
                infohash=b2a_hex(b64decode(soup.infohash.get_text(strip=True))),
                torrent=acelive_data,
            )
            stream.save()
        return stream

    def get_stream_from_url(self, url):
        r = self.s.get(url)
        r.raise_for_status()
        acelive_data = r.content
        checksum = sha1(acelive_data).hexdigest()
        try:
            stream = Stream.get(Stream.checksum == checksum)
        except DoesNotExist:
            if acelive_data.startswith(b"AceStreamTransport"):
                stream = self.get_stream_from_acelive(acelive_data)
            else:
                print(f"Not AceStreamTransport {url}")
                return None
        return stream

    def get_stream_from_cid(self, cid):
        api_url = f"http://{self.host}/gettorrent"
        try:
            stream = Stream.get(Stream.content_id == cid)
        except DoesNotExist:
            params = {
                "_ap": "org.acestream.core",
                "_av": "301670312",
                "_n": "3.1.67",
                "_p": "android",
                "_r": random.randrange(2000000, 20000000),
                "_v": "3016700",
                "pid": cid,
            }
            params["_s"] = self.get_ace_api_sig(params)
            r = self.s.get(api_url, params=params)
            r.raise_for_status()
            soup = BeautifulSoup(r.text, "html.parser")
            if soup.torrent:
                acelive_data = b64decode(soup.torrent.get_text(strip=True))
            else:
                return None
            if acelive_data.startswith(b"AceStreamTransport"):
                stream = self.get_stream_from_acelive(acelive_data)
            else:
                print(f"Not AceStreamTransport {cid}")
                return None
        return stream

    def get_last_streams(self, count):
        return Stream.select().order_by(Stream.last_played.desc()).limit(count)

    def mark_played(self, stream):
        stream.last_played = int(time.time())
        stream.save()
