//
//  DataSourceDelegate.m
//  DataSource Categorizer
//
//  Created by Andrew Pontious on 8/6/05.
//

#import "DataSourceDelegate.h"

#import "DataSourceCategory.h"
#import "DataSourceTableView.h"

@implementation DataSourceDelegate

static DataSourceDelegate *_sharedDelegate = nil;

- (id)init
{
	self = [super init];
	if (self != nil) 
	{
		categories = [[NSMutableArray alloc] init];
		
		DataSourceCategory *allCategory = [[DataSourceCategory alloc] init];
		[allCategory setName:@"All"];
		
		// Add special initial category.
		[categories addObject:allCategory];
		
		[allCategory release];
		
		_sharedDelegate = self;
	}
	return self;
}

- (void)dealloc
{
	[categories release];
	
	[super dealloc];
}

#pragma Helper Methods

- (BOOL)tableViewIsEditingRowAndShouldNotEndEditing:(DataSourceTableView *)tableView
{
	BOOL result = NO;
	
	if ([tableView editedRow] >= 0)
	{
		result = ![tableView textShouldEndEditing:[tableView currentEditor]];
	}
	return result;
}

static NSComparisonResult _comparisonFunction(DataSourceCategory *first, DataSourceCategory *second, void *context)
{
	NSComparisonResult result = NSOrderedSame;
	
	NSString *firstName = [first name];
	NSString *secondName = [second name];

	NSLog(@"compare %@ and %@", firstName, secondName);
	
	// TODO localize @"All".
	
	// @"All" always goes first.
	if ([firstName isEqualToString:@"All"]) {
		if (![secondName isEqualToString:@"All"]) {
			result = NSOrderedAscending;
		} //else { result = NSOrderedSame; }
	} else if([secondName isEqualToString:@"All"]) {
		result = NSOrderedDescending;
	// Blank names always go last.
	} else if ([firstName length] == 0) {
		if ([secondName length] > 0) {
			result = NSOrderedDescending;
		} //else { result = NSOrderedSame; }
	} else if ([secondName length] == 0) {
		result = NSOrderedAscending;
	} else {
		result = [firstName caseInsensitiveCompare:secondName];
	}
	
	return result;
}

- (void)rearrangeCategories
{
	NSIndexSet *selectedRowIndexes = [categoryTable selectedRowIndexes];
	
	NSAssert1([selectedRowIndexes count] == 1, 
		@"Currently, rearrangeCategories only supports rearranging 1 selected row, but was called with %d selected rows",
		[selectedRowIndexes count]);
		
	NSString *selectedCategoryName = [[categories objectAtIndex:[selectedRowIndexes firstIndex]] name];

	[categories sortUsingFunction:_comparisonFunction context:NULL];
	
	[categoryTable reloadData];
	
	unsigned int row = [[categories valueForKey:@"name"] indexOfObject:selectedCategoryName];
	
	[categoryTable selectRowIndexes:[NSIndexSet indexSetWithIndex:row]
		byExtendingSelection:NO];
	[categoryTable scrollRowToVisible:row];
}

#pragma Actions

- (IBAction)newCategory:(id)sender
{
	if ([self tableViewIsEditingRowAndShouldNotEndEditing:categoryTable])
		return;

	if ([categoryTable editedRow] >= 0 && [[[categoryTable currentEditor] string] length] == 0 &&
		[[[categories objectAtIndex:[categoryTable editedRow]] name] length] == 0)
	{
		NSBeep();
		return;
	}

	DataSourceCategory *category =
		[[DataSourceCategory alloc] init];

	[categories addObject:category];
	[categoryTable reloadData];
	
	[category release];

	int count = [categoryTable numberOfRows];
	
	// Have to select row first, or editing doesn't work.
	[categoryTable selectRowIndexes:[NSIndexSet indexSetWithIndex:count-1]
			   byExtendingSelection:NO];
	
	[categoryTable editColumn:0 row:count-1 withEvent:nil select:NO];
}

#pragma mark NSControl delegate

// Only called if you actually type something in, so this
// *cannot* catch when you have a blank row, and only hit return on it.
- (BOOL)control:(NSTableView *)tableView textShouldEndEditing:(NSText *)textObject
{
	BOOL result = YES;

	if (tableView == categoryTable)
	{
		NSString *string = [textObject string];
		if ([string length] > 0)
		{
			NSMutableArray *names = [NSMutableArray arrayWithArray:[categories valueForKey:@"name"]];
			[names removeObjectAtIndex:[tableView editedRow]];
			
			result = ([names indexOfObject:string] == NSNotFound);
		}
	}
	return result;
}

#pragma mark NSTableView delegate

- (BOOL)tableView:(DataSourceTableView *)tableView shouldEditTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
	BOOL result = YES;

	if (tableColumn == categoryTableColumn)
	{
		if (row == 0)
		{
			// Don't edit "All" row.
			result = NO;
		}
	}
	return result;
}

#pragma mark NSTableView data source

- (int)numberOfRowsInTableView:(DataSourceTableView *)tableView
{
	if (tableView == categoryTable)
	{
		return [categories count];
	}
	return 0;
}
- (id)tableView:(DataSourceTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
	if (tableColumn == categoryTableColumn)
	{
		return [[categories objectAtIndex:row] name];
	}
	return nil;
}
- (void)tableView:(DataSourceTableView *)tableView setObjectValue:(id)object forTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
	if (tableColumn == categoryTableColumn)
	{
		DataSourceCategory *category = [categories objectAtIndex:row];
	
		[[categoryTable window] endEditingFor:categoryTable];
		// Without this, table isn't selected, as it was before we started editing.
		[[categoryTable window] makeFirstResponder:categoryTable];
	
		if ([object length] > 0)
		{
			if (![[category name] isEqualToString:object])
			{
				[category setName:object];
				[self rearrangeCategories];
			}
		}
		else
		{
			if ([[[categories objectAtIndex:row] name] length] == 0)
			{
				[categories removeObjectAtIndex:row];
				[categoryTable reloadData];
				[categoryTable selectRowIndexes:[NSIndexSet indexSetWithIndex:row-1]
					byExtendingSelection:NO];
			}
			// If we erase the name of an existing category, assume we want
			// to undo any editing we did and retain the old name.
			else 
			{
				[categoryTable reloadData];
			}
		}
	}
}

#pragma mark DataSourceTableView delegate

- (BOOL)tableView:(DataSourceTableView *)tableView shouldHandleString:(NSString *)string forTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
	BOOL shouldHandle = YES;
	
	if (tableView == categoryTable)
	{
		shouldHandle = NO;

		[self tableView:categoryTable setObjectValue:string forTableColumn:tableColumn row:row];
	}

	return shouldHandle;
}

@end
