import java.io.*;
import java.util.*;

public class Hull{
    
    private ConvexHull hullA, hullB;
    private ConvexHull hullAo, hullBo;
    private Vector vertA, vertB, intVert;
    private BBox bboxA, bboxB;
    private int noA, noB;
    private double intVol;
    private Vertex center;
    
    public Hull(Domain DA, Domain DB){
	intVert = new Vector();
	intVol=0;
	vertA = DA.getAtomsVector();
	vertB = DB.getAtomsVector();
	bboxA = DA.getBBox();
	bboxB = DB.getBBox();
	
	Vector vectA = DA.copyAtoms();
	Vector vectB = DB.copyAtoms();
	
	try{
	    hullAo = new ConvexHull(vectA);
	    //print2File(hullA,DA.getDomainName());
	}catch(ConvexHullException ce){System.out.println(ce);}
	
	try{
	    hullBo = new ConvexHull(vectB);
	    //print2File(hullB,DB.getDomainName());
	}catch(ConvexHullException ce){System.out.println(ce);}
	
	//Calculates the intersection, default transf method is the first one
	transfHulls(2);
	calcInter();
	transfHulls(1);
	print2File(DA.getDomainName()+"_"+DB.getDomainName());
    }
    
    public Hull(Domain DA, Domain DB, int method){
	intVert = new Vector();
	intVol=0;
	vertA = DA.getAtomsVector();
	vertB = DB.getAtomsVector();
	
	Vector vectA = DA.copyAtoms();
	Vector vectB = DB.copyAtoms();
	
	try{
	    hullAo = new ConvexHull(vectA);
	    //print2File(hullA,DA.getDomainName());
	}catch(ConvexHullException ce){System.out.println(ce);}
	
	try{
	    hullBo = new ConvexHull(vectB);
	    //print2File(hullB,DB.getDomainName());
	}catch(ConvexHullException ce){System.out.println(ce);}
	
	//Calculates the intersection
	transfHulls(method);
	calcInter();
	print2File(DA.getDomainName()+"_"+DB.getDomainName());
    }
    
    public double getIntVol(){
	return intVol;
    }
    
    public int getAtomA(){
	return noA;
    }
    
    public int getAtomB(){
	return noB;
    }

    public void transfHulls(int method){
	if(method==1){
	    hullA = transfHull(hullAo,hullAo.getCenter());
	    hullB = transfHull(hullBo,hullBo.getCenter());
	}else{
	    hullA = transfHull(hullAo);
	    hullB = transfHull(hullBo);
	}
	//print2File(hullA,"TransA");
	//print2File(hullB,"TransB");
    }
    
    public void calcInter(){
	boolean in;
	
	noA = 0;
	
	for (Enumeration e = vertA.elements();e.hasMoreElements();){
	    Atom k = (Atom)e.nextElement();
	    in = true;

	    Enumeration fb = (hullB.getFaces()).elements();
	    while(in&&fb.hasMoreElements()){
		Triangle p = (Triangle)fb.nextElement();
		if(p.volumeSign(k)<0)
		    in=false;
	    }
	    
	    if(in==true){
		k.setInteraction(true);
		intVert.add(k);
		noA++;
	    }
	}
	
	noB = 0;
	
	for (Enumeration e = vertB.elements();e.hasMoreElements();){
	    Atom k = (Atom)e.nextElement();
	    in = true;

	    Enumeration fa = (hullA.getFaces()).elements();
	    while(in&&fa.hasMoreElements()){
		Triangle p = (Triangle)fa.nextElement();
		if(p.volumeSign(k)<0)
		    in=false;
	    }
	    
	    if(in==true){
		k.setInteraction(true);
		intVert.add(k);
		noB++;
	    }
	}

	try{
	    ConvexHull intHull = new ConvexHull(intVert);
	    //print2File(intHull,"Inter");
	    intVol=calcVolume(intHull);
	    center=intHull.getCenter();
	}catch(ConvexHullException ce){//System.out.println(ce);
	    intVol=0;
	    center=getCenter();
	}
    }
    
    public static double calcVolume(ConvexHull ch){
	Vertex oC = new Vertex(0.0,0.0,0.0);
	double volume = 0;
	
	//Intersection Volume
	for (Enumeration f = (ch.getFaces()).elements();f.hasMoreElements();){
	    Triangle p = (Triangle)f.nextElement();
	    volume+=p.volume6(oC)/6;
	}
	return volume;
    }
    
    //Transforms the Hull to a swelling Hull by THRES
    public static ConvexHull transfHull(ConvexHull hull, Vertex center){
	Vector vertices = hull.getVertices();
	Vector newVert = new Vector();
	ConvexHull newCH;
	for (Enumeration v = vertices.elements();v.hasMoreElements();){
	    Vertex vert = (Vertex)v.nextElement();
	    Vertex vc = Vertex.vectorDiff(vert,center);
	    vc=Vertex.vectorMult(vc,((DomainInter.THRES+vc.norm())/vc.norm()));
	    vc=Vertex.vectorAdd(vc,center);
	    newVert.addElement(new Atom(vc.x,vc.y,vc.z));
	}
	try{
	    newCH = new ConvexHull(newVert);
	}catch(ConvexHullException ce){System.out.println(ce);newCH=null;}
	return newCH;
    }

    //Transforms the Hull to a swelling Hull by THRES
    public static ConvexHull transfHull(ConvexHull hull){
	Vector faces = hull.getFaces();
	DSurface newFaces = new DSurface();
	for (Enumeration t = faces.elements();t.hasMoreElements();){
	    Triangle tr = (Triangle)t.nextElement();
	    Vector newTr = new Vector();
	    for (Enumeration v = (tr.getVertices()).elements();v.hasMoreElements();){
		Vertex vert = (Vertex)v.nextElement();
		Vertex n = Vertex.vectorMult(tr.triangleNorm(),DomainInter.THRES);
		newTr.addElement(Vertex.vectorAdd(vert,n));
	    }
	    newFaces.addFace(new Triangle(newTr));
	}
	ConvexHull newCH = new ConvexHull(newFaces);
	return newCH;
    }
    
    //Finds the center of the surface
    public Vertex getCenter(){
	Vector vertices = new Vector();
	vertices.addAll(hullA.getVertices());
	vertices.addAll(hullB.getVertices());

	int size = vertices.size();
	double[] c = {0.0,0.0,0.0};

	for (Enumeration v = vertices.elements();v.hasMoreElements();){
	    double[] x = ((Vertex)v.nextElement()).getCoords();
	    c[0]+=x[0];
	    c[1]+=x[1];
	    c[2]+=x[2];
	}
	c[0]=c[0]/size;
	c[1]=c[1]/size;
	c[2]=c[2]/size;

	return new Vertex(c[0],c[1],c[2]);
    }

    public static void print2File(ConvexHull hull, String prefix){
	try{
	    PrintWriter pw = new PrintWriter(new BufferedWriter(new FileWriter(prefix+"_hull.wrl")));
	    //hull.writeOFF(pw);
	    hull.writeVRML(pw);
	    pw.close();
	}catch(IOException e){System.out.println(e);}
    }

    public void print2File(String prefix){
	try{
	    PrintWriter pw = new PrintWriter(new BufferedWriter(new FileWriter(prefix+".wrl")));
	    //writeOFF(pw);
	    writeVRML(pw);
	    pw.close();
	}catch(IOException e){System.out.println(e);}
    }
    
    public void writeOFF(PrintWriter pw) throws IOException{
	Vector faces = new Vector();
	faces.addAll(hullA.getFaces());
	faces.addAll(hullB.getFaces());
	Surface cS = new Surface(faces);
	Vector vertices = cS.getVertices();
	Vertex center = getCenter();

	//Write header
	pw.println("OFF");
	pw.println(vertices.size() + " " + " " + faces.size() + " " +
		   (3 * faces.size()));
	
	//Write vertex list
	for (Enumeration v = vertices.elements();v.hasMoreElements();){
	    double[] c = ((Vertex)v.nextElement()).getCoords();
	    pw.println(c[0]+" "+c[1]+" "+c[2]);
	}
	
	//Write polygon list
	for (Enumeration f = faces.elements();f.hasMoreElements();){
	    Vector pV = ((Polygon)f.nextElement()).getVertices();
	    pw.print(pV.size());
	    for (Enumeration v = pV.elements();v.hasMoreElements();)
		pw.print(" " + vertices.indexOf((Vertex)v.nextElement()));
	    pw.println();
	}
    }
    
    public void writeVRML(PrintWriter pw) throws IOException{

	//Write header
	pw.println("#VRML V1.0 ascii");
	pw.println("Separator {");
	pw.println("DirectionalLight {");
	pw.println("direction 0 0 -1");
	pw.println("}");

	setViewPoints(pw);
	
	Vector faces = hullA.getFaces();
	Vector vertices = hullA.getVertices();
	
	//Write HullA
	pw.println("Separator {");
	pw.println("Material {");
	pw.println("diffuseColor 1 0 0");
	pw.println("transparency 0.5");
	pw.println("}");
	pw.println("Coordinate3 {");
	pw.println("point [");
	
	//Write vertexes
	for (Enumeration v = vertices.elements();v.hasMoreElements();){
	    double[] c = ((Vertex)v.nextElement()).getCoords();
	    pw.println(c[0]+" "+c[1]+" "+c[2]+",");
	}
	
	pw.println("]}");
	pw.println("IndexedFaceSet {");
	pw.println("coordIndex [");
	
	//Write polygons
	for (Enumeration f = faces.elements();f.hasMoreElements();){
	    Vector pV = ((Polygon)f.nextElement()).getVertices();
	    for (Enumeration v = pV.elements();v.hasMoreElements();)
		pw.print(vertices.indexOf((Vertex)v.nextElement()) + " ,");
	    pw.println("-1,");
	}
	pw.println("]}}");

	//Write atoms of domain A
	for (Enumeration v = vertA.elements();v.hasMoreElements();){
	    Atom a = (Atom)v.nextElement();
	    pw.println("Separator {");
	    pw.println("Material {");
	    pw.println("diffuseColor 1 0.8 0.4");
	    pw.println("transparency 0.3");
	    pw.println("}");
	    pw.println("Translation { translation "+a.x+" "+a.y+" "+a.z+" }");
	    pw.println("WWWAnchor {");
	    pw.println("name \"Residue: "+a.getResidue()+",  Atom: "+a.getElement()+"\"");
	    if(a.getInteraction())
		pw.println("Cube{width 1 height 1 depth 1}");
	    else
		pw.println("Cone{bottomRadius 0.25 height 0.5}");
	    pw.println("}");
	    pw.println("}");
	}

	faces = hullB.getFaces();
	vertices = hullB.getVertices();

	//Write HullB
	pw.println("Separator {");
	pw.println("Material {");
	pw.println("diffuseColor 0 0 1");
	pw.println("transparency 0.5");
	pw.println("}");
	pw.println("Coordinate3 {");
	pw.println("point [");
	
	//Write vertexes
	for (Enumeration v = vertices.elements();v.hasMoreElements();){
	    double[] c = ((Vertex)v.nextElement()).getCoords();
	    //pw.println((c[0]-center.x) + " " + (c[1]-center.y) + " " + (c[2]-center.z) + ",");
	    pw.println(c[0]+" "+c[1]+" "+c[2]+",");
	}
	
	pw.println("]}");
	pw.println("IndexedFaceSet {");
	pw.println("coordIndex [");
	
	//Write polygons
	for (Enumeration f = faces.elements();f.hasMoreElements();){
	    Vector pV = ((Polygon)f.nextElement()).getVertices();
	    for (Enumeration v = pV.elements();v.hasMoreElements();)
		pw.print(vertices.indexOf((Vertex)v.nextElement()) + " ,");
	    pw.println("-1,");
	}
	pw.println("]}}");

	//Write atoms of domain B
	for (Enumeration v = vertB.elements();v.hasMoreElements();){
	    Atom a = (Atom)v.nextElement();
	    pw.println("Separator {");
	    pw.println("Material {");
	    pw.println("diffuseColor 1 0.8 0.4");
	    pw.println("transparency 0.3");
	    pw.println("}");
	    pw.println("Translation { translation "+a.x+" "+a.y+" "+a.z+" }");
	    pw.println("WWWAnchor {");
	    pw.println("name \"Residue: "+a.getResidue()+",  Atom: "+a.getElement()+"\"");
	    if(a.getInteraction())
		pw.println("Cube{width 1 height 1 depth 1}");
	    else
		pw.println("Sphere{radius 0.25}");
	    pw.println("}");
	    pw.println("}");
	}
	
	pw.println("}");
    }

    private void printViewPoint(Vertex pos, Vertex at, String name, PrintWriter pw) {
	printViewPoint(pos,at,new Vertex(0,0,1),name,pw);
    }
    
    private void printViewPoint(Vertex pos, Vertex at, Vertex up, String name, PrintWriter pw) {
	
	Orientation q = new Orientation();
	
	pw.println("DEF "+name+" PerspectiveCamera{");
	pw.println("position "+pos.x+" "+pos.y+" "+pos.z);
	
	q.convertCameraModel(pos,at,up);
	pw.print("orientation ");
	if (new Double(q.getx()).isNaN())
	    pw.print("0 ");
	else
	    pw.print(q.getx()+" ");
	if (new Double(q.gety()).isNaN())
	    pw.print("0 ");
	else
	    pw.print(q.gety()+" ");
	if (new Double(q.getz()).isNaN())
	    pw.print("0 ");
	else
	    pw.print(q.getz()+" ");
	if (new Double(q.geta()).isNaN())
	    pw.print("0");
	else
	    pw.print(q.geta());
	pw.println("}");
    }
    
    private void setViewPoints(PrintWriter pw){
	Vertex position;
	double padx, pady, padz;

	padx = (bboxA.xmax+bboxA.xmin)/4;
	pady = (bboxA.ymax+bboxA.ymin)/4;
	padz = (bboxA.zmax+bboxA.zmin)/4;
	
	position = Vertex.vectorAdd(new Vertex(0,0,100),center);
	//Default Camera
	printViewPoint(position, center, "default", pw);
	
	position = Vertex.vectorAdd(new Vertex(0,0,1),center);
	//Camera from the intersection
	printViewPoint(position, center, "inter", pw);

	position = hullA.getCenter();
	//Camera from the center of HullA
	printViewPoint(position, center, "centerA", pw);

	position = Vertex.vectorAdd(new Vertex(bboxA.xmin,bboxA.ymax,bboxA.zmax), new Vertex(-padx,pady,padz));
	//Camera from the NWF of HullA
	printViewPoint(position, center, "NWFA", pw);

	position = Vertex.vectorAdd(new Vertex(bboxA.xmax,bboxA.ymax,bboxA.zmax), new Vertex(padx,pady,padz));
	//Camera from the NEF of HullA
	printViewPoint(position, center, "NEFA", pw);

	position = Vertex.vectorAdd(new Vertex(bboxA.xmin,bboxA.ymin,bboxA.zmax), new Vertex(-padx,-pady,padz));
	//Camera from the SWF of HullA
	printViewPoint(position, center, "SWFA", pw);

	position = Vertex.vectorAdd(new Vertex(bboxA.xmax,bboxA.ymin,bboxA.zmax), new Vertex(padx,-pady,padz));
	//Camera from the SEF of HullA
	printViewPoint(position, center, "SEFA", pw);

	position = Vertex.vectorAdd(new Vertex(bboxA.xmin,bboxA.ymax,bboxA.zmin), new Vertex(-padx,pady,-padz));
	//Camera from the NWB of HullA
	printViewPoint(position, center, "NWBA", pw);

	position = Vertex.vectorAdd(new Vertex(bboxA.xmax,bboxA.ymax,bboxA.zmin), new Vertex(padx,pady,-padz));
	//Camera from the NEB of HullA
	printViewPoint(position, center, "NEBA", pw);

	position = Vertex.vectorAdd(new Vertex(bboxA.xmin,bboxA.ymin,bboxA.zmin), new Vertex(-padx,-pady,-padz));
	//Camera from the SWB of HullA
	printViewPoint(position, center, "SWBA", pw);

	position = Vertex.vectorAdd(new Vertex(bboxA.xmax,bboxA.ymin,bboxA.zmin), new Vertex(padx,-pady,-padz));
	//Camera from the SEB of HullA
	printViewPoint(position, center, "SEBA", pw);

	padx = (bboxB.xmax+bboxB.xmin)/4;
	pady = (bboxB.ymax+bboxB.ymin)/4;
	padz = (bboxB.zmax+bboxB.zmin)/4;

	position = hullB.getCenter();
	//Camera from the center of HullB
	printViewPoint(position, center, "centerB", pw);

	position = Vertex.vectorAdd(new Vertex(bboxB.xmin,bboxB.ymax,bboxB.zmax), new Vertex(-padx,pady,padz));
	//Camera from the NWF of HullB
	printViewPoint(position, center, "NWFB", pw);

	position = Vertex.vectorAdd(new Vertex(bboxB.xmax,bboxB.ymax,bboxB.zmax), new Vertex(padx,pady,padz));
	//Camera from the NEF of HullB
	printViewPoint(position, center, "NEFB", pw);

	position = Vertex.vectorAdd(new Vertex(bboxB.xmin,bboxB.ymin,bboxB.zmax), new Vertex(-padx,-pady,padz));
	//Camera from the SWF of HullB
	printViewPoint(position, center, "SWFB", pw);

	position = Vertex.vectorAdd(new Vertex(bboxB.xmax,bboxB.ymin,bboxB.zmax), new Vertex(padx,-pady,padz));
	//Camera from the SEF of HullB
	printViewPoint(position, center, "SEFB", pw);

	position = Vertex.vectorAdd(new Vertex(bboxB.xmin,bboxB.ymax,bboxB.zmin), new Vertex(-padx,pady,-padz));
	//Camera from the NWB of HullB
	printViewPoint(position, center, "NWBB", pw);

	position = Vertex.vectorAdd(new Vertex(bboxB.xmax,bboxB.ymax,bboxB.zmin), new Vertex(padx,pady,-padz));
	//Camera from the NEB of HullB
	printViewPoint(position, center, "NEBB", pw);

	position = Vertex.vectorAdd(new Vertex(bboxB.xmin,bboxB.ymin,bboxB.zmin), new Vertex(-padx,-pady,-padz));
	//Camera from the SWB of HullB
	printViewPoint(position, center, "SWBB", pw);

	position = Vertex.vectorAdd(new Vertex(bboxB.xmax,bboxB.ymin,bboxB.zmin), new Vertex(padx,-pady,-padz));
	//Camera from the SEB of HullB
	printViewPoint(position, center, "SEBB", pw);
    }
}
