# CD Player Plugin for SqueezeCenter

# Copyright (C) 2008 Bryan Alton and others
# All rights reserved.

# This program is free software; you can redistribute it and/or
# modify it under the same terms as Perl itself.

package Plugins::CDplayer::Plugin;

use strict;

use base qw(Slim::Plugin::Base);

use Slim::Utils::Prefs;
use Slim::Utils::Log;
use Slim::Utils::Misc;
use Slim::Utils::Strings qw(string);
use Slim::Utils::OSDetect;
use File::Spec::Functions qw(:ALL);
use Plugins::CDplayer::Settings;
use Plugins::CDplayer::CDhandler;
use Data::Dumper;

# create log categogy before loading other modules
my $log = Slim::Utils::Log->addLogCategory( {
	category     => 'plugin.cdplayer',
	defaultLevel => 'ERROR',
	description  => getDisplayName(),
} );


my $prefsServer = preferences('server');
my $prefs       = preferences('plugin.cdplayer');

use Plugins::CDplayer::CDPLAY;

my $cdInfo;  # Store the potiner to CDInfo object whichs describes the state of CD drive

# If filename below is changed - change also in CDHandler.pm file.
use constant CDTOCOPMLFILE => 'tmp-CDplayer-CDTOC.opml';


################################
### Plugin Interface ###########
################################


# Get toc
# cdda2wav -device $device -verbose-level=toc -N -g -J
#
# Rip a track
# cdda2wav  -device $device -no-infofile -track $tracknum 
#

sub initPlugin() 
{
	my $class = shift;
	my $device;
	$log->info("Initialising CDPlayer" . $class->_pluginDataFor('version'));


	Plugins::CDplayer::Settings->new($class);
	Plugins::CDplayer::Settings->init();


	$cdInfo = Plugins::CDplayer::CDhandler->new( );
	$cdInfo->init();

	if (!$class->_pluginDataFor('icon')) {

		Slim::Web::Pages->addPageLinks("icons", { $class->getDisplayName => 'html/images/cdplayer_svg.png' });
	}

	$class->SUPER::initPlugin();

	Slim::Buttons::Common::addMode('PLUGIN.CDplayer', getFunctions(), \&setMode);

#        |requires Client
#        |  |is a Query
#        |  |  |has Tags
#        |  |  |  |Function to call
#        C  Q  T  F
	Slim::Control::Request::addDispatch(['cdplayer', 'items', '_index', '_quantity'],
        [0, 1, 1, \&cliQuery]);
	Slim::Control::Request::addDispatch(['cdplayer', 'playlist', '_method' ],
	[1, 1, 1, \&cliQuery]);

	my @item = ({
			text           => Slim::Utils::Strings::string(getDisplayName()),
			weight         => 20,
			id             => 'cdplayer',
			node           => 'extras',
			displayWhenOff => 0,
			window         => { 
						'icon-id'  => $class->_pluginDataFor('icon'),
						titleStyle => 'album'
					 },
			actions => {
				go =>      {
					'cmd' => ['cdplayer', 'items'],
					'params' => {
						'menu' => 'cdplayer',
					},
				},
			},
		});

	Slim::Control::Jive::registerPluginMenu(\@item);

# 7.3 onwards - use the protocol Handler - canDoAction
#	Slim::Control::Request::subscribe( \&pauseCallback, [['pause']] );

}

# When CD TOC is scanned a number of opml files are created. 
# The file name will be resued with each CD.
# Record the name created so that on LMS shutdown, opml files can be deleted.

my %opmlfiles;
sub noteOPMLfile {
	$opmlfiles{@_[0]} = 1;
	$log->debug(" Adding file ".@_[0]);
}

sub shutdownPlugin 
{
	my $class = shift;

	# unsubscribe
#	Slim::Control::Request::unsubscribe(\&pauseCallback);

	$log->info("Plugin shutdown - kill any cdda2wav processes left behind");
	$cdInfo->killOrphans();
	
#  remove any OPML files that were created
#	$log->error(" Dump of has files ". Dumper(\%opmlfiles));
	foreach my $opmlfile (keys %opmlfiles) {
		$log->info ("Deleting file $opmlfile"); 
		unlink($opmlfile) or $log->error(" Could not delete/unlink $opmlfile: $!");
	}
	
	return;
}

sub getFunctions {
	return {};
}


sub getDisplayName() 
{ 
	return('PLUGIN_CDPLAYER')
}

sub cdInfo()
{
	my $class = shift;

	return $cdInfo ;
}


sub setMode {
	my $class =  shift;
	my $client = shift;
	my $method = shift;

	if ($method eq 'pop') {
		Slim::Buttons::Common::popMode($client);
		return;
	}

	if( $cdInfo->isCDplaying() ) {
		my $url =  Plugins::CDplayer::CDhandler::saveOPMLfile($cdInfo->renderAsOPML(), CDTOCOPMLFILE);

		my %params = (
			modeName => 'LoadCDContents',
			url      => $url,
			title    => 'CDplayer pushmode',
		);

		Slim::Buttons::Common::pushMode($client, 'xmlbrowser', \%params);

#	Not exactly sure abouth what the follwing does but it is in OPMLbased.pm
#	we'll handle the push in a callback
		$client->modeParam('handledTransition',1)
	}
	else {
		Slim::Buttons::Common::pushMode($client, 'loadcd');
	}
}

sub webPages {
	my $class = shift;

	my $title = getDisplayName();
	my $url   = 'plugins/CDplayer/index.html';

# Add CDplayer menu item under Extras
	Slim::Web::Pages->addPageLinks('plugins', { $title => $url });
	
# assumes at least SC 7.0
	if ( substr($::VERSION,0,3) lt 7.4 ) {
		Slim::Web::HTTP::addPageFunction($url, \&indexHandler);
	} else {
	    # $::noweb to detect TinySC or user with no web interface
	    if (!$::noweb) {
		Slim::Web::Pages->addPageFunction($url, \&indexHandler);
	    }
	}

}

sub indexHandler
{
	my ( $client, $stash, $callback, $httpClient, $response ) = @_;
	$log->info("CDplayer - Indexhandler called");
	my ($success, $statuscode, $errmsg) = Plugins::CDplayer::CDhandler::cdromstatus($prefs->get('device'));
	if ( $success != 1) {

		$log->info("Error after CDROM status sucess: $success  status code: $statuscode  ");
		$stash->{'errormsg'} =  string($errmsg) . "($statuscode)";
		my $output = Slim::Web::HTTP::filltemplatefile('plugins/CDplayer/index.html', $stash);
		&$callback($client, $stash, $output,  $httpClient, $response);
	}
	elsif ($cdInfo->isCDinUse() ) {
		$log->info("CD drive is currently loading a TOC for another client ");
		$stash->{'errormsg'} = sprintf(string('PLUGIN_CDPLAYER_WEB_ERROR'), -1 , 
					string('PLUGIN_CDPLAYER_CD_BUSY') . ' '. string('PLUGIN_CDPLAYER_TRY_AGAIN'));
 
		my $output = Slim::Web::HTTP::filltemplatefile('plugins/CDplayer/index.html', $stash);
		&$callback($client, $stash, $output,  $httpClient, $response);
	} 
	else {
		if( $cdInfo->isCDplaying() ) {
			ReadCDTOCSuccessWebCallback($client,\@_);
		} else { 
			$cdInfo->LoadCDandIdentify($client, \&ReadCDTOCSuccessWebCallback, \&ReadCDTOCFailedWebCallback, \@_);
		}
	}
	return 0;
}


sub ReadCDTOCSuccessWebCallback
{
	my $client = shift;
	my ( $clientparam, $stash, $callback, $httpClient, $response ) = @_;

	my $title = getDisplayName();
	$log->debug('Read MusicBrainz record OK - web callback');

	# Get OPML list of feeds from cache
	my $url =  Plugins::CDplayer::CDhandler::saveOPMLfile($cdInfo->renderAsOPML(), CDTOCOPMLFILE);
	
	Slim::Web::XMLBrowser->handleWebIndex( {
		client => $client,
		feed   => $url,
		title  => $title,
		args   => ($clientparam, $stash, $callback, $httpClient, $response)
	} );
}

sub ReadCDTOCFailedWebCallback
{
	my $client = shift;
	my $params = shift;

	my ( $clientparam, $stash, $callback, $httpClient, $response ) = @$params;

	$log->debug('Read MusicBrainz record failed - web callback');

	$stash->{'errormsg'} = sprintf(string('PLUGIN_CDPLAYER_WEB_ERROR'), $cdInfo->{loaderror} , $cdInfo->getErrorText() );
	my $output = Slim::Web::HTTP::filltemplatefile('plugins/CDplayer/index.html', $stash);
	&$callback($clientparam, $stash, $output,  $httpClient, $response);
}

sub cliQuery {
	my $request = shift;
	my $client = $request->client;

	$log->info("CDplayer - cliQuery called");

	$request->setStatusProcessing();	

	if (defined( $request->getParam('item_id')) || $cdInfo->isCDplaying()  ) {
		ReadCDTOCSuccessCLICallback($client, $request );
	} else {

		my ($success, $statuscode, $errmsg) = Plugins::CDplayer::CDhandler::cdromstatus($prefs->get('device'));
		if ( $success != 1) {
			$request->addResult("networkerror", Slim::Utils::Strings::string($errmsg) . "($statuscode)" );
			$request->addResult('count', 0);
			$request->setStatusDone();
			return;
		}

		if ($cdInfo->isCDinUse() ) {
			$log->info("CD drive is currently loading a TOC for another client ");
			$request->addResult("networkerror", Slim::Utils::Strings::string('PLUGIN_CDPLAYER_CD_BUSY') . "  ". 
							Slim::Utils::Strings::string('PLUGIN_CDPLAYER_TRY_AGAIN'));
			$request->addResult('count', 0);
			$request->setStatusDone();
			return;
		}

		$cdInfo->LoadCDandIdentify($client, \&ReadCDTOCSuccessCLICallback, \&ReadCDTOCFailedCLICallback, $request);
	};
	# show feedback if this action came from jive cometd session
	if ($request->source && $request->source =~ /\/slim\/request/) {
		if (!defined( $request->getParam('item_id')) ) {
			$client->showBriefly({
				'jive' => { 
				'text'    => [ '',Slim::Utils::Strings::string('PLUGIN_CDPLAYER_LOADING_CD_WAIT'),
					   ],
				}
			});
		}
	}


}	

sub ReadCDTOCSuccessCLICallback
{
	my $client = shift;
	my $request = shift;
	# Get OPML list of Album

	my $url =  Plugins::CDplayer::CDhandler::saveOPMLfile($cdInfo->renderAsOPML(), CDTOCOPMLFILE);

	$log->info("CDplayer - executing XMLbrowser cliQuery with MB results");

	Slim::Control::XMLBrowser::cliQuery('cdplayer', $url, $request);
}

sub ReadCDTOCFailedCLICallback
{
	my $client = shift;
	my $request = shift;


	$request->addResult("networkerror", string('PLUGIN_CDPLAYER_CLI_ERROR') . $cdInfo->getErrorText());
	$request->addResult('count', 0);

	$request->setStatusDone();
}


#sub pauseCallback {
#	my $request = shift;
#	my $client  = $request->client;
#
#	my $stream  = Slim::Player::Playlist::song($client)->path;
#	my $playmode= Slim::Player::Source::playmode($client);
#	my $mode    = Slim::Buttons::Common::mode($client);
#
#	$log->debug("cli Pause - playmode=$playmode  stream=$stream ");
#
#	if ($stream =~ /^cdplay:/ ) {
##	if ($playmode eq 'pause' && $stream =~ /^cdplay:/ ) {
#		if ($prefs->get('pausestop')) {
#			$log->debug("Issuing stop");
#			$client->execute([ 'stop' ]);
#		}
#	}
#
#}


1;

# Local Variables:
# tab-width:4
# indent-tabs-mode:t
# End:
