# CD Player Plugin for SqueezeCenter

# Copyright (C) 2008 Bryan Alton and others
# All rights reserved.

# This program is free software; you can redistribute it and/or
# modify it under the same terms as Perl itself.

package Plugins::CDplayer::CDPLAY;

use strict;

use base qw(Slim::Player::Pipeline);
#use utf8;

use Slim::Utils::Strings qw(string);
use Slim::Utils::Misc;
use Slim::Utils::Log;
use Slim::Utils::Prefs;
use URI::Escape;
use HTML::Entities;
use XML::Simple;
use Tie::Cache::LRU;
# use Digest::SHA::PurePerl qw(hmac_sha256_base64);

use File::Spec::Functions qw(catdir);
use Data::Dumper;

my $log   = logger('plugin.cdplayer');
my $prefs = preferences('plugin.cdplayer');
my $osdetected = Slim::Utils::OSDetect::OS();

require Digest::SHA::PurePerl;

use constant CDPLAYING => 1;
use constant CDSTOPPED => 0;


Slim::Player::ProtocolHandlers->registerHandler('cdplay', __PACKAGE__);

#  Hash holds any URL for album art  - keyed on searchstring. 
tie my %urlimg, 'Tie::Cache::LRU', 32;


sub canDoAction {
	my ( $class, $client, $url, $action ) = @_;
	$log->info("Action=$action");
	if (($action eq 'pause') && $prefs->get('pausestop') ) {
		$log->info("Stopping track because pref is set to stop");
		return 0;
	}
	
	return 1;
}

# use a small buffer threshold to make CD start playing quickly 

sub bufferThreshold { 70 }

sub isRemote { 1 }
#
#  CDROM result codes.
#

use constant 	CDS_AUDIO           => 100 ;

sub new {
	my $class = shift;
	my $args  = shift;

	my $fullurl      = $args->{'url'};
	my $client       = $args->{'client'};
	my $transcoder   = $args->{'transcoder'};
	my $song         = $args->{'song'};
	my $command;

	my $seekdata;
	if ($song->can("seekdata")) {
		$seekdata     = $song->seekdata();
	} else {
		$seekdata     = $args->{'song'}->{'seekdata'};
	}

	my $offsetSectors;
	
	if (defined($seekdata)) {
		$log->debug("Seekdata \n". Dumper($seekdata));
	}

	if (defined ($seekdata->{timeOffset})) {
		my $newtime = int($seekdata->{timeOffset});
		$offsetSectors = int($seekdata->{timeOffset} * 75) ; # 75 sectors in a second.

		if ($client->playingSong->can('startOffset')) {
			$client->playingSong->startOffset( $newtime);
		} else {
			$client->playingSong->{startOffset} = $newtime;
		};

		$client->master->remoteStreamStartTime( Time::HiRes::time() - $newtime );
	}


	my $cdInfo = Plugins::CDplayer::Plugin::cdInfo();
	my $cddevice; 
	if ($osdetected eq 'win') { # Use drop downlist built at startup
		$cddevice = $prefs->get('cddevice');
	} elsif ($osdetected eq 'unix') { # use text box
		$cddevice = $prefs->get('device');
	} else { # use drop down box filled with predefined OSX devcies name for CD/DVD drive.
		$cddevice = $prefs->get('cddevice');
	}

	$log->debug(" Host OS = $osdetected  CD device name=\'$cddevice\'");
	my $client  = $args->{'client'};
#	my $fullurl = $args->{'url'} ;
	my ($baseurl,$params) = split (/\?/,$fullurl); 
	$baseurl =~ m|^cdplay://(\d+)|;	
	my $tracknum = $1;

	my $restoredurl;

	$log->debug("params length =". length ($params) ."\n" . Data::Dumper::Dumper($params));

	my %ops = map {
			my ( $k, $v ) = split( '=' );
			$k  => Encode::decode_utf8(uri_unescape( $v ))
		} split( '&', $params );


#  Some added Unix only actions - now that eject had been added.

	if ($osdetected eq 'unix') {
		# Check if command is to eject CD - only execute if CD is stopped otherwise cdda2wav prcoess can linger.
		if (exists($ops{'Command'}) && ($ops{'Command'} eq 'eject')) {
			my $ejectpath   = Slim::Utils::OSDetect::getOS->decodeExternalHelperPath(Slim::Utils::Misc::findbin('eject'));
			$log->info(" Eject CD -  Eject path \"$ejectpath\"");
			Proc::Background->new($ejectpath, $cddevice) if ( ! $cdInfo->CDplaying ());
			return;	
		}

# 		Add in a new check to make a CD is in drive and is audio, in case CD has been ejected but playlist remains
# 		- otherwise a cdda2wav process can spin on "load drive". 
		my ($retval, $drivestatus, $errormessage) = Plugins::CDplayer::CDhandler::cdromstatus($prefs->get('device'));
		return unless ($drivestatus == CDS_AUDIO);
	};

	Slim::Music::Info::setContentType($fullurl, 'cdplay');

	$log->debug("Full URL= $fullurl Format=" . $transcoder->{'streamformat'});

	my $maxRate = 0;
	my $quality = 1;

	if (defined($client)) {
		$maxRate = Slim::Utils::Prefs::maxRate($client);
		$quality = preferences('server')->client($client)->get('lameQuality');
	}

	$restoredurl = $baseurl;
	my $cdspeed = 4;

	if (defined($offsetSectors)) {
		$transcoder->{'command'} =~ s/\$CDOFFSET\$/-offset $offsetSectors/;
	} else {
		$transcoder->{'command'} =~ s/\$CDOFFSET\$//;
	}

	$transcoder->{'command'} =~ s/\$CDTRACK\$/$tracknum/;
	$transcoder->{'command'} =~ s/\$CDDEVICE\$/$cddevice/;
	$transcoder->{'command'} =~ s/\$CDSPEED\$/$cdspeed/;

	$command = Slim::Player::TranscodingHelper::tokenizeConvertCommand2( $transcoder, $fullurl, $fullurl, 1, $quality );

	$log->debug("about to execute:$command");

	$cdInfo->{PlayingAlbumTitle}   =  $ops{AlbumTitle}    || Slim::Utils::Strings::string('PLUGIN_CDPLAYER_UNKNOWN_ALBUM'); 
	$cdInfo->{PlayingAlbumArtist}  =  $ops{AlbumArtist}   || Slim::Utils::Strings::string('PLUGIN_CDPLAYER_UNKNOWN_ARTIST') ;
	$cdInfo->{PlayingTrackTitle}   =  $ops{TrackTitle}    || Slim::Utils::Strings::string('PLUGIN_CDPLAYER_UNKNOWN_TRACK') ;
	$cdInfo->{PlayingTrackArtist}  =  $ops{TrackArtist} ;
	$cdInfo->{PlayingLengths}      =  $ops{Lengths} ;
	$cdInfo->{PlayingOffsets}      =  $ops{Offsets} ;
	$cdInfo->{PlayingMBDiscid}     =  $ops{MBDiscid};



	$log->info("SetCurrentTitle for $fullurl ");
	Slim::Music::Info::setCurrentTitle( $fullurl, $cdInfo->{PlayingTrackTitle} . ' (' . $cdInfo->{durations}[$tracknum] .')' );
	Slim::Music::Info::setDuration( $fullurl, (int($cdInfo->{PlayingLengths})/75) );
	$song->duration((int($cdInfo->{PlayingLengths})/75) );

	$cdInfo->CDplaying ( CDPLAYING );

	my $self = $class->SUPER::new(undef, $command,'local');

	${*$self}{'contentType'} = $transcoder->{'streamformat'};

	return $self;
}

sub contentType 
{
	my $self = shift;
	return ${*$self}{'contentType'};
}


sub canHandleTranscode {
	my ($self, $song) = @_;
	
	return 1;
}

sub getStreamBitrate {
	my ($self, $maxRate) = @_;
	
	return Slim::Player::Song::guessBitrateFromFormat(${*$self}{'contentType'}, $maxRate);
}


sub isAudioURL { 1 }

sub OnCommandCDInfoCallback()
{
	my $client = shift;

	my $foundDiscId = shift;
	my $callback = $client->pluginData( 'onCommandCallback' );

	$log->debug("OnCommandCallback callback=$callback");
	return $callback->();
}

sub close 
{
	my $class = shift;
	$log->debug("closing cdda2wav stream to SC ");
	my $cdInfo = Plugins::CDplayer::Plugin::cdInfo();

	my $self = $class->SUPER::close();
	$cdInfo->CDplaying ( CDSTOPPED );

	$cdInfo->killOrphans();

}

# Metadata for a URL, used by CLI/JSON clients
sub getMetadataFor {
	my ( $class, $client, $url, $forceCurrent ) = @_;

	my $cdInfo = Plugins::CDplayer::Plugin::cdInfo();
	my $icon   = Plugins::CDplayer::Plugin->_pluginDataFor('icon'); 

	$icon = $cdInfo->{coverarturl} if (defined($cdInfo->{coverarturl})) ;

	my ($baseurl,$params) = split (/\?/,$url); 

	my %ops = map {
			my ( $k, $v ) = split( '=' );
			$k  => Encode::decode_utf8(uri_unescape( $v ))
		} split( '&', $params );


	return {
		artist   =>  $ops{TrackArtist}  || $ops{AlbumArtist} || Slim::Utils::Strings::string('PLUGIN_CDPLAYER_UNKNOWN_ARTIST') ,
		album    =>  $ops{AlbumTitle}                        || Slim::Utils::Strings::string('PLUGIN_CDPLAYER_UNKNOWN_ALBUM')  ,
		title    =>  $ops{TrackTitle}                        || Slim::Utils::Strings::string('PLUGIN_CDPLAYER_UNKNOWN_TRACK')  ,
		duration =>  int($ops{Lengths})/75,
		type     =>  'CD',
		icon     =>  $icon ,
		cover    =>  $icon,
	};
}

# XXX - I think that we scan the track twice, once from the playlist and then again when playing
sub scanUrl {
	my ( $class, $url, $args ) = @_;
	
	Slim::Utils::Scanner::Remote->scanURL($url, $args);

}

sub canSeek {
	my ( $class, $client, $song ) = @_;
	
# Can only seek if duration is known
	my $seconds = $song->duration();

	if ($seconds) {
	  return 1;
	}

	$log->info("Cannot seek duration ($seconds) may be undefined or 0");

	return 0;
}

sub canSeekError {
	my ( $class, $client, $song ) = @_;
	
	my $url = $song->currentTrack()->url;
	
	if ($log->is_debug) {
		my $ct = Slim::Music::Info::contentType($url);
		$log->debug( " CanSeekError content type $ct url=$url " );
	}

	if ( !$song->duration() ) {
		return 'SEEK_ERROR_CDPLAY_UNKNOWN_DURATION';
	}
	
	return 'SEEK_ERROR_CDPLAY';
}

sub getSeekData {

	my ( $class, $client, $song, $newtime ) = @_;
	
	# Do it all later at open time
	return {timeOffset => $newtime};
}

1;

# Local Variables:
# tab-width:4
# indent-tabs-mode:t
# End:
