// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This example program demonstrates the circular buffer API.
//



/**
 @file 
*/
#include "circularbuffer.h"

// Literals.
_LIT(KPressAKey, "\n\tPress any key to continue....");
_LIT(KConstruct,"\n\n ****Construct****");
_LIT(KAdd,"\n ****Add objects****\n");
_LIT(KNumberOfObjects,"\nNumber of objects in the circular buffer: %d\n");
_LIT(KAddFailed,"\nElement cannot be added because the circular buffer is full\n");
_LIT(KRemove,"\n ****Remove objects****\n"); 


/**
Constructor.
*/
TMyClass::TMyClass(){};
/**
Constructor. This overload TMyClass constructor using a descriptor
*/
TMyClass::TMyClass(const TDesC& aDes)
	{ 
	iBuf = aDes;
	}
/**
Get TMyClass buffers. 	
@return A buffer contains the objects.  
*/
const TDesC& TMyClass::GetBuf()
	{
	return iBuf;
	}
/**
Set TMyClass buffer using a descriptor.
@param aDes contains the new value for buffer.
*/
void TMyClass::SetBuf(const TDesC& aDes)
	{
	iBuf = aDes;
	}
	
/**
Constructor.
*/
CCircularBufferExample::CCircularBufferExample()
	{
	}

/**
Constructor.
*/
void CCircularBufferExample::ConstructL()
	{
	_LIT(KTitle, "Circular Buffer Example" );
	iConsole = Console::NewL(KTitle, TSize(KConsFullScreen, KConsFullScreen));
	
	_LIT(KWelcome, "\n   Welcome to the circular buffer example application");
	iConsole->Printf(KWelcome);
	
	_LIT(KPressAKeyMsg, "\n\n Press any key to step through the example\n");
	iConsole->Printf(KPressAKeyMsg );
	iConsole->Getch();
	}

/**
Destructor
*/
CCircularBufferExample::~CCircularBufferExample()
	{
	delete iConsole;
	}

/**
Allocates and constructs a CCircularBufferExample object.
Initialises all member data to their default values.
@return A CCircularBufferExample object.
*/
CCircularBufferExample* CCircularBufferExample::NewL()
	{
	CCircularBufferExample* self=new(ELeave)CCircularBufferExample();
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}
/**
Constructs a circular buffer containing integers, and adds and removes integers.
*/
void CCircularBufferExample::CircularBufferOfIntsL()
	{
	_LIT(KCircularBufferForInt,"\n *****Circular buffer for integers*****");
	iConsole->Printf(KCircularBufferForInt);
	
	// Creates a circular buffer containing integers.
	CCirBuffer* circularBuffer = new(ELeave)CCirBuffer;
	// Push the circular buffer onto the cleanup stack.
	CleanupStack::PushL(circularBuffer);
	const TInt KMaxElements = 4;
	// Sets the maximum capacity of this circular buffer.
	circularBuffer->SetLengthL(KMaxElements);  // Maximum capacity is KMaxElements.
  		
	iConsole->Printf(KConstruct);
	_LIT(KConstructCircularBuffer, "\n Construction of circular buffer containing integers is successful");
	iConsole->Printf(KConstructCircularBuffer);		

	
	iConsole->Printf(KAdd);
	TUint element[6]= {1, 2, 3, 4, 5, 6};
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());
		
	_LIT(KAddedElements,"Added Element: %d\n");
	TInt result=circularBuffer->Put(element[0]); // Add integers to the circular buffer.
	User::LeaveIfError(result);
	iConsole->Printf(KAddedElements,element[0]);
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());		
	
	// Add multiple integers to the circular buffer.
	TUint numberOfObjects= 3;
	for(TUint index=1;index<=numberOfObjects; index++)
		{
		result= circularBuffer->Put(element[index]);
		User::LeaveIfError(result);
		// Print the element added to the circular buffer.
		iConsole->Printf(KAddedElements,element[index]);
		iConsole->Printf(KNumberOfObjects, circularBuffer->Count());			
		}
	
	_LIT(KAddIntegersToCircularBuffer,"\nAdding integers to circular buffer is successful");
	iConsole->Printf(KAddIntegersToCircularBuffer);
	
	iConsole->Printf(KPressAKey);
	iConsole->Getch();
	
	_LIT(KTryingAddToCircularBuffer,"\nTrying to add when buffer is full.");
	iConsole->Printf(KTryingAddToCircularBuffer);
	result=circularBuffer->Put(element[4]);
	ASSERT(result == KErrGeneral);
	iConsole->Printf(KAddFailed);
	
	// Remove integers from circular buffer.
	iConsole->Printf(KRemove);
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());

	result = circularBuffer->Get();
	_LIT(KElementRemoved,"Removed Element: %d\n");
	User::LeaveIfError(result);
	ASSERT( (TUint) result == element[0]);
	// Print the element removed from the circular buffer.
	iConsole->Printf(KElementRemoved, result);
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());
	

	result = circularBuffer->Get();
	User::LeaveIfError(result);
	ASSERT((TUint)result == element[1]);
	// Print the element removed from the circular buffer.
	iConsole->Printf(KElementRemoved, result);
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());
	
	//Add remaining elements from array to buffer.
	for(TUint index=4;index<=5; index++)
		{
		result= circularBuffer->Put(element[index]);
		User::LeaveIfError(result);
		iConsole->Printf(KAddedElements,element[index]);
		iConsole->Printf(KNumberOfObjects, circularBuffer->Count());
		}
	iConsole->Printf(KPressAKey);
	iConsole->Getch();
	_LIT(KNewLine,"\n");
	iConsole->Printf(KNewLine);
	// Remove multiple integers from the circular buffer.
	for(TUint index=2;index<=5; index++)
		{
		result= circularBuffer->Get();// Removed integer is element[index].
		User::LeaveIfError(result);
		ASSERT((TUint)result == element[index]);
		// Print the elements removed from the circular buffer.
		iConsole->Printf(KElementRemoved,result);
		iConsole->Printf(KNumberOfObjects, circularBuffer->Count());
		}

	_LIT(KRemoveIntegersFromCircularBuffer,"\nRemoving integers from the circular buffer is successful");
	iConsole->Printf(KRemoveIntegersFromCircularBuffer);
	iConsole->Printf(KPressAKey);
	iConsole->Getch();

	//Pop the circular buffer off the cleanup stack and destroy it.
	CleanupStack::PopAndDestroy(circularBuffer);
	}

/**
Constructs a circular buffer of objects of my own created class, add and remove object from buffer.
*/

void CCircularBufferExample::CircularBufferOfMyObjectsL()
	{
	_LIT(KCircularBufferForMyObject,"\n *****Circular buffer of objects of user defined class TMyClass*****");
	iConsole->Printf(KCircularBufferForMyObject);
		
	// Create a circular buffer containing instances of TMyClass.
	CCirBuf<TMyClass>* circularBuffer=new(ELeave) CCirBuf<TMyClass>;
	// Push the circular buffer onto the cleanup stack.
	CleanupStack::PushL(circularBuffer);
	// Set the maximum capacity of this circular buffer.
	const TInt KMaxElements = 4;
	circularBuffer->SetLengthL(KMaxElements); // Maximum capacity is KMaxElements.
	iConsole->Printf(KConstruct);
	
	_LIT(KConstructCircularBufferForMyObject, "\n Construction of circular buffer of user defined class is successful\n");
	iConsole->Printf(KConstructCircularBufferForMyObject);	
	
	iConsole->Printf(KAdd);
	// Creates an array of object of TMyClass.	
	TMyClass myObjectsToAdd[6];

	_LIT(KBuffer1,"first");
	myObjectsToAdd[0].SetBuf(KBuffer1);
	_LIT(KBuffer2,"second");
	myObjectsToAdd[1].SetBuf(KBuffer2);
	_LIT(KBuffer3,"third");
	myObjectsToAdd[2].SetBuf(KBuffer3);
	_LIT(KBuffer4,"fourth");
	myObjectsToAdd[3].SetBuf(KBuffer4);
	_LIT(KBuffer5,"fifth");
	myObjectsToAdd[4].SetBuf(KBuffer5);
	_LIT(KBuffer6,"sixth");
	myObjectsToAdd[5].SetBuf(KBuffer6);
	
	_LIT(KAddedMyObjectElements,"\nTMyClass object added is: ");
	_LIT(KPrintMsgFormat," \"%S\" ");
	
	TInt result= circularBuffer->Add(&myObjectsToAdd[0]); // Add a TMyClass to the circular buffer.
	User::LeaveIfError(result);
	ASSERT(result==1);
	iConsole->Printf(KAddedMyObjectElements);
	iConsole->Printf(KPrintMsgFormat,&(myObjectsToAdd[0].GetBuf()));
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());
	
	
	_LIT(KAddedMultipleMyObjectElements,"\nTMyClass objects added are : ");
	// Add multiple TMyClasss to the circular buffer
	result= circularBuffer->Add(&myObjectsToAdd[1], 3); // Add three objects to circular buffer.
	User::LeaveIfError(result);
	ASSERT(result == 3); //Making sure 3 objects are added.
	iConsole->Printf(KAddedMultipleMyObjectElements);
	iConsole->Printf(KPrintMsgFormat,&(myObjectsToAdd[1].GetBuf()));
	iConsole->Printf(KPrintMsgFormat,&(myObjectsToAdd[2].GetBuf()));
	iConsole->Printf(KPrintMsgFormat,&(myObjectsToAdd[3].GetBuf()));
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());
	
	_LIT(KAddMyObjectsToCircularBuffer,"\nAdding objects of TMyClasss to circular buffer is successful");
	iConsole->Printf(KAddMyObjectsToCircularBuffer);
	iConsole->Printf(KPressAKey);
	iConsole->Getch();
	
	
	//Array to hold removed elements from circular buffer.
	TMyClass myRemovedObjects[6];
	_LIT(KRemovedMultipleMyObjectElements,"\nTMyClass objects removed are: ");
	
	// Remove multiple TMyClasss from the circular buffer.
	result= circularBuffer->Remove(&myRemovedObjects[0],2);
	ASSERT(result == 2);
	iConsole->Printf(KRemovedMultipleMyObjectElements);
	iConsole->Printf(KPrintMsgFormat,&(myRemovedObjects[0].GetBuf()));
	iConsole->Printf(KPrintMsgFormat,&(myRemovedObjects[1].GetBuf()));
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());	

	// Add two elements to circular buffer
	result= circularBuffer->Add(&myObjectsToAdd[4],2);
	User::LeaveIfError(result);
	ASSERT(result == 2);
	iConsole->Printf(KAddedMultipleMyObjectElements);
	iConsole->Printf(KPrintMsgFormat,&(myObjectsToAdd[4].GetBuf()));
	iConsole->Printf(KPrintMsgFormat,&(myObjectsToAdd[5].GetBuf()));
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());
	
	// Remove multiple TMyClasss from the circular buffer.
	result= circularBuffer->Remove(&myRemovedObjects[2],4);
	ASSERT(result == 4);
	iConsole->Printf(KRemovedMultipleMyObjectElements);
	iConsole->Printf(KPrintMsgFormat,&(myRemovedObjects[2].GetBuf()));
	iConsole->Printf(KPrintMsgFormat,&(myRemovedObjects[3].GetBuf()));
	iConsole->Printf(KPrintMsgFormat,&(myRemovedObjects[4].GetBuf()));
	iConsole->Printf(KPrintMsgFormat,&(myRemovedObjects[5].GetBuf()));
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());	
	
	for(TInt index=0;index<6;index++)
		{
		// The removed objects are same as added ones and are in order.
		ASSERT(myRemovedObjects[index].GetBuf() == myObjectsToAdd[index].GetBuf());
		}
	
	_LIT(KRemoveMyObjectsFromCircularBuffer,"\nRemoving TMyClass objects from circular buffer is successful");
	iConsole->Printf(KRemoveMyObjectsFromCircularBuffer);
	iConsole->Printf(KPressAKey);
	iConsole->Getch();
	
	//Pop the circular buffer off the cleanup stack and destroy it.
	CleanupStack::PopAndDestroy(circularBuffer);
	}

/**
Constructs a circular buffer of RBuf objects.
*/
void CCircularBufferExample::CircularBufferOfRClasssL()
	{
	_LIT(KCircularBufferForRBuf,"\n *****Circular buffer of R Class*****");
	iConsole->Printf(KCircularBufferForRBuf);
	// Creates a circular buffer containing RBuf.
	CCirBuf<RBuf>* circularBuffer=new(ELeave)CCirBuf<RBuf>;
	// Push the circular buffer onto the cleanup stack.
	CleanupStack::PushL(circularBuffer);
	
	const TInt KMaxElements = 2;
	// Sets the maximum capacity of this circular buffer.
	circularBuffer->SetLengthL(KMaxElements);// max capacity is KMaxElements
	iConsole->Printf(KConstruct);
	_LIT(KConstructCircularBufferForRBuf, "\n Construction of circular buffer of R Class is successful");
	iConsole->Printf(KConstructCircularBufferForRBuf);		

	RBuf bufToAdd1;
	_LIT(KBuffer1,"11111");
	// Create an buffer descriptor and assign KBuffer1 data to the descriptor.
	bufToAdd1.CreateL(KBuffer1);

	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());
	
	_LIT(KAddedElements,"Added Element: ");
	iConsole->Printf(KAdd);
	TInt result=circularBuffer->Add(&bufToAdd1); // Add a Rbuf object to the circular buffer.
	User::LeaveIfError(result);
	ASSERT(result == 1);
	iConsole->Printf(KAddedElements);
	iConsole->Printf(bufToAdd1);
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());		
	
	_LIT(KBuffer2,"22222");
	RBuf bufToAdd2;
	// Create an buffer descriptor and assign KBuffer2 data to the descriptor.
	bufToAdd2.CreateL(KBuffer2);
	
	result=circularBuffer->Add(&bufToAdd2); // Add a Rbuf object to the circular buffer.
	User::LeaveIfError(result);
	ASSERT(result == 1);
	iConsole->Printf(KAddedElements);
	iConsole->Printf(bufToAdd2);
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());		
	
	_LIT(KAddRBufferToCircularBuffer,"\nAdding to circular buffer of R Class is successful\n");
	iConsole->Printf(KAddRBufferToCircularBuffer);
	
	RBuf bufToRemove;
	iConsole->Printf(KRemove);
	_LIT(KElementRemoved,"Removed Element: ");
	
	result=circularBuffer->Remove(&bufToRemove); //bufToRemove will point to bufToAdd1 location.
	ASSERT(result == 1);
	iConsole->Printf(KElementRemoved);
	iConsole->Printf(bufToRemove);
	iConsole->Printf(KNumberOfObjects, circularBuffer->Count());		

	
	bufToAdd2.Close(); 
	bufToRemove.Close(); 
	CleanupStack::PopAndDestroy(circularBuffer);
	_LIT(KPressAnyKeyToExit,"\nPress any key to exit ");
	iConsole->Printf(KPressAnyKeyToExit);
	iConsole->Getch();
	}


void MainL()
	{
	CCircularBufferExample* app= CCircularBufferExample::NewL();
	CleanupStack::PushL(app);
	
	// Circular buffer containing integers.
	app->CircularBufferOfIntsL();

	// Circular buffer containing TMyClasss.
	app->CircularBufferOfMyObjectsL();
	
	// Circular buffer containing RBuf 
	app->CircularBufferOfRClasssL();
	CleanupStack::PopAndDestroy(app);
	} 

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;

	CTrapCleanup* cleanup = CTrapCleanup::New();
	if(cleanup == NULL)
		{
		return KErrNoMemory;
		}
	TRAPD(err, MainL());
	if(err !=KErrNone)
		{
		_LIT(KFailed, "\nFailed to complete");
		User::Panic(KFailed, err);
		}	
	delete cleanup;

	__UHEAP_MARKEND;
	return KErrNone;
	}
