// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This provides various functions for window event handling, control and access.
//



/**
 @file
*/

#include "transparent.h"
/**
Calls DoSetTransparencyFactorL() when CPeriodic is active
@param aPtr Pointer to the callback function
*/
static TInt CallDoChangeTransparencyL(TAny *aPtr)
	{
	return (static_cast<CTransparentUtil *> (aPtr)->DoChangeTransparencyL());
	}

/**
Calls DoSetTransparentRegion() when CPeriodic is active
@param aPtr Pointer to the callback function
*/
static TInt CallDoSetTransparentRegion(TAny *aPtr)
	{
	return (static_cast<CTransparentUtil *> (aPtr)->DoSetTransparentRegion());
	}

/**
Constructor
*/
CTransparentUtil::CTransparentUtil()
	{
	}

/**
Creates an object of type CTransparentUtil
@return New CTransparentUtil object
*/
CTransparentUtil* CTransparentUtil::NewL()
	{
	CTransparentUtil* self = new(ELeave) CTransparentUtil();
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

/**
Destructor
*/
CTransparentUtil::~CTransparentUtil()
	{
	if(iPeriodicObject->IsActive())
	    {
	    iPeriodicObject->Cancel();
	    }
	delete iPeriodicObject;
	delete iBitmapGc;
	delete iBitmapDevice;
	delete iEventHandler;
	
	iForegndWindowTrans.Close();
	iFgndWinTransRegion.Close();	
	iBackgndWindow.Close();
	iGroup.Close();

	delete iGc;
	delete iScr;
	iWs.Close();
	}

/**
Initializations for window server session, screen device, graphics context
window group, background window, foreground window, event handler,
blank window and active object for delay.
@leave KErrNoMemory
@leave System-wide error codes.
*/
void CTransparentUtil::ConstructL()
	{
	// Connect the client session to the window server.
	User::LeaveIfError(iWs.Connect());

	// Construct a new screen device attached to the window server session.
	iScr = new(ELeave) CWsScreenDevice(iWs);
	User::LeaveIfError(iScr->Construct(0));	// screen number (0 is first screen)

	// Create the graphics context	but do not initialise
	iGc = new(ELeave) CWindowGc(iScr);

	// Complete construction
	User::LeaveIfError(iGc->Construct());

	// Create an initialised window group handle within the session.
	iGroup  = RWindowGroup(iWs);
	User::LeaveIfError(iGroup.Construct((TUint)&iGroup, ETrue));

	// Initialise the background window handle
	iBackgndWindow = RWindow(iWs);

	// Initialise the foreground window handle to demonstrate TransparencyVariance(). 
	iForegndWindowTrans = RWindow(iWs);

    // Initialise the foreground window handle to demonstrate TransparentRegionL()
	iFgndWinTransRegion = RWindow(iWs);
	
	// Create the CPeriodic object to generate periodic events
	iPeriodicObject=CPeriodic::NewL(CActive::EPriorityUserInput);
	
    // Construct the event handler
    iEventHandler = CEventHandler::NewL(iWs);	

	// Construct foreground and background windows
	User::LeaveIfError(iBackgndWindow.Construct(iGroup, (TUint)&iBackgndWindow));
	User::LeaveIfError(iForegndWindowTrans.Construct(iGroup, (TUint)&iForegndWindowTrans));
	User::LeaveIfError(iFgndWinTransRegion.Construct(iGroup, (TUint)&iFgndWinTransRegion));

	iForegndWindowTrans.SetRequiredDisplayMode(EColor16MA); 
	iFgndWinTransRegion.SetRequiredDisplayMode(EColor16MA); 
	
    // Set background window size
    TSize bgRectSize(iScr->SizeInPixels().iWidth/2,iScr->SizeInPixels().iHeight);
    iBackgndWindow.SetExtent(TPoint(),bgRectSize);
    iBackgndWindow.Activate();

    // Set the size of the foreground window to span the upper half of the background window
    TSize fgRectSize(iScr->SizeInPixels().iWidth/2,iScr->SizeInPixels().iHeight/2);
    iForegndWindowTrans.SetExtent(TPoint(),fgRectSize);
    //Enable the use of alpha channel to vary foreground window's transparency
    User::LeaveIfError(iForegndWindowTrans.SetTransparencyAlphaChannel());
    iForegndWindowTrans.Activate();
    
    // Set the size of the foreground window to overlap the bottom half of the background window
    iFgndWinTransRegion.SetExtent(TPoint(0, iScr->SizeInPixels().iHeight/2),fgRectSize);

    //Enable the use of alpha channel to vary foreground window's transparency
    User::LeaveIfError(iFgndWinTransRegion.SetTransparencyAlphaChannel()); 
    iFgndWinTransRegion.Activate();
    
    TRect circleRect(TPoint(80,50), TSize(170, 170));   
    //Draw the background window using the colour specified by the second parameter 
    //and draw a circle on top of it with the colour specified by the third parameter. 
    _LIT(KBackGrndWindow, "Background window");
    DrawWindow(iBackgndWindow, TRgb(234,234,236), TRgb(0,255,0), circleRect , KBackGrndWindow);
    
    //issue request to the event handler. 
    iEventHandler->IssueRequest();
	}

/**
Clears the window to the selected colour and redraws the background window
@param aWindow The handle to the window
@param aColor1 Colour to be used when the window is cleared
@param aColor2 Colour in which the window must be drawn
@param aRect Area to be drawn
*/
void CTransparentUtil::DrawWindow(RWindow& aWindow, TRgb aColor1, TRgb aColor2, TRect aRect, const TDesC& aText)
	{
	aWindow.Invalidate();
	aWindow.BeginRedraw();
	iGc->Activate(aWindow);

	iGc->SetBrushStyle(CGraphicsContext::ESolidBrush);
	iGc->SetPenStyle(CGraphicsContext::ENullPen);
	iGc->SetBrushColor(aColor1);
	iGc->Clear();
	iGc->SetBrushColor(aColor2);
	iGc->DrawEllipse(aRect);
    iGc->UseFont(iEikonEnv->NormalFont());
    iGc->DrawText(aText, TPoint(10, aWindow.Size().iHeight/2 + 5));
	iGc->Deactivate();
	aWindow.EndRedraw();
	iWs.Flush();
	}

/**
Creates an opaque background window and draws a circle onto it.
Creates a transparent foreground window and draws an ellipse onto it.
Waits for button press events and starts the active scheduler upon
the corresponding button being pressed.
@leave System-wide error codes.
*/
void CTransparentUtil::TransparencyVarianceL()
	{ 
	//Clear the foreground windows to the selected colour. 
	ClearWindow(iWs, iForegndWindowTrans, iGc, TRgb::Gray16(10));
	      
	// Issue request to the active scheduler after receiving the event.
	 if((iPeriodicObject->IsActive()))
	     {	     
	     iPeriodicObject->Cancel();
	     }
     iPeriodicObject->Start(0,KDelay,TCallBack(CallDoChangeTransparencyL,this));
	}

/**
Clears the window to the selected colour
@param aSession	The window server session
@param aWindow	The handle to the window
@param aGc Graphics context of the window
@param aColor Colour to be used when the window is cleared
*/
void CTransparentUtil::ClearWindow(RWsSession& aSession, RWindow& aWindow, CWindowGc* aGc, TRgb aColor)
	{
	aWindow.Invalidate();
	aWindow.BeginRedraw();
	aGc->Activate(aWindow);
	aGc->SetBrushColor(aColor);
	aGc->Clear();
	aGc->Deactivate();
	aWindow.EndRedraw();
	aSession.Flush();
	}

/**
Varies the alpha value of the window up to 255
i.e. until the window turns opaque
@return ETrue if the alpha value is less than its maximum limit otherwise return EFalse.
*/
TBool CTransparentUtil::DoChangeTransparencyL()
	{
	//iAlphaValue is the alpha value and the foreground window is redrawn to show 
	//periodic transparency change. 
	iForegndWindowTrans.SetBackgroundColor( TRgb(255,0, 0, iAlphaValue));

	TRect drawEllipse(TPoint(170, 45), TSize(65, 45));
	_LIT(KForeGrndWindowforTrans, "Foreground window for Vary Tra...");
	DrawWindow(iForegndWindowTrans, TRgb(255,0,0,iAlphaValue), TRgb(100,100,100, iAlphaValue), drawEllipse, KForeGrndWindowforTrans);

	iAlphaValue += 20;
	if (iAlphaValue < 255)
		{
		return ETrue;
		}
	else
		{
		iAlphaValue=0;
		iPeriodicObject->Cancel();
		return EFalse;
		}
	}

/**
Creates an transparent foreground window with some region of it made opaque.
Waits for button press events and starts the active scheduler upon
the corresponding button being pressed.
@leave System-wide error codes.
*/
void CTransparentUtil::TransparentRegionL()
	{   
    //iBoolVisible is a boolean variable used to toggle the visibility of foreground
    //window on TransparentRegionL().. button press
    if(iBoolVisible)
        {
        iFgndWinTransRegion.SetVisible(EFalse);
        iBoolVisible = EFalse; 
        iWs.Flush();
        }
    else
        {  
        iFgndWinTransRegion.SetVisible(ETrue);       
        
        //set background colour of the foreground window as blue and alpha value as 50. 
        iFgndWinTransRegion.SetBackgroundColor(TRgb(0, 0 , 255 , 50));
    
        //create a rectsize which is half the size of the foreground window. 
        TSize rectsize1(iScr->SizeInPixels().iWidth/2,iScr->SizeInPixels().iHeight/4);
        TRect chopRect1(TPoint(0, iScr->SizeInPixels().iHeight/4), rectsize1);
        
        //Take a region which is the lower half of the foreground window. 
        iRegion = TRegionFix<2>(chopRect1);
        //Use SetTransparentRegion() to keep only the region specified by iRegion as transparent.
        iFgndWinTransRegion.SetTransparentRegion(iRegion);
    
        //Invalidate and begin redraw on the opaque region of the foreground window. 
        TRect chopRect2(TPoint(0, 0), rectsize1);
        //Draw a circle on the opaque region of the foreground window. 
        TRect circleRect2(TPoint(60,20), TSize(40, 40));
        _LIT(KOpqRegion, "Opaque region of the transparent window");
        DrawWindowRegion(iFgndWinTransRegion, TRgb(255, 255, 0), TRgb(100, 100, 0), chopRect2, circleRect2, KOpqRegion, TPoint(10, 15)); 
        
        //Invalidate and begin redraw on the transparent region of the foreground window. 
        TRect circleRect3(TPoint(110,80), TSize(40, 40));
        _LIT(KTransRegion, "Transparent region of the transparent window");
        DrawWindowRegion(iFgndWinTransRegion, TRgb(0, 0 , 255 , 50), TRgb(255,0,0, 100), chopRect1, circleRect3, KTransRegion, TPoint(10, 75)); 
        iBoolVisible = ETrue; 
        }
	}

/**
Clears the window to the selected colour and redraws the region of the foreground window
@param aWindow The handle to the window
@param aColor1 Colour to be used when the window is cleared
@param aColor2 Colour in which the window must be drawn
@param aChopRect Region of the foreground window to be drawn. 
@param aRect Area to be drawn
@param aText Text to be drawn. 
@param aPoint Point at which text should be drawn. 
*/
void CTransparentUtil::DrawWindowRegion(RWindow& aWindow, TRgb aColor1, TRgb aColor2, TRect aChopRect, TRect aRect, const TDesC& aText, const TPoint& aPoint)
    {
    aWindow.Invalidate(aChopRect);
    aWindow.BeginRedraw(aChopRect);
    iGc->Activate(aWindow);

    iGc->SetBrushStyle(CGraphicsContext::ESolidBrush);
    iGc->SetPenStyle(CGraphicsContext::ENullPen);

    iGc->SetBrushColor(aColor1);
    iGc->Clear();
        
    iGc->SetBrushColor(aColor2);
    iGc->DrawEllipse(aRect);
    iGc->SetBrushColor(TRgb(0, 0, 0)); 
    iGc->UseFont(iEikonEnv->NormalFont());
    iGc->DrawText(aText, aPoint);
    iGc->Deactivate();
    aWindow.EndRedraw();
    iWs.Flush();
    }
