/*
 * Copyright  2008 Nokia Corporation.
 */

#ifndef __DBMSAPPUI_H__
#define __DBMSAPPUI_H__

#include <aknappui.h>
#include "DBMSEngine.h"

// Forward declarations
class CDBMSAppView;
class CDBMSListboxView;
class CDBMSEditorView;
class CDesC16ArrayFlat;

// Name of database file in the application directory
_LIT(KDatabaseFile,"DBMS.dat");


/**
 * Class: CEComCalculatorAppUi
 *
 * Description: An instance of class CDBMSAppUi is the UserInterface
 *              part of the AVKON application framework for the DBMS
 *              example application.
 *
 *              The relevant part of this example is the database engine class
 *              CDBMSDb in DBEngine.h. The GUI is only a simple tester
 *              for the engine.
 */
class CDBMSAppUi : public CAknAppUi
    {
public:

    /**
     * Function:    ConstructL
     *
     * Description: Perform the second phase construction of a
     *              CDBMSAppUi object. This needs to be public due to
     *              the way the framework constructs the AppUi.
     */
    void ConstructL();

    /**
     * Function:    CDBMSAppUi
     *
     * Description: Perform the first phase of two phase construction. This
     *              needs to be public due to the way the framework constructs
     *              the AppUi.
     */
    CDBMSAppUi();


    /**
     * Function:    ~CDBMSAppUi
     *
     * Description: Destructor of the object
     */
    ~CDBMSAppUi();


public: // from CAknAppUi

    /**
     * Function:    DynInitMenuPaneL
     *
     * Description: Dynamically determine what menu items are visible, when
     *              the menu is shown. This method is called by the framework
     *              just before the menu is shown.
     *
     * Param:       aResourceId menu pane resource id. Identifies the menu
     *              pane (this example has only one, so not relevant)
     *
     * Param:       aMenuPane The in-memory representation of the menu pane
     */
    void DynInitMenuPaneL(TInt aResourceId, CEikMenuPane* aMenuPane);


    /**
     * Function:    HandleCommandL
     *
     * Description: Handle user menu selections
     *
     * Param:       aCommand the enumerated code for the menu option selected.
     */
    void HandleCommandL(TInt aCommand);
    
	/**
     * HandleStatusPaneSizeChange.
     * Called by framework when the view layout is changed.
     */
    void HandleStatusPaneSizeChange();

private: // Menu command helpers

    /**
     * Function:    OpenDatabaseL
     *
     * Description: Create instance of iDBMSDb and open existing database.
     *              The database will be ready for operations.
     */
    void OpenDatabaseL();

    /**
     * Function:    CreateDatabaseL
     *
     * Description: Create instance of iDBMSDb and create new database.
     *              The database will be ready for operations.
     */
    void CreateDatabaseL();

    /**
     * Function:    RemoveDatabaseL
     *
     * Description: Detele existing database. Removes the database file.
     */
    void RemoveDatabaseL();

    /**
     * Function:    CloseDatabaseL
     *
     * Description: Close an open database. Database opened with OpenDatabaseL
     *              or CreateDatabaseL must be closed, when not used any more.
     */
    void CloseDatabaseL();

    /**
     * Function:    ShowDBMSEditorView
     *
     * Description: Show a book editor. User can give in book details.
     */
    void ShowBookEditorViewL();

    /**
     * Function:    AddBookL
     *
     * Description: Adds a book to database. Queries book details from the
     *              book editor view.
     *
     * Param:       aUseSql determines the CDBMSDb method used for
     *              insertion (uses sql or proprietary DBMS API).
     */
    void AddBookL(TBool aUseSql);

    /**
     * Function:    RemoveBookL
     *
     * Description: Remove selected book from the database. The selection is
     *              done from the lisbox having book titles. If there are
     *              multiple matches for the selected book title, they are all
     *              are removed.
     */
    void RemoveBookL();

    /**
     * Function:    RemoveAllBooksL
     *
     * Description: Remove all books from database.
     */
    void RemoveAllBooksL();

    /**
     * Function:    UpdateBookTitleL
     *
     * Description: Change the title for a selected book.
     */
    void UpdateBookTitleL();

    /**
     * Function:    ShowAllBooksL
     *
     * Description: Get list of all books from database. Show the books
     *              using the list view.
     */
    void ShowAllBooksL();

    /**
     * Function:    SearchBooksL
     *
     * Description: Query user for a book title and find matching books.
     *              Show the result in the lisbox view.
     */
    void SearchBooksL();

    /**
     * Function:    IndexFindL
     *
     * Description: Find a book using index. The method finds book details
     *              for the book currently selected in the listbox view.
     *              Show the full book info in raw format.
     */
    void IndexFindL();

    /**
     * Function:    AddDateColumnL
     *
     * Description: Add a date column to DBMS database. Show the columns.
     *              This is here just to demonstrate schema updates.
     */
    void AddDateColumnL();

    /**
     * Function:    RemoveDateColumnL
     *
     * Description: Remove the date column from DBMS database. Show the
     *              columns after removal. This is here just to demonstrate
     *              schema updates.
     */
    void RemoveDateColumnL();

    /**
     * Function:    ShowColumnsL
     *
     * Description: Show all columns in the DBMS database in Books table.
     */
    void ShowColumnsL();

private: // Other helpers

    /**
     * Function:    ApplicationDriveAndPathL
     *
     * Description: Retrieve full path of the application.
     */
    TFileName ApplicationDriveAndPathL() const;
    
    /**
     * Function:    DatabaseDriveAndPathL
     *
     * Description: Retrieve drive and path where the DB is created.
     */
	TFileName DatabaseDriveAndPathL() const;

    /** View states */
    enum TViewId
        {
        ENone,          // On the startup there is no view
        EMainView,      // Main view initially visible.
        EListView,  // Showing list of books
        EColumnsView,   // Showing columns of Books table
        EEditorView // For adding a new book to database
        };

    /**
     * Function:    ChangeViewL
     *
     * Description: Switches the view. Deletes previous view and creates new
     */
    void ChangeViewL(TViewId aNewView);

    /**
     * Function:    ShowNoteL
     *
     * Description: Show simple information note.
     */
    void ShowNoteL(const TDesC& aMessage) const;

    /**
     * Function:    TitlesArrayL
     *
     * Description: Construct array of book titles. Take in a aFullInfoArray,
     *              which contains full info for the books. Parse the titles
     *              and append them to result array.
     */
    CDesCArrayFlat* TitlesArrayL(const CDesCArrayFlat* aFullInfoArray) const;

    /**
     * Function:    QueryTextL
     *
     * Description: Provide simple text query.
     */
    TBool QueryTextL(TDesC& aPrompt, TDes& aResultText) const;

private:

    TViewId iCurrentView;

    /** The main view of application               */
    CDBMSAppView* iAppView;

    /** List box view to show list of books        */
    CDBMSListboxView* iListboxView;

    /** Editor view to allow user input for a book */
    CDBMSEditorView* iBookEditorView;

    /** The database engine                        */
    CBookDb* iBookDb;

    /** Full path to database file                 */
    TFileName     iDatabaseFile;
    };

#endif // __DBMSAPPUI_H__

