// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

#include "FeatureChecker.h"
#include <e32cons.h>

void CFeatureChecker::ConstructL()
	{
	iFeatNotifier = CFeatureNotifier::NewL(*this);
	}

CFeatureChecker* CFeatureChecker::NewL()
	{
	CFeatureChecker* self=new(ELeave)CFeatureChecker();
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

/**
Cancels any outstanding notification request before deleting the feature notifier.
*/
CFeatureChecker::~CFeatureChecker()
	{
	delete iFeatNotifier;
	}

/**
Subscribes to receive change notification for an array of feature UIDs.
*/
void CFeatureChecker::NotifyFeatureL()
	{
	RFeatureUidArray featArray;
	CleanupClosePushL(featArray);
	TUid uidFeature;

	//Number of feature UIDs which are subscribed to for notification.
	const TInt KNumFeaturesforNotification(10);

	//Base feature UID for which notification requests are made. This is the same UID range which 
	//the user can use when adding features to the FeatMngrExample process.
	TUid KFeatureUidforNotification = {0x12000000};
	
	//Append the UIDs to featArray.
	for( int k=0; k<=KNumFeaturesforNotification; k++)
		{
		uidFeature.iUid = KFeatureUidforNotification.iUid + k;
		featArray.Append(uidFeature);
		}
	
	//Request notification for user defined features.
	User::LeaveIfError(iFeatNotifier->NotifyRequest(featArray));
	CleanupStack::PopAndDestroy(&featArray);
	}

/**
Any error in the notifier is handled by this function, not used in this example implementation.
*/
void CFeatureChecker::HandleNotifyError(TInt /*err*/)
	{
	}

/**
This is a callback method of MFeatureObserver which is used to notify the FeatureChecker 
process about changes to a feature. TFeatureChangeType specifies the type of change.
The change type is displayed to the user in the form of an InfoPrint.
*/
void CFeatureChecker::HandleNotifyChange(TFeatureChangeType aType, TFeatureEntry /*aFeature*/)
	{
	const TInt KFeatureChangeTypeMaxSize(45);
	TBuf<KFeatureChangeTypeMaxSize> changetype;
	switch(aType)
		{
		case EFeatureNoChange: 
			_LIT(KNoChange, "Feature status not changed");
			changetype.Copy(KNoChange);
			break;
		case EFeatureStatusUpdated :
			_LIT(KStatusUpdated, "Feature status changed to enabled or disabled");
			changetype.Copy(KStatusUpdated);
			break;
		case EFeatureDataUpdated: 
			_LIT(KDataUpdated, "Feature data changed");
			changetype.Copy(KDataUpdated);
			break;
		case EFeatureStatusDataUpdated: 
			//not used in the current version.
			_LIT(KStatusDataUpdated, "Feature status and data changed");
			changetype.Copy(KStatusDataUpdated);
			break;
		case EFeatureRediscover: 
			//not used in the current version.
			_LIT(KFeatureRediscover, "Complex change occurred, reload all features");
			changetype.Copy(KFeatureRediscover);
			break;
		case EFeatureFeatureCreated:
			_LIT(KFeatureCreated, "New feature has been added to the system");
			changetype.Copy(KFeatureCreated);
			break;
		case EFeatureFeatureDeleted: 
			_LIT(KFeatureDeleted, "Feature has been deleted");
			changetype.Copy(KFeatureDeleted);
			break;
		}
	//Print the change type as an infoprint.
	User::InfoPrint(changetype);
	}

/**
This is the main function which creates an object of CFeatureChecker that issues
the request for feature change notification.
*/
static void MainL()
	{
	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
	CleanupStack::PushL(scheduler);
	CActiveScheduler::Install(scheduler);
	
	CFeatureChecker* featChecker = CFeatureChecker::NewL();
	CleanupStack::PushL(featChecker);
	//Issue feature change notification request.
	featChecker->NotifyFeatureL();
	CActiveScheduler::Start();
	
	CleanupStack::PopAndDestroy(featChecker);
	CleanupStack::PopAndDestroy(scheduler);
	}

extern TInt E32Main()
	{
	// Create cleanup stack
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();
	if(cleanup == NULL)
		{
		return KErrNoMemory;
		}
	// Run application code inside TRAP harness.
	TRAPD(mainError, MainL());
	if(mainError != KErrNone)
		{
		_LIT(KUserPanic,"Failed to complete");	
		User::Panic(KUserPanic, mainError);
		}
	delete cleanup;
	__UHEAP_MARKEND;
	return KErrNone;
	}
