// TXUT.CPP
//
//  2004 Nokia Corporation.  All rights reserved.
//

#include <msvstd.h>		// TMsvEntry
#include <msvstore.h>	// CMsvStore
#include <MSVUIDS.H>	//KUidMsvFolderEntry
#include <centralrepository.h>
#include "txut.h"

// system includes
#include <msvuids.h>	//KUidMsvFolderEntry
const TUint32 KNullId = 0x00000000;
const TUint32 KIncrementAccount = 0x00100000;
const TInt KMaxAccount = 2048;
const TUint32 KDefaultServiceId = 0x80000000; // set top bit
const TUint32 EAccountMask = 0x800FFFFF;

// Key IDs for particular text mtm settings in repository
enum
	{
	EServiceId = 0,
	EFolderSettingId = 1
	};
// user includes
// standard NewL
EXPORT_C CMTMTxtSettings* CMTMTxtSettings::NewL()
	{
	CMTMTxtSettings* self = new (ELeave) CMTMTxtSettings();
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop();
	return self;
	}

// destructor
CMTMTxtSettings::~CMTMTxtSettings()
	{
	delete iRepository;
	}

// delete all settings for specified service
EXPORT_C void CMTMTxtSettings::DeleteSettingsL(TMsvId aServiceId)
	{
	TUint32 serviceKey = FindAccountL(aServiceId);
	DeleteSettingL(serviceKey);
	DeleteSettingL(serviceKey + EFolderSettingId);
	}

/**
Loads settings for textmtm service.

@param	aServiceId ID of service entry

@param	aSettings Service settings
*/
EXPORT_C void CMTMTxtSettings::LoadSettingsL(TMsvId aServiceId, TMTMTxtSettings& aSettings) const
	{
	TUint32 serviceKey = FindAccountL(aServiceId);
	TFileName rootFolder;
	User::LeaveIfError(iRepository->Get(serviceKey + EFolderSettingId, rootFolder));
	aSettings.SetRootFolder(rootFolder);
	}

/**
Stores settings for textmtm service.

@param	aServiceId ID of service entry

@param	aSettings Service settings
*/
EXPORT_C void CMTMTxtSettings::SaveSettingsL(TMsvId aServiceId, const TMTMTxtSettings& aSettings)
	{
	TUint32 accountId = 0;
	TInt error = 0;
	// see if account already exists
	TRAP(error, accountId = FindAccountL(aServiceId));
	if (error != KErrUnknown) User::LeaveIfError(error);
	// doesn't already exist, so get id of new account
	if (error == KErrUnknown) accountId = GetNextAccountSlotL();

	TRAP( error,
		// Save settings to CenRep
		CreateOrSetL(accountId, static_cast<TInt>(aServiceId));
		CreateOrSetL(accountId + EFolderSettingId, aSettings.RootFolder());
		);
	if (error != KErrNone)
		{
		// saving settings to CenRep failed, so cleanup account and leave
		DeleteSettingsL(aServiceId);
		User::Leave(error);
		}
	}

/**
Sets the default textmtm service.

@param	aService The default service
*/
EXPORT_C void CMTMTxtSettings::SetDefaultServiceL(TMsvId aService)
	{
	CreateOrSetL(KDefaultServiceId, static_cast<TInt>(aService));
	}

/**
Gets the default textmtm service.

@return The default service

@leave KErrNotFound If default service setting does not exist.
*/
EXPORT_C TMsvId CMTMTxtSettings::DefaultServiceL() const
	{
	// Get the service Id from CenRep
	TInt temp = 0;
	User::LeaveIfError(iRepository->Get(KDefaultServiceId, temp));
	return static_cast<TMsvId>(temp);
	}

/**
Remove the default service setting.
*/
EXPORT_C void CMTMTxtSettings::DeleteDefaultServiceSettingL()
	{
	DeleteSettingL(KDefaultServiceId);
	}

// create cenrep repository in which to store textmtm settings
void CMTMTxtSettings::ConstructL()
	{

//#ifdef __WINS__
//	const TUid KUidMtm = { KUidMsgTypeText };

//#else
	const TInt KUidDefaultMtmRepositoryFileValue = 0x10274556;
	const TUid KUidMtm = { KUidDefaultMtmRepositoryFileValue };
//#endif

	iRepository = CRepository::NewL(KUidMtm);

	}

// sets (or creates if it does not already exist) a string key
void CMTMTxtSettings::CreateOrSetL(TUint aKey, const TDesC& aValue)
	{
	TInt error = iRepository->Set(aKey, aValue);
	if (error == KErrNotFound)
		{
		// setting does not exist, so create it
		User::LeaveIfError(iRepository->Create(aKey, aValue));
		}
	else
		{
		User::LeaveIfError(error);
		}
	}

// sets (or creates if it does not already exist) an integer key
void CMTMTxtSettings::CreateOrSetL(TUint aKey, TInt aValue)
	{
	TInt error = iRepository->Set(aKey, aValue);
	if (error == KErrNotFound)
		{
		// setting does not exist, so create it
		User::LeaveIfError(iRepository->Create(aKey, aValue));
		}
	else
		{
		User::LeaveIfError(error);
		}
	}

// Leaves with KErrUnknown if account does not exist
TUint32 CMTMTxtSettings::FindAccountL(TMsvId aService) const
	{
	RArray<TUint32> accounts;
	CleanupClosePushL(accounts);
	TInt error = iRepository->FindEqL(KNullId, static_cast<TUint32>(EAccountMask), static_cast<TInt>(aService), accounts);
	if (error == KErrNotFound)
		{
		// account does not exist
		User::Leave(KErrUnknown);
		}
	else
		{
		User::LeaveIfError(error);
		}

	if (accounts.Count()>1)
		{
		// There should be only one account for the service
		User::Leave(KErrOverflow);
		}

	TUint32 account = accounts[0];
	CleanupStack::PopAndDestroy(&accounts);
	return account;
	}
//	TMTMTxtSettings: service settings storage
//

EXPORT_C void TMTMTxtSettings::ExternalizeL( RMsvWriteStream& aWriteStream ) const
	{
	// store iRootFolder
	aWriteStream << iRootFolder;
	return;
	}

EXPORT_C void TMTMTxtSettings::InternalizeL( RMsvReadStream& aReadStream )
	{
	aReadStream >> iRootFolder;
	return;
	}

EXPORT_C void TMTMTxtSettings::StoreL(CMsvStore& aMsvStore) const
	{
	RMsvWriteStream out;
	out.AssignLC( aMsvStore, KUidTxtMTMSettings ); // pushes 'out' to the stack
	ExternalizeL(out);
	out.CommitL();
	CleanupStack::PopAndDestroy();
	}

// get a base (account) id to identify all keys for a particular service
TUint CMTMTxtSettings::GetNextAccountSlotL()
	{
	TUint32 accountId = KNullId;
	TInt serviceId = 0;
	TInt error = 0;
	TBool found = EFalse;

	for (TInt count = 0; count < KMaxAccount; ++count)
		{
		accountId = accountId + KIncrementAccount;
		error = iRepository->Get(accountId, serviceId);
		if (error == KErrNotFound)
			{
			found = ETrue;
			break;
			}
		else
			{
			User::LeaveIfError(error);
			}
		}

	if (found == EFalse)
		{
		// No empty slot available
		User::Leave(KErrNotFound);
		}

	return accountId;
	}

// delete a setting, and don't give an error if it doesn't exist
void CMTMTxtSettings::DeleteSettingL(TUint32 settingId)
	{
	TInt error = iRepository->Delete(settingId);
	if (error != KErrNotFound)
		{
		User::LeaveIfError(error);
		}
	}
EXPORT_C void TMTMTxtSettings::RestoreL(const CMsvStore& aMessageStore )

	{
	if (aMessageStore.IsPresentL(KUidTxtMTMSettings))
		{
		RMsvReadStream in;
		in.OpenLC( aMessageStore, KUidTxtMTMSettings ); // pushes 'in' to the stack
		InternalizeL(in);
		CleanupStack::PopAndDestroy();
		}
	}

//
// TxtUtils: Generic static utility functions
//

EXPORT_C void TxtUtils::GetEntryFileNameL(TFileName& aFileName, TMsvEntry& aEntry)
// Create absolute file name: default path + aEntry.iDetails + aEntry.iDescription
	{
	CMTMTxtSettings* settings = CMTMTxtSettings::NewL();
	CleanupStack::PushL(settings);
	TMTMTxtSettings root;
	settings->LoadSettingsL(aEntry.iServiceId, root);
	CleanupStack::PopAndDestroy(); //settings
	aFileName = root.RootFolder();
	aFileName.Append(aEntry.iDetails);
	aFileName.Append(aEntry.iDescription);
	if (aEntry.iType == KUidMsvFolderEntry)
		aFileName.Append(KPathDelimiter);
	}

EXPORT_C void TxtUtils::FindFileL(const TDesC& aFileName, const TDesC& aLocation, TFileName& aRetVal)
    {
    RFs fs;
    User::LeaveIfError(fs.Connect());
    CleanupClosePushL(fs);
    TFindFile finder(fs);
    User::LeaveIfError(finder.FindByDir(aFileName, aLocation));
    aRetVal = finder.File();
    CleanupStack::PopAndDestroy();
    }

//
// CMsvOpWait
//	Allows a synchronous wait on a operation
//
EXPORT_C CMsvOpWait* CMsvOpWait::NewLC(TInt aPriority)
	{
	CMsvOpWait* self = new (ELeave) CMsvOpWait(aPriority);
	CleanupStack::PushL(self);
	return self;
	}

CMsvOpWait::CMsvOpWait(TInt aPriority)
: CActive(aPriority)
	{
	CActiveScheduler::Add(this);
	}

EXPORT_C CMsvOpWait::~CMsvOpWait()
	{
	Cancel();
	}

EXPORT_C void CMsvOpWait::Start()
	{
	SetActive();
	}

void CMsvOpWait::RunL()
	{
	CActiveScheduler::Stop();
	}

void CMsvOpWait::DoCancel()
	{
	TRequestStatus* s=&iStatus;
	User::RequestComplete(s, KErrCancel);
	}
//
// CMsvCompOperation
//	An operation which is already completed on construction
//
EXPORT_C CMsvCompOperation* CMsvCompOperation::NewL(CMsvSession& aSession, TUid aMtm,
	const TDesC8& aProgress, TMsvId aService, TRequestStatus& aObserverRequestStatus, TInt aError)
// aMtm and aService set CMsvOperation protected data members
// aProgress is progress information
// aObserverRequestStatus is the active object to signal on completion
// aError is the error code from the relevant operation
	{
	CMsvCompOperation* self = new (ELeave) CMsvCompOperation(aSession, aObserverRequestStatus);
	CleanupStack::PushL(self);
	self->ConstructL(aMtm, aError, aProgress, aService);
	CleanupStack::Pop(self);
	return self;
	}

CMsvCompOperation::CMsvCompOperation(CMsvSession& aSession, TRequestStatus& aObserverRequestStatus)
: CMsvOperation(aSession, EPriorityStandard, aObserverRequestStatus)
	{
	}

EXPORT_C CMsvCompOperation::~CMsvCompOperation()
	{
	Cancel();
	delete iProgress;
	}

void CMsvCompOperation::ConstructL(TUid aMtm, TInt aError, const TDesC8& aProgress, TMsvId aService)
	{
	iProgress = HBufC8::NewL(aProgress.Length());
	*iProgress = aProgress;
	iMtm=aMtm;
	iService=aService;
	//
	CActiveScheduler::Add(this);
	iStatus = KRequestPending;
	SetActive();
	//
	TRequestStatus* pstat=&iStatus;
	User::RequestComplete(pstat, aError);
	iObserverRequestStatus = KRequestPending;
	}

const TDesC8& CMsvCompOperation::ProgressL()
	{
	return *iProgress;
	}

void CMsvCompOperation::DoCancel()
	{
	// does nothing as iStatus has already been completed in in ConstructL()
	}

void CMsvCompOperation::RunL()
	{
	TRequestStatus* pstat=&iObserverRequestStatus;
	User::RequestComplete(pstat, iStatus.Int());
	}
