/*
 * Copyright  2008 Nokia Corporation.
 */


#include "InternetEmailEngine.h"


// ================= MEMBER FUNCTIONS =======================
//

// -------------------------------------------------
// CInternetEmailEngine* CInternetEmailEngine::NewL(
//      MInternetEmailEngineObserver& aObserver )
//  static creation
// --------------------------------------------------
//
CInternetEmailEngine* CInternetEmailEngine::NewL( MInternetEmailEngineObserver& aObserver )
    {
    CInternetEmailEngine* engine=new(ELeave) CInternetEmailEngine( aObserver );
    CleanupStack::PushL(engine);
    engine->ConstructL();
    CleanupStack::Pop(engine);
    return engine;
    }

// ------------------------------------------------------------------
// CInternetEmailEngine::CInternetEmailEngine(
//      MInternetEmailEngineObserver& aObserver )
//      : CActive(CActive::EPriorityStandard), iObserver( aObserver )
//  actual constructor. Creates ao with standard prioritory
// ------------------------------------------------------------------
//
CInternetEmailEngine::CInternetEmailEngine( MInternetEmailEngineObserver& aObserver )
    : CActive(CActive::EPriorityStandard), iObserver( aObserver )
    {
    // add this object to the active scheduler
    CActiveScheduler::Add(this);
    }

// ---------------------------------------------
// CInternetEmailEngine::~CInternetEmailEngine()
// Frees reserved resources
// ---------------------------------------------
//
CInternetEmailEngine::~CInternetEmailEngine()
    {
    if( IsActive() )
        {
        Cancel();
        }
    if(iMsvOp)
        {
        iMsvOp->Cancel();
        }
    delete iMsvOp;
    iMsvOp = NULL;

    delete iEntry;
    iEntry = NULL;

    delete iMtm;
    iMtm = NULL;

    delete iRemoteEntries;
    iRemoteEntries = NULL;

    delete iUiReg;
    iUiReg = NULL;

    delete iClientReg;
    iClientReg = NULL;

    delete iMsvSession; //Session must be deleted last
    iMsvSession = NULL;
    }

// ----------------------------------
// CInternetEmailEngine::ConstructL()
//  EPOC two phased constructor
// ----------------------------------
//
void CInternetEmailEngine::ConstructL()
    {
    // create and open a session towards messaging
    // the open operation is asynchronous and HandleSessionEventL
    // will be called when the open is complete
    iMsvSession=CMsvSession::OpenAsyncL(*this);

    // Default protocol values
    iProtocolType=EProtocolNULL;
    iProtocolUid=KUidMsgTypeSMTP;
    iIsProtocolSet=EFalse;
    iHasImapInbox=EFalse;

    //for message handling
    iServiceId=KMsvRootIndexEntryId;
    iId=KMsvRootIndexEntryId;
    iRemoteEntries = new(ELeave) CMsvEntrySelection();

    // other member initalization
    iState=EInitialising;
    }

// ------------------------------------------
// CInternetEmailEngine::CompleteConstructL()
//  Called when MsvServer session is ready
// ------------------------------------------
//
void CInternetEmailEngine::CompleteConstructL()
    {
    iClientReg = CClientMtmRegistry::NewL( *iMsvSession );
    iUiReg = CMtmUiRegistry::NewL( *iMsvSession );
    }

// --------------------------------------------------------
// void CInternetEmailEngine::RunL()
//  The main statemachine
//
//  Cornerstone i of this module.
// --------------------------------------------------------
//
void CInternetEmailEngine::RunL()
    {

    TBuf8<1> null;

    switch( iState ) // separation between states
        {

        // ------------------------------------
        //  STATE: EConnecting
        //   usually opens internet connection
        // ------------------------------------
        case EConnecting:
            {

            if( iProtocolType==EProtocolPop3 )
                {
                MailCommandL(KPOP3MTMConnect, null);
                }
            else
                {
                //As KIMAP4MTMConnect, but, after connection to
                //the IMAP service, starts a background synchronisation.
                //The call completes when the connection occurs
                //and the synchronisation starts.
                MailCommandL(KIMAP4MTMConnectAndSynchronise, null);
                }

            iState=EFetching;
            SetActive(); //returns to active scheduler which waits for our request
            }
            break;

        // ------------------------------------
        //  STATE: EFetching
        //   handles the remote mail fetch
        // ------------------------------------
        case EFetching:
            {

            // SEPARATION BETWEEN IMAP AND POP
            if( iProtocolType==EProtocolPop3 )
                {
                TImPop3GetMailInfo Pop3GetMailInfo; // setup a mail info buffer
                // setting iMaxEmailSize to KMaxTUint tells messaging to get the headers only
                Pop3GetMailInfo.iMaxEmailSize = KMaxTUint;
                // setup which service to download the messages from
                Pop3GetMailInfo.iDestinationFolder = iServiceId;
                // package up the information
                TPckgBuf<TImPop3GetMailInfo> package(Pop3GetMailInfo);
                // execute a pop3 copy mail command
                MailCommandL(KPOP3MTMCopyAllMailWhenAlreadyConnected,package);
                }
            else
                {
                //Completes (with KErrNone) only when the background synchronisation
                //has finished. It is used to ensure that no UI-initiated
                //folder syncing is performed during the background synchronisation.
                MailCommandL(KIMAP4MTMWaitForBackground,null);
                }

            iState=EDisconnecting;
            SetActive(); //returns to active scheduler which waits for our request
            }
            break;

        // ------------------------------------
        //  STATE: EDisconnecting
        //   Disconnects from remote mailbox
        //   note: state only for pop as
        //   it is only offline mode protocol
        // ------------------------------------
        case EDisconnecting:
            {
            // SEPARATION BETWEEN IMAP AND POP
            if( iProtocolType==EProtocolPop3 )
                {
                MailCommandL(KPOP3MTMDisconnect,null); // execute a disconnect from service command
                }
            else
                {
                //Cancels any operations in progress
                //and sends logout messages to server.
                //As long as the background synchronisation,
                //this will also run any pending delete operations queued while offline.
                MailCommandL(KIMAP4MTMDisconnect,null);
                }

            iState=EDone;
            SetActive();
            }
            break;

        // ------------------------------------
        //  STATE: EDone
        //   Returns our machine to ready state
        // ------------------------------------
        case EDone:
            {
            UpdateRemoteCountL(); // updates model and informs observers
            iState = EReady;
            if(iMsvOp)
                {
                iMsvOp->Cancel();
                }
            delete iMsvOp;
            iMsvOp=NULL;
            }
            break;

        // ------------------------------------
        //  STATE: ECanceling
        //   waits for mailcommands to cancel
        // ------------------------------------
        case ECanceling:
            {
            UpdateRemoteCountL(); // updates model and informs observers
            iState= EReady;
            delete iMsvOp;
            iMsvOp=NULL;
            }
            break;

        default:
            //EReady is default, EInitialising, EReadyButNeedsProtocol
            //handled elsewhere
            break;
        }
    }

// ----------------------------------------------------------------------------
// void CInternetEmailEngine::MailCommandL( const TInt aCommand,TDes8& aParams)
//  Excecutes asynchronous calls for both protocols. Note the specific
//  aSelection and aParams given to InvokeAsyncFunctionL (and later to
//  TransferCommandL) define the functioncall.
//
//  Cornerstone ii of this module.
// ----------------------------------------------------------------------------
void CInternetEmailEngine::MailCommandL( const TInt aCommand, TDes8& aParams)
    {
    // make sure iServiceId is a remote type service
    if( iServiceId != KMsvLocalServiceIndexEntryId )
        {
        iRemoteEntries->Reset();
        iRemoteEntries->AppendL(iServiceId); //aSelection[0] is used in many mtm commands

        // get the msventry for the service id
        CMsvEntry* service = iMsvSession->GetEntryL(iServiceId);
        CleanupStack::PushL(service);

        // make sure the mtm for the service is of approriate type
        if(service->Entry().iMtm == iProtocolUid)
            {
            // delete and reset any outstanding operation
            if(iMsvOp)
                {
                iMsvOp->Cancel();
                }
            delete iMsvOp;
            iMsvOp=NULL;
            iMsvOp=iMtm->InvokeAsyncFunctionL(aCommand,*iRemoteEntries,aParams/*ptr*/,iStatus);
            if( iStatus != KRequestPending )
                {
                User::Leave( iStatus.Int() ); //leave if error
                }
            }
        CleanupStack::PopAndDestroy(service);
        }
    else
        {
        User::Leave(KErrNotFound);
        }
    }

// -----------------------------------------------------------------------
// void CInternetEmailEngine::HandleSessionEventL(TMsvSessionEvent aEvent,
//      TAny* /*aArg1*/,TAny* /*aArg2*/,TAny* /*aArg3*/ )
//  Messaging callback from MMsvSessionObserver
//
//  Cornerstone iii of this module.
// -----------------------------------------------------------------------
void CInternetEmailEngine::HandleSessionEventL(TMsvSessionEvent aEvent,
    TAny* /*aArg1*/,TAny* /*aArg2*/,TAny* /*aArg3*/ )
    {
    switch(aEvent)
        {
         // the messaging server is ready
        case MMsvSessionObserver::EMsvServerReady:
            {
            // ---------------------------------------------
            //  STATE: EInitialising
            //   Completes constructs and changes state when
            //   MsvServer session is ready
            // ---------------------------------------------
            if(iState==EInitialising)
                {
                iState=EReadyButNeedsProtocol;
                CompleteConstructL();
                }
            }
            break;

        case MMsvSessionObserver::EMsvEntriesCreated:
        case MMsvSessionObserver::EMsvEntriesChanged:
        case MMsvSessionObserver::EMsvEntriesDeleted:
        case MMsvSessionObserver::EMsvEntriesMoved:
            //note: we could notify observers here to redraw if entry
            //would be approriate type and we would need more robust ui
        case MMsvSessionObserver::EMsvMtmGroupInstalled:
        case MMsvSessionObserver::EMsvMtmGroupDeInstalled:
        case MMsvSessionObserver::EMsvGeneralError:
        case MMsvSessionObserver::EMsvServerFailedToStart:
        case MMsvSessionObserver::EMsvCorruptedIndexRebuilt:
        case MMsvSessionObserver::EMsvCloseSession:
        case MMsvSessionObserver::EMsvServerTerminated:
        case MMsvSessionObserver::EMsvMediaChanged:
        case MMsvSessionObserver::EMsvMediaUnavailable:
        case MMsvSessionObserver::EMsvMediaAvailable:
        case MMsvSessionObserver::EMsvMediaIncorrect:
        case MMsvSessionObserver::EMsvCorruptedIndexRebuilding:
            break;
        }
    }

// ---------------------------------------------------------------
// void CInternetEmailEngine::DisplayMessageL( const TMsvId &aId )
//  here we use the generic mtm ui (view) architecture
//  and its views and dialogs to display messages in standard way.
//
//  cornerstone iv: of this module.
// ---------------------------------------------------------------
//
void CInternetEmailEngine::DisplayMessageL( const TMsvId &aId )
    {
    // 1. construct the client MTM
    TMsvEntry indexEntry;
    TMsvId serviceId;
    User::LeaveIfError( iMsvSession->GetEntry(aId, serviceId, indexEntry));
    CBaseMtm* mtm = iClientReg->NewMtmL(indexEntry.iMtm);
    CleanupStack::PushL(mtm);

    // 2. construct the user interface MTM
    CBaseMtmUi* uiMtm = iUiReg->NewMtmUiL(*mtm);
    CleanupStack::PushL(uiMtm);

    // 3. display the message
    uiMtm->BaseMtm().SwitchCurrentEntryL(indexEntry.Id());
    CMsvOperationWait* waiter=CMsvOperationWait::NewLC();
    waiter->Start(); //we use synchronous waiter
    CMsvOperation* op = uiMtm->OpenL(waiter->iStatus); //the main "async-sync" call
    CleanupStack::PushL(op);
    CActiveScheduler::Start(); //notice! nested active scheduler for modal operation

    // 4. cleanup for example even members
    CleanupStack::PopAndDestroy(4); // op,waiter, mtm, uimtm
    }


// ---------------------------------------------
// void CInternetEmailEngine::CancelOperation()
//  public interface method for user initiated
//  cancelling of email operations.
// ---------------------------------------------
//
void CInternetEmailEngine::CancelOperation()
    {
    iState= ECanceling; //we switch state and wait for op to cancel

    if( iMsvOp )
        {
        iMsvOp->Cancel(); //iStatus now KErrCancel (-3)
        }
    }

// ---------------------------------------------
// void CInternetEmailEngine::DoCancel()
//  framework cancel method, no implemented here
// ---------------------------------------------
//
void CInternetEmailEngine::DoCancel()
    {
    // no implementation required
    }

// -------------------------------------------------------
// TInt CInternetEmailEngine::RunError( const TInt aError)
//  simply return error value
// -------------------------------------------------------
//
TInt CInternetEmailEngine::RunError( const TInt aError)
    {
    return aError;
    }

// -------------------------------------
// void CInternetEmailEngine::LoadMtmL()
//  common helper to load mtm context
// -------------------------------------
//
void CInternetEmailEngine::LoadMtmL()
    {
    if(iId == KMsvRootIndexEntryId)
        {
        User::Leave(KErrNotFound);
        }

    TMsvEntry tmp;
    User::LeaveIfError(iMsvSession->GetEntry(iId,iServiceId,tmp));
    iMtm =InstantiateClientMtmL(iServiceId,iProtocolUid);
    }

// ------------------------------------------------------
// CBaseMtm* CInternetEmailEngine::InstantiateClientMtmL(
//  TMsvId aService, const TUid aType)
//  Helper to load either imap/pop mtm using base class
// ------------------------------------------------------
//
CBaseMtm* CInternetEmailEngine::InstantiateClientMtmL(TMsvId aService, const TUid aType)
    {
    // create a client mtm
    CBaseMtm* newMtm = iClientReg->NewMtmL(aType);
    CleanupStack::PushL(newMtm);
    // get the entry associated with the given servive ID
    CMsvEntry* entry = iMsvSession->GetEntryL(aService);
    CleanupStack::PushL(entry);
    // set the entry as the current entry
    // mtm takes ownership of the entry
    newMtm->SetCurrentEntryL(entry);
    CleanupStack::Pop(2); //newMtm, entry
    return newMtm;
    }

// -------------------------------------------------------------------
// void CInternetEmailEngine::Queue()
// SetActive() or complete our own status.
// this allows the active scheduler chance to
// service other active objects before return back to our RunL
// -------------------------------------------------------------------
//
void CInternetEmailEngine::Queue()
    {
    if(!IsActive())
        {
        SetActive();
        }

    TRequestStatus* st= &iStatus;
    User::RequestComplete(st,KErrNone);
    }

// -----------------------------------------------------------------
// TBool CInternetEmailEngine::CheckIfExistsL( const TInt typeenum )
//  returns ETrue if given type protocol type exists
// -----------------------------------------------------------------
//
TBool CInternetEmailEngine::CheckIfExistsL( const TInt aTypeEnum )
    {
    TUid temp = ( aTypeEnum == EProtocolImap4 ) ? KUidMsgTypeIMAP4 : KUidMsgTypePOP3;
    TMsvId id=FindServiceL(temp);
    return ( id == KMsvRootIndexEntryId ) ? EFalse : ETrue;
    }


// ---------------------------------------------------------------
// TPtrC CInternetEmailEngine::RemoteEmailTextL(const TInt aIndex)
//      Get the subject of the email from message header
// ---------------------------------------------------------------
//
TPtrC CInternetEmailEngine::RemoteEmailTextL( const TInt aIndex )
    {
    if( iState == ECanceling )
        {
        User::Leave( KErrCancel );
        }

    TMsvId entryId=(*iRemoteEntries)[aIndex];
    if(iEntry==NULL || iEntry->Entry().Id()!=entryId)
        {
        delete iEntry;
        iEntry=NULL;
        iEntry=iMsvSession->GetEntryL(entryId);
        }
    return iEntry->Entry().iDescription;
    }

// ------------------------------------------------------------------
// TPtrC CInternetEmailEngine::RemoteEmailSenderL( const TInt aIndex)
//      Get the sender of the email from message header
// ------------------------------------------------------------------
//
TPtrC CInternetEmailEngine::RemoteEmailSenderL( const TInt aIndex )
    {
    if( iState == ECanceling )
        {
        User::Leave( KErrCancel );
        }

    TMsvId entryId=(*iRemoteEntries)[aIndex];
    if(iEntry==NULL || iEntry->Entry().Id()!=entryId)
        {
        delete iEntry;
        iEntry=NULL;
        iEntry=iMsvSession->GetEntryL(entryId);
        }
    return iEntry->Entry().iDetails;
    }

// ---------------------------------------------
// TInt CInternetEmailEngine::RemoteEmailCount()
//      get the email count as integer
// ---------------------------------------------
//
TInt CInternetEmailEngine::RemoteEmailCount()
    {
    return iRemoteCount;
    }

// ------------------------------------------------------------------
// void CInternetEmailEngine::RemoteOpenEmailL( TInt aIndex )
//  open remote email. may cause system to go online to retrieve mail
// ------------------------------------------------------------------
//
void CInternetEmailEngine::RemoteOpenEmailL( TInt aIndex )
    {
    TMsvId entryId=(*iRemoteEntries)[aIndex];
    if(aIndex<0 || aIndex>=iRemoteCount)
        {
        User::Leave(KErrGeneral);
        }
    iState=EReady;
    DisplayMessageL(entryId);
    }

// --------------------------------------------------------
// void CInternetEmailEngine::RemoteFetchL()
//  enters connect state and queues own request accordingly
// --------------------------------------------------------
//
void CInternetEmailEngine::RemoteFetchL()
    {
    iState=EConnecting;
    Queue();
    }

// ---------------------------------------------------------
// CMsvEntrySelection* CInternetEmailEngine::UpdateEntriesL(
//      const TMsvId& aServiceId)
//  updates the message entries for viewing
// ---------------------------------------------------------
//
CMsvEntrySelection* CInternetEmailEngine::UpdateEntriesL(
    const TMsvId& aServiceId)
    {
    CMsvEntry* entry=iMsvSession->GetEntryL(aServiceId); // get the entry
    CleanupStack::PushL(entry);
    CMsvEntrySelection* entries=entry->ChildrenL(); // get the entry's children
    CleanupStack::PopAndDestroy(entry);
    return entries;
    }

// -----------------------------------------------
// void CInternetEmailEngine::UpdateRemoteCountL()
//  update the remote mail count
//
//  Note: only method where observers are notified
// -----------------------------------------------
//
void CInternetEmailEngine::UpdateRemoteCountL()
    {
    // reset the remote mail count and entries
    iRemoteCount=0;
    delete iRemoteEntries;
    iRemoteEntries=NULL;

    // SEPARATION BETWEEN IMAP AND POP
    if( iProtocolType == EProtocolImap4 && !iHasImapInbox )
        {
        iId=FindFolderThenServiceL(iProtocolUid); //leaves with KErrFound if still no folder set
        }

    // get the current entries. Note that iId is normally a remote
    // mailbox for pop and first folder( inbox ) for imap. which is
    // why we have to check if folders are already sync if protocol
    // is imap, so we wont draw the screen with wrong parent and output
    // folders to the listbox and not the message entries we want to.
    iRemoteEntries=UpdateEntriesL(iId);


    // if the nunber has changed then inform the observers
    if(iRemoteEntries->Count()!=iRemoteCount)
        {
        iRemoteCount=iRemoteEntries->Count();
        iObserver.HandleEngineChangedEventL(ERemoteCountChanged);
        }
    }

// -------------------------------------------
// TBool CInternetEmailEngine::IsProtocolSet()
//  returns ETrue if protocol is set
// -------------------------------------------
//
TBool CInternetEmailEngine::IsProtocolSet()
    {
    return iIsProtocolSet;
    }

// ----------------------------------------------------
// TBool CInternetEmailEngine::IsEngineReady()
//  returns ETrue if server is ready thus engine is too
// ----------------------------------------------------
//
TBool CInternetEmailEngine::IsEngineReady()
    {
    return (iClientReg!=NULL)?(ETrue):(EFalse);
    }

// --------------------------------------------------------------
// void CInternetEmailEngine::SetProtocolL( const TInt aType )
//  sets protocol and loads mtm context for given protocol.
//  note that the structure of mailbox is fundamentally different
//  between simple POP and better IMAP. IMAP has folders like
//  inbox under the service entry so we must the pick different
//  nesting level with IMAP than with POP.
// --------------------------------------------------------------
//
void CInternetEmailEngine::SetProtocolL( const TInt aType )
    {
    iProtocolType=aType;
    iProtocolUid=(iProtocolType==EProtocolPop3)?KUidMsgTypePOP3:KUidMsgTypeIMAP4;
    iIsProtocolSet=ETrue;
    iState=EReady;
    iId=FindServiceL(iProtocolUid);
    LoadMtmL();

    if( iProtocolType!=EProtocolPop3) //Note IMAP has folders!
        {
        TRAPD( err, iId=FindFolderThenServiceL(iProtocolUid));
        if( err != KErrNone )
            {
            //now we know we have fresh imap service without any folders
            return;
            }
        }

    UpdateRemoteCountL(); //we draw the message entries if any


    }



// --------------------------------------------------------------
// TMsvId CInternetEmailEngine::FindServiceL(TUid aType)
//  find the message id of a service given a UID
//  this function will return the first service id for given type
// --------------------------------------------------------------
//
TMsvId CInternetEmailEngine::FindServiceL(TUid aType)
    {
    // select the root index to start the search
    CMsvEntry* currentEntry = iMsvSession->GetEntryL(KMsvRootIndexEntryId);
    CleanupStack::PushL(currentEntry);

    // don't sort the entries
    currentEntry->SetSortTypeL(TMsvSelectionOrdering(KMsvNoGrouping,EMsvSortByNone, ETrue));

    TMsvId rc = KMsvRootIndexEntryId;
   TInt count=currentEntry->Count();
    // loop for every child entry of the root index
    for(TInt i = 0;i<count;i++)
        {
        const TMsvEntry& child = (*currentEntry)[i];

        // is the current child the same type as the type we are looking for ?
        if (child.iMtm == aType)
            {
            rc=child.Id();
            break;
            }
        }//for

    CleanupStack::PopAndDestroy(currentEntry);
    // return the service id of the type if found.
    // otherwise return the root index
    return rc;
    }

// ---------------------------------------------------------------
// TMsvId CInternetEmailEngine::FindFolderThenServiceL(TUid aType)
//  this function will return the first folder of first
//  service of given type or if it doesnt exist then
//  this function will leave with KErrNotFound as most of
//  the usage sitations couldnt handle mailbox instead of
//  folder.
// ----------------------------------------------------------------
//
TMsvId CInternetEmailEngine::FindFolderThenServiceL(TUid aType)
    {

    // select the root index to start the search
    CMsvEntry* currentEntry = iMsvSession->GetEntryL(KMsvRootIndexEntryId);
    CleanupStack::PushL(currentEntry);

    // don't sort the entries
    currentEntry->SetSortTypeL(TMsvSelectionOrdering(KMsvNoGrouping,EMsvSortByNone, ETrue));

    TMsvId rc = KMsvRootIndexEntryId;
    TInt count=currentEntry->Count();
    // loop for every child entry of the root index
    for(TInt i = 0;i<count;i++)
        {
        const TMsvEntry& child = (*currentEntry)[i];

        // is the current child the same type as the type we are looking for ?
        if (child.iMtm == aType)
            {
            // selects first entry as index entry
            CMsvEntry* firstEntry = iMsvSession->GetEntryL(child.Id());
            CleanupStack::PushL(firstEntry);
            TInt innercount=firstEntry->Count();
            if( innercount )
                { //if has childs == folders take first
                const TMsvEntry& folder = (*firstEntry)[0];
                rc=folder.Id();
                iHasImapInbox=ETrue;
                }
            else
                {
                iHasImapInbox=EFalse;
                }
            CleanupStack::PopAndDestroy(firstEntry);
            break;
            }
        }//for

    CleanupStack::PopAndDestroy(currentEntry);

    if( !iHasImapInbox )
        {
        User::Leave( KErrNotFound );
        }

    // return the service id of the type if found.
    return rc;
    }

// End of File
