// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

#include "processserver.h"
#include "processserverstart.h"
#include "processserversession.h"
#include<e32debug.h>

const TInt KProcessServerShutDownDelay = 50000000; //50 sec

//******CProcessServer******//
void CProcessServer::NewLC()
	{
	CProcessServer* s = new(ELeave) CProcessServer();
	CleanupStack::PushL(s);
	s->ConstructL();
	}
/**
	Constructor
*/	
CProcessServer::CProcessServer()
	:CServer2(EPriorityStandard, ESharableSessions),iDriverState(EStateUnknown)
	{
	}
/**
	Second phase construction
*/
void CProcessServer::ConstructL()
	{
	// Call CServer2::StartL() before any other functions inside ConstructL()
	// to avoid an error other than KErrAlreadyExists when the duplicate server starts
	StartL(KProcessServerName);
	iDelayProcessServerShutDown = CDelayServerShutDown::NewL();
	}
/**
	Destructor
*/
CProcessServer::~CProcessServer()
	{
	if (iDelayProcessServerShutDown)
		{
		iDelayProcessServerShutDown->Cancel();
		delete iDelayProcessServerShutDown;
		}
		
	// unload device driver when server terminated
	UnloadDevice();
	}
/**
	Creates a new server-side session
	@param aVersion	Version argument
	@param aMessage The message to process
	@return Server side client session object
*/
CSession2* CProcessServer::NewSessionL(const TVersion& aVersion,const RMessage2& aMessage) const
	{
	//check whether version is compatible
	TVersion v(KProcessServerVersion,KProcessServerMinorVersionNumber,KProcessServerBuildVersionNumber);
	if(!User::QueryVersionSupported(v, aVersion))
		User::Leave(KErrNotSupported);
	
	// check client process has the required capability
	if(!aMessage.HasCapability(ECapabilityMultimediaDD))
		User::Leave(KErrPermissionDenied);		

	// construct and return the server side client session object
	CProcessServer& ncThis = const_cast<CProcessServer&>(*this);
	CProcessServerSession* serverSession = CProcessServerSession::NewL(ncThis);
	return serverSession;
	}
/**
	Increment the count for the number of clients connected
*/
void CProcessServer::IncrementRefCount()
	{
	iRefCount++;
	iDelayProcessServerShutDown->Cancel(); // Cancel shutdown if it has started due to no clients being connected 
	}
	
/**
	Decrement the count for the number of clients connected
*/	
void CProcessServer::DecrementRefCount() 
	{
	iRefCount--;
	if ( iRefCount == 0 )
		{
		iDelayProcessServerShutDown->SetDelay(TTimeIntervalMicroSeconds32(KProcessServerShutDownDelay));
 		}
	}
/**
	Load both LDD and PDD
	@return KErrNone or standard error code.
*/		
TInt CProcessServer::LoadDevice()
	{
	if (iDriverState>=EDriverLoaded && iDriverState<EDriverUnloaded)
		return KErrNone; //Device has been loaded, return immediately
		
	TInt r=User::LoadPhysicalDevice(KDriver1PddFileName);
	if (r!=KErrNone && r!=KErrAlreadyExists)
		{
		return r; //some error occurred
		}

	r = User::LoadLogicalDevice(KDriver1LddFileName);
	if (r!=KErrNone && r!=KErrAlreadyExists)
		{
		return r; //some error occurred 
		}

	//both PDD and LDD have been loaded
	UpdateDriverState(EDriverLoaded);
	return KErrNone; 
	}
/**
	Unload both LDD and PDD
	@return KErrNone or standard error code.
*/	
TInt CProcessServer::UnloadDevice()
	{
	if (iDriverState==EDriverUnloaded || iDriverState == EStateUnknown)
		return KErrNone; //no device is loaded, return immediately
		
	// close device
	if (iDriver.Handle())
		iDriver.Close();
	// Unload Logical Device
	TInt r = User::FreeLogicalDevice(RDriver1::Name());
	if (r!=KErrNone)
		return r;
	// Unload Physical Device
	TName pddName(RDriver1::Name());
	_LIT(KVariantExtension,".template");
	pddName.Append(KVariantExtension);
	r=User::FreePhysicalDevice(pddName);
	
	if (KErrNone==r)
		UpdateDriverState(EDriverUnloaded);	
	return r;
	}
/**
	Open LDD
	@return KErrNone or standard error code.
*/		
TInt CProcessServer::OpenLogicalChannel()
	{
	if (iDriverState>=ELogicalChannelOpened && iDriverState<ELogicalChannelClosed)
		return KErrNone;
	
	TInt r = iDriver.Open();
	if (KErrNone==r)
		UpdateDriverState(ELogicalChannelOpened);
	
	return r;
	}
/**
	Close LDD
*/	
void CProcessServer::CloseLogicalChannel()
	{
	if (iDriver.Handle())
		{
		iDriver.Close();
		UpdateDriverState(ELogicalChannelClosed);
		}
	}
/**
	Send data to LDD device
	@param 	aStatus	A TRequestStatus reference
	@param	aData Data to be sent
	@return KErrNone or standard error code.
*/
TInt CProcessServer::SendDataToDevice(TRequestStatus& aStatus, const TDesC8& aData)
	{
	TInt r = KErrNone;
	if (iDriverState>=ELogicalChannelOpened && iDriverState<ELogicalChannelClosed)
		{
		iDriver.SendData(aStatus, aData);
		UpdateDriverState(ESendingData);
		}
	else
		{
		r = KErrArgument;
		}
	return r;
	}
/**
	Cancel sending data 
*/
void CProcessServer::CancelSendData()
	{
	iDriver.SendDataCancel();
	UpdateDriverState(ELogicalChannelOpened);
	}
/**
	Update the states of the driver
	@param aState State of the driver to be updated
*/
void CProcessServer::UpdateDriverState(TDriverState aState)
	{
	iDriverState = aState;
	}

//EOF
