// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains the definition of member functions of the CConsumer class.
//



/**
 @file 
*/

#include "consumer.h"

/**
Performs the two-phase construction of an object of the CConsumer class.
@param aConsole The console object.
@param aTokens A pointer to the CQueue object containing the tokens' queue.
@return A CConsumer object.
*/
CConsumer* CConsumer::NewL(CConsoleBase* aConsole,CQueue* aTokens)
	{
	CConsumer* self = new (ELeave)CConsumer;
	CleanupStack::PushL(self);
	self->ConstructL(aConsole,aTokens);
	CleanupStack::Pop(self);
	return self;
	}

/**
The second phase constructor of the CConsumer class.
It creates the following member objects of the class:
- iPeriodicCons
- iConsThread
The call back function of the iPeriodicCons object - RemoveFunction() invokes the iConsThread every second.
@param aConsole The console object.
@param aTokens A pointer to the CQueue object containing the tokens' queue.
@see CPeriodic::Start().
@see RThread::Create().
*/
void CConsumer::ConstructL(CConsoleBase* aConsole,CQueue* aTokens)
	{
	iConsole = aConsole;
	iQueue = aTokens;

	// Create an object of the CPeriodic class.
	iPeriodicCons = CPeriodic::NewL(CActive::EPriorityUserInput);

	// Create the Consumer thread.
	_LIT(KConsumer,"ConsumerThread");
	User::LeaveIfError(iConsThread.Create(KConsumer,ConsThreadFunc,KDefaultStackSize,KMinHeapSize,256*KMinHeapSize,iQueue,EOwnerThread));
	iConsThread.SetPriority(EPriorityMore);

	// Make the Consumer thread more intensive than the Producer thread.
	// Consumer thread frequency is twice the Producer thread frequency.
	// Associate RemoveFunction() as the call back function for the iPeriodicCons object.
	iPeriodicCons->Start(2000000,2000000,TCallBack(RemoveFunction,this));
	}

/**
Constructor.
*/
CConsumer::CConsumer()
	{
	}

/**
The consumer thread function.
It removes a token from the queue.
@param aPtr A pointer to the arguments passed to the thread function.
*/
TInt CConsumer::ConsThreadFunc(TAny* aPtr)
	{
	// The while loop mechanism ensures that the thread performs the CQueue::Remove() operation each time RThread::Resume() is invoked on the thread handle.
	while(true)
		{
		CQueue* ptr = (CQueue*)aPtr;
		__ASSERT_ALWAYS(ptr,User::Panic(KTxtPanic,-1));
		// Remove a token from the queue.
		ptr->Remove();
		// Suspend the thread.
		RThread().Suspend();
		}
	return KErrNone;
	}


/**
Resumes the execution of the consumer thread.
*/
void CConsumer::ResumeCons()
	{
	// Call the RThread::Resume() function on the consumer thread.
	// Control goes to the CConsumer::ConsThreadFunc() function after this statement.
	iConsThread.Resume();
	}

/**
Destructor.
*/
CConsumer::~CConsumer()
	{
	iConsThread.Kill(KErrCancel);
	delete iPeriodicCons;
	}


/**
The call back function associated with the CPeriodic object of the CConsumer class.
@see CConsumer::ConstructL().
@see CPeriodic.
*/
TInt CConsumer::RemoveFunction(TAny* aPtr)
	{
	CConsumer* ptr = static_cast<CConsumer*> (aPtr);
	__ASSERT_ALWAYS(ptr,User::Panic(KTxtPanic,-1));
	// Invoke the RThread::Resume() function on the consumer thread repeatedly.
	ptr->ResumeCons();
	return KErrNone;
	}
