// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains the definition of functions defined in the CLocaleSettings class.
// It also contains the definition of the DoStartL()function.
// This function displays a number of locale settings for a variety of locale DLLs.
//



/**
 @file
*/
#include "localeupdate.h"

LOCAL_D CConsoleBase* console;

LOCAL_C void DoStartL();

LOCAL_C void CallExampleL();

/**
Creates an object of the CLocaleSettings class and returns a pointer to it.
@param aLocaleDLLName The descriptor containing the name of the locale DLL.
@param aConsole The console interface.
@return A pointer to an object of the CLocaleSettings class.
*/
CLocaleSettings* CLocaleSettings::NewL(TDesC16& aLocaleDLLName, CConsoleBase* aConsole)
	{
	CLocaleSettings* self = new (ELeave) CLocaleSettings;
	CleanupStack::PushL(self);
	self->ConstructL(aLocaleDLLName, aConsole);
	CleanupStack::Pop(self);
	return self;
	}

/**
Constructs the CLocaleSettings object.
It loads the locale settings from the locale DLL.
@param aLocaleDLLName The descriptor containing the name of the locale DLL.
@param aConsole The console interface.
*/
void CLocaleSettings::ConstructL(TDesC16& aLocaleDLLName, CConsoleBase* aConsole)
	{
	// Store the name of the locale DLL.
	iLocaleDLLName.Copy(aLocaleDLLName);
	iConsole = aConsole;

	// Load the locale DLL.
	User::LeaveIfError(iExtendedLocale.LoadLocale(iLocaleDLLName));

	// Get the TLocale object from TExtendedLocale.
	iLocale = iExtendedLocale.GetLocale();

	// Get the first day of the week.
	// Monday is enumerated to 0, Tuesday to 1 and so on.
	iStartOfWeek = iLocale->StartOfWeek();

	// Get the working days mask.
	// Least significant bit corresponds to Monday.
	// A bit is set if it is a working day.
	iWorkingDaysMask = iLocale->WorkDays();

	// Get the collation method for the locale.
	iCollationMethod = iExtendedLocale.GetPreferredCollationMethod();

	// Get the country code.
	iCountryCode = iLocale->CountryCode();

	// Get the date format.
	iDateFormat = iLocale->DateFormat();

	// Get the time format.
	iTimeFormat = iLocale->TimeFormat();

	// Get numeric value settings.
	iThousandsSeparator = iLocale->ThousandsSeparator();
	iDecimalSeparator = iLocale->DecimalSeparator();

	// Get the units of measurement.
	iUnitsFormat = iLocale->UnitsGeneral();

	// Get the time zone information.
	// A locale's time zone is defined as an offset from UTC.
	iUTCOffset = User::UTCOffset();
	}

/**
Constructor.
*/
CLocaleSettings::CLocaleSettings()
	{
	}

/**
Prints all locale settings of the locale DLL.
*/
void CLocaleSettings::DisplayLocaleSettings()
	{
	// Print the name of the locale DLL.
	_LIT(KTextDLLName, "\r\nLocale Settings for the DLL:");
	console->Printf(KTextDLLName);
	console->Printf(iLocaleDLLName);
	console->Printf(KTextNewLine);

	// Print the calendar settings.
	_LIT(KTextCalenderSettings, "Calendar Settings:\r\n1. First day of the week: %d\r\n");
	iConsole->Printf(KTextCalenderSettings, iStartOfWeek);

	// Print the mask.
	_LIT(KTextWorkingDays, "2. Working days of the week [LSB - Monday]: %07b\r\n");
	iConsole->Printf(KTextWorkingDays, iWorkingDaysMask);

	// Print the collation method.
	_LIT(KTextCollationMethod, "Collation method:");
	iConsole->Printf(KTextCollationMethod);
	if(iCollationMethod.iFlags & TCollationMethod::EIgnoreNone)
		{
		_LIT(KTextIgnoreNone, "IgnoreNone\r\n");
		iConsole->Printf(KTextIgnoreNone);
		}
	else if(iCollationMethod.iFlags & TCollationMethod::ESwapCase)
		{
		_LIT(KTextSwapCase, "SwapCase\r\n");
		iConsole->Printf(KTextSwapCase);
		}
	else if(iCollationMethod.iFlags & TCollationMethod::EAccentsBackwards)
		{
		_LIT(KTextAccents, "AccentsBackwards\r\n");
		iConsole->Printf(KTextAccents);
		}
	else if(iCollationMethod.iFlags & TCollationMethod::ESwapKana)
		{
		_LIT(KTextSwapKana, "SwapKana\r\n");
		iConsole->Printf(KTextSwapKana);
		}
	else if(iCollationMethod.iFlags & TCollationMethod::EFoldCase)
		{
		_LIT(KTextFoldCase, "FoldCase\r\n");
		iConsole->Printf(KTextFoldCase);
		}
	else
		{
		_LIT(KTextStandard, "Standard\r\n");
		iConsole->Printf(KTextStandard);
		}

	// Print the country code.
	_LIT(KTextCountryCode, "Country code: %d\r\n");
	iConsole->Printf(KTextCountryCode, iCountryCode);

	// Get the currency format.
	TBuf<50> currency;
	// A value for the currency.
	TInt currencyValue = 10241024;
	iLocale->FormatCurrency(currency, currencyValue);
	// Print the currency format.
	_LIT(KTextCurrencyFormat, "Currency format of %d:");
	iConsole->Printf(KTextCurrencyFormat, currencyValue);
	iConsole->Printf(currency);
	iConsole->Printf(KTextNewLine);

	// Print the date format.
	_LIT(KTextDateFormat, "Date Format: ");
	iConsole->Printf(KTextDateFormat);
	switch(iDateFormat)
		{
		case EDateAmerican:
			_LIT(KTextUSFormat, "US Format (mm/dd/yy)\r\n");
			iConsole->Printf(KTextUSFormat);
			break;
		case EDateEuropean:
			_LIT(KTextEuropeanFormat, "European format (dd/mm/yyyy)\r\n");
			iConsole->Printf(KTextEuropeanFormat);
			break;
		case EDateJapanese:
			_LIT(KTextJapFormat, "Japanese format (yyyy/mm/dd)\r\n");
			iConsole->Printf(KTextJapFormat);
			break;
		}

	// Print the time format.
	_LIT(KTextTimeFormat, "Time Format: ");
	iConsole->Printf(KTextTimeFormat);
	if(iTimeFormat == ETime12)
		{
		_LIT(KText12Hour, "12 hour format\r\n");
		iConsole->Printf(KText12Hour);
		}
	else
		{
		_LIT(KText24Hour, "24 hour format\r\n");
		iConsole->Printf(KText24Hour);
		}

	// Print the numeric value settings.
	_LIT(KTextNumericValues, "Numeric Values: \r\n");
	iConsole->Printf(KTextNumericValues);
	_LIT(KTextThousandsSeparator, "\tThousands separator: %c\r\n");
	iConsole->Printf(KTextThousandsSeparator, TUint(iThousandsSeparator));
	_LIT(KTextDecimalSeparator, "\tDecimal separator: %c\r\n");
	iConsole->Printf(KTextDecimalSeparator, TUint(iDecimalSeparator));

	// Print the time zone information.
	_LIT(KTextTimeZone, "UTC Offset in seconds (Time Zone Information): %d\r\n");
	iConsole->Printf(KTextTimeZone, iUTCOffset.Int());

	// Print the units of measurement.
	_LIT(KTextUnitsOfMeasure, "Units of measurement: ");
	iConsole->Printf(KTextUnitsOfMeasure);
	if(iUnitsFormat == EUnitsImperial)
		{
		_LIT(KTextImperial, "Imperial\r\n");
		iConsole->Printf(KTextImperial);
		}
	else
		{
		_LIT(KTextMetric, "Metric\r\n");
		iConsole->Printf(KTextMetric);
		}
	}

/**
Destructor.
*/
CLocaleSettings::~CLocaleSettings()
	{
	}

LOCAL_C void DoStartL()
	{
	// UK English locale DLL.
	_LIT(KLocaleName1, "elocl.01");
	TBufC<KLength> localeName1(KLocaleName1);
	// Create a CLocaleSettings object for the locale DLL eloc1.01.
	CLocaleSettings* obj = CLocaleSettings::NewL(localeName1, console);
	CleanupStack::PushL(obj);
	// Display the locale settings for the locale DLL eloc1.01.
	obj->DisplayLocaleSettings();
	// Delete the CLocaleSettings object.
	CleanupStack::PopAndDestroy(obj);
	// Wait for a key press.
	console->Printf(KTextPressAnyKey);
	console->Getch();

	// American locale DLL.
	_LIT(KLocaleName2, "elocl.10");
	TBufC<KLength> localeName2(KLocaleName2);
	// Create a CLocaleSettings object for the locale DLL eloc1.10.
	obj = CLocaleSettings::NewL(localeName2, console);
	CleanupStack::PushL(obj);
	// Display the locale settings for the locale DLL eloc1.10.
	obj->DisplayLocaleSettings();
	// Delete the CLocaleSettings object.
	CleanupStack::PopAndDestroy(obj);
	// Wait for a key press.
	console->Printf(KTextPressAnyKey);
	console->Getch();

	// Taiwanese Chinese locale DLL.
	_LIT(KLocaleName3, "elocl.29");
	TBufC<KLength> localeName3(KLocaleName3);
	// Create a CLocaleSettings object for the locale DLL eloc1.29.
	obj = CLocaleSettings::NewL(localeName3, console);
	CleanupStack::PushL(obj);
	// Display the locale settings for the locale DLL eloc1.29.
	obj->DisplayLocaleSettings();
	// Delete the CLocaleSettings object.
	CleanupStack::PopAndDestroy(obj);
	// Wait for a key press.
	console->Printf(KTextPressAnyKey);
	console->Getch();

	// Japanese locale DLL.
	_LIT(KLocaleName4, "elocl.32");
	TBufC<KLength> localeName4(KLocaleName4);
	// Create a CLocaleSettings object for the locale DLL eloc1.32.
	obj = CLocaleSettings::NewL(localeName4, console);
	CleanupStack::PushL(obj);
	// Display the locale settings for the locale DLL eloc1.32.
	obj->DisplayLocaleSettings();
	// Delete the CLocaleSettings object.
	CleanupStack::PopAndDestroy(obj);
	}

GLDEF_C TInt E32Main() // main function called by E32
    {
	__UHEAP_MARK;
	CTrapCleanup* cleanup=CTrapCleanup::New(); // get clean-up stack
	TRAPD(error, CallExampleL()); // more initialization, then do example
	delete cleanup; // destroy clean-up stack
	__ASSERT_ALWAYS(!error, User::Panic(KTxtEPOC32EX, error));
	__UHEAP_MARKEND;
	return 0; // and return
    }

LOCAL_C void CallExampleL() // initialize and call example code under cleanup stack
    {
	console=Console::NewL(KTxtExampleCode, TSize(KConsFullScreen, KConsFullScreen));
	CleanupStack::PushL(console);
	TRAPD(error, DoStartL()); // perform example function
	if (error)
		console->Printf(KFormatFailed, error);
	else
		console->Printf(KTxtOK);
	console->Printf(KTextPressAnyKey);
	console->Getch(); // get and ignore character
	CleanupStack::PopAndDestroy(); // close console
    }
