// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains definition of functions defined in the CTimerEntry class.
//



/**
 @file
*/

#include "timerentry.h"

/**
Creates an object of the CTimerEntry class.
@param aConsole The user console.
*/
CTimerEntry* CTimerEntry::NewL(CConsoleBase* aConsole)
	{
	CTimerEntry* self = new (ELeave) CTimerEntry(aConsole);
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

/**
The second phase construction of an object of the CTimerEntry class.
*/
void CTimerEntry::ConstructL()
	{
	// Construct the timer object.
	CTimer::ConstructL();
	// Add the object to the active scheduler.
	CActiveScheduler::Add(this);
	}

/**
Constructor.
@param aConsole The user console.
*/
CTimerEntry::CTimerEntry(CConsoleBase* aConsole):
												// Constructor of the timer object.
												CTimer(CActive::EPriorityStandard),
												// Create the iterator object.
												iIterator(iQueue),
												// Set count to -1.
												iCount(-1),
												// Set increment operation as
												// the default operation.
												iOp(EInsert),
												iConsole(aConsole),
												iWaitFlag(ETrue)
	{
	}

/**
Handles the request completion.
*/
void CTimerEntry::RunL()
	{
	// Call the arbitrator function.
	ArbitratorL();
	// Print the contents of the queue.
	PrintQue();
	// Maintain an outstanding request.
	IssueRequest();
	}


/**
Issues a wait request.
*/
void CTimerEntry::IssueRequest()
	{
	// Get a random number for number of seconds to wait.
	// Issue the wait request.
	TUint waitTime = KWaitTime;
	if (iWaitFlag)
		{
		waitTime = Math::Random() % KMaxTimeToWait;
		// Make sure that time to wait is not zero;
		waitTime++;
		}
	CTimer::After((waitTime) * 1000000);
	}

/**
The arbitrator function.
It either adds elements to the list or removes elements from the list
based on the iOp variable.
*/
void CTimerEntry::ArbitratorL()
	{
	// The increment operation.
	// Add elements to the list.
	if(iOp == EInsert)
		{
		// Get the tick count.
		TInt tc = (TInt)User::TickCount();
		// Try to add an element to the list.
		TBool redo = AddToQueueL(tc);
		// The queue is full.
		if(!redo)
			{
			_LIT(KTextQueFull,"Queue is full!\n");
			iConsole->Printf(KTextQueFull);
			// Stop adding elements to the queue.
			// Set iWaitFlag to EFalse.
			// The timer will wait for 1 second before removing an element
			// from the queue.
			iWaitFlag = EFalse;
			}
		}
	// The decrement operation.
	// Remove elements from the list.
	else
		{
		// Try to remove an element from the list.
		TBool redo = RemoveFromQueue();
		// The queue is empty.
		if(!redo)
			{
			_LIT(KTextQueEmpty,"Queue is empty!\n");
			iConsole->Printf(KTextQueEmpty);
			// Stop removing elements from the queue.
			// Set iWaitFlag to ETrue.
			// The timer will wait for a random interval of time
			// before adding an element to the queue.
			iWaitFlag = ETrue;
			}
		}
	}

/**
Adds a tick count to the delta queue.
@param aTicks The tick count.
@return EFalse if the queue is full, else EFalse.
*/
TBool CTimerEntry::AddToQueueL(TInt aTicks)
	{
	// Check if the queue is full.
	if(iCount >=  (KMaxEntry - 1))
			{
			iOp = ERemove;
			return EFalse;
			}

	// Add the tick count to the delta queue.
	_LIT(KTextAddTick,"Added %d to the queue\n");
	iConsole->Printf(KTextAddTick,aTicks);
	iCount++;

	TTimerEntry* ptr = new (ELeave) TTimerEntry;
	iQueue.Add(*ptr,aTicks);

	return ETrue;
	}

/**
Prints the contents of the delta queue.
*/
void CTimerEntry::PrintQue()
	{
	_LIT(KTextCont,"Contents of the List:\n***\nINDEX\tDELTA\n");
	iConsole->Printf(KTextCont);
	// Set the iterator to the first node of the list.
	iIterator.SetToFirst();
	// Iterate the list and print all TDigit objects
	TTimerEntry *ptr = iIterator++;
	// List index.
	TInt ix = 0;
	while (ptr != NULL)
		{
		_LIT(KTextFormat,"%d\t\t%d\n");
		iConsole->Printf(KTextFormat,ix,ptr->iLink.iDelta);
		ptr = iIterator++;
		ix++;
		}
	_LIT(KTextEnd,"***\n");
	iConsole->Printf(KTextEnd);
	}

/**
Removes an element from the delta queue.
@return EFalse if the queue is empty, else EFalse.
*/
TBool CTimerEntry::RemoveFromQueue()
	{
	// Check if the queue is empty.
	if(iCount <  0)
			{
			iOp = EInsert;
			return EFalse;
			}

	iIterator.SetToFirst();

	// The index of the node to be removed in the list.
	TInt pos = 0;

	// Get a random value for position.
	if(iCount > 0)
		{
		pos = Math::Random() % iCount;
		}

	// Iterate the list until the node pointed to by
	// the pos variable is reached.
	for(TInt ix = 1; ix <= pos; ix++)
		{
		iIterator++;
		}

	TTimerEntry* ptr = iIterator;

	_LIT(KTextRemove,"Removed an element from the queue. Index: %d\n");
	iConsole->Printf(KTextRemove,pos);

	// Remove an element from the delta queue.
	iQueue.Remove(*ptr);
	delete ptr;

	iCount--;

	return ETrue;
	}

/**
Destructor.
The TTimerEntry objects were allocated memory at runtime.
Iterate over the list to delete them.
*/
CTimerEntry::~CTimerEntry()
	{
	// Cancel any outstanding request.
	Cancel();
	// Delete the elements from the list
	if(iIterator)
		{
		iIterator.SetToFirst();
		// Iterate the list and delete all TDigit objects
		TTimerEntry *ptr = iIterator++;
		while (ptr != NULL)
			{
			delete ptr;
			ptr = iIterator++;
			}
		}
	}

/**
Cancel any outstanding request.
*/
void CTimerEntry::DoCancel()
	{
	// Call DoCancel() on the base class.
	CTimer::DoCancel();
	}

