// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

#include "threadserver.h"
#include "threadserverstart.h"
#include "threadserversession.h"

const TInt KThreadServerShutDownDelay = 50000000; //50 sec


//******CThreadServer******//
/**
	Starts two phase construction
*/
void CThreadServer::NewLC()
	{
	CThreadServer* s = new(ELeave) CThreadServer();
	CleanupStack::PushL(s);
	s->ConstructL();
	}
/**
	Constructor
*/
CThreadServer::CThreadServer()
	:CServer2(EPriorityStandard, ESharableSessions),iDriverState(EStateUnknown)
	{
	}
void CThreadServer::ConstructL()
	{
	// Call CServer2::StartL() before any other functions inside ConstructL
	// to avoid errors other than KErrAlreadyExists when the duplicate Server starts
	StartL(KThreadServerName);
	iDelayThreadServerShutDown = CDelayServerShutDown::NewL();
	}
/**
	Destructor
*/
CThreadServer::~CThreadServer()
	{
	if (iDelayThreadServerShutDown)
		{
		iDelayThreadServerShutDown->Cancel();
		delete iDelayThreadServerShutDown;
		}
	// unload device driver when server terminated
	UnloadDevice();
	}
/**
	Create a new server-side session
	@param aVersion	Version argument
	@param aMessage The message to process
	@return Instance of server side client session.
*/
CSession2* CThreadServer::NewSessionL(const TVersion& aVersion,const RMessage2& /*aMessage*/) const
	{
	//check whether version is compatible
	TVersion v(KThreadServerVersion,KThreadServerMinorVersionNumber,KThreadServerBuildVersionNumber);
	if(!User::QueryVersionSupported(v, aVersion))
		User::Leave(KErrNotSupported);

	// construct and return the server side client session object
	CThreadServer& ncThis = const_cast<CThreadServer&>(*this);
	CThreadServerSession* serverSession = CThreadServerSession::NewL(ncThis);
	return serverSession;
	}
/**
	Increment the count of the number of clients connected
*/
void CThreadServer::IncrementRefCount()
	{
	iRefCount++;
	iDelayThreadServerShutDown->Cancel(); // Cancel shutdown if it has started due to no clients being connected.
	}
/**
	Decrement the count of the number of clients connected
*/
void CThreadServer::DecrementRefCount() 
	{
	iRefCount--;
	if ( iRefCount == 0 )
		{
		iDelayThreadServerShutDown->SetDelay(TTimeIntervalMicroSeconds32(KThreadServerShutDownDelay));
 		}
	}
/**
 Recreating the same name is prevented by appending the thread ID to the name.
*/
void CThreadServer::RenameServer()
	{
	//Rename thread server name in order to prevent same name creation
	RThread serverThread;
	TThreadId threadId;
	TName name;
	name.Append(KThreadServerName); 
	threadId = serverThread.Id();
	name.AppendNum(threadId.Id(),EHex);
	//We are ignoring the error code returned from User::RenameThread
	//as it is not important here, it may be used for profiling
	User::RenameThread(name);
	}
/**
	Load both LDD and PDD
	@return KErrNone or standard error code.
*/
TInt CThreadServer::LoadDevice()
	{
	if (iDriverState>=EDriverLoaded && iDriverState<EDriverUnloaded)
		return KErrNone; //Device has been loaded, return immediately
		
	TInt r=User::LoadPhysicalDevice(KThreadServerDriverPddFileName);
	if (r!=KErrNone && r!=KErrAlreadyExists)
		return r; //some error occurred
	
	r = User::LoadLogicalDevice(KThreadServerDriverLddFileName);
	if (r!=KErrNone && r!=KErrAlreadyExists)
		return r; //some error occurred 
	
	//both PDD and LDD have been loaded
	UpdateDriverState(EDriverLoaded);
	return KErrNone; 
	}
/**
	Unload both LDD and PDD
	@return KErrNone or standard error code.
*/	
TInt CThreadServer::UnloadDevice()
	{
	if (iDriverState==EDriverUnloaded || iDriverState == EStateUnknown)
		return KErrNone; //no device is loaded, return immediately
		
	// close device
	if (iDriver.Handle())
		iDriver.Close();
	// Unload Logical Device
	TInt r = User::FreeLogicalDevice(RDriver1::Name());
	if (r!=KErrNone)
		return r;
	// Unload Physical Device
	TName pddName(RDriver1::Name());
	_LIT(KVariantExtension,".template");
	pddName.Append(KVariantExtension);
	r=User::FreePhysicalDevice(pddName);
	
	if (KErrNone==r)
		UpdateDriverState(EDriverUnloaded);	
	return r;
	}
/**
	Open LDD
	@return KErrNone or standard error code.
*/	
TInt CThreadServer::OpenLogicalChannel()
	{
	if (iDriverState>=ELogicalChannelOpened && iDriverState<ELogicalChannelClosed)
		return KErrNone;
	
	TInt r = iDriver.Open();
	if (KErrNone==r)
		UpdateDriverState(ELogicalChannelOpened);
	
	return r;
	}
/**
	Close LDD
*/	
void CThreadServer::CloseLogicalChannel()
	{
	if (iDriver.Handle())
		{
		iDriver.Close();
		UpdateDriverState(ELogicalChannelClosed);
		}
	}
/**
	Send data to LDD device
	@param 	aStatus	
	@param	aData Data to be sent
	@return KErrNone or standard error code.
*/
TInt CThreadServer::SendDataToDevice(TRequestStatus& aStatus, const TDesC8& aData)
	{
	TInt r = KErrNone;
	if (iDriverState>=ELogicalChannelOpened && iDriverState<ELogicalChannelClosed)
		{
		iDriver.SendData(aStatus, aData);
		UpdateDriverState(ESendingData);
		}
	else
		{
		r = KErrArgument;
		}
	return r;
	}
/**
	Cancel sending data 
*/
void CThreadServer::CancelSendData()
	{
	iDriver.SendDataCancel();
	UpdateDriverState(ELogicalChannelOpened);
	}
/**
	Update the states of the driver
	@param aState State of the driver to be updated
*/
void CThreadServer::UpdateDriverState(TDriverState aState)
	{
	iDriverState = aState;
	}
//EOF
