// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

#include "threadserversession.h"


/*****************CThreadServerSession**********************/
/**
	Starts two phase construction
*/
CThreadServerSession* CThreadServerSession::NewL(CThreadServer& aServer)
	{
	CThreadServerSession* s = new(ELeave) CThreadServerSession(aServer);
	CleanupStack::PushL(s);
	s->ConstructL();
	CleanupStack::Pop();
	return s;	
	}
/**
	Constructor
*/	
CThreadServerSession::CThreadServerSession(CThreadServer& aServer)
:iServer(aServer)
	{
	}
void CThreadServerSession::ConstructL()
	{
	}
/**
	Destructor
*/		
CThreadServerSession::~CThreadServerSession()
	{
	delete iAsyncRequestHandler;		
	iServer.CloseLogicalChannel();
	iServer.DecrementRefCount();
	}
/**
	Completes construction of this server-side client session object when
	a client makes a connection request and increases the reference count
*/	
void CThreadServerSession::CreateL()
//
// 2nd phase construct for sessions - called by the CServer framework
//
	{
	iServer.IncrementRefCount();		
	//load dummy ldd
	User::LeaveIfError(iServer.LoadDevice());
	iAsyncRequestHandler = CAsyncHandler::NewL(iServer); 	
	}
/**
	Handles the servicing of a client request that has been passed to the server	
*/	
void CThreadServerSession::ServiceL(const RMessage2& aMessage)
	{
	TInt r = KErrNone;
	switch(aMessage.Function())
		{				
		case EThreadServerLoadDeviceDriver:
			aMessage.Complete(iServer.LoadDevice());
			break;
		
		case EThreadServerOpenDriver:
			aMessage.Complete(iServer.OpenLogicalChannel());
			break;
			
		case EThreadServerSendData: //async
			r = iAsyncRequestHandler->ProcessRequest(aMessage);
			if (r!=KErrNone)
				aMessage.Complete(r);
			break;
			
		case EThreadServerSendDataCancel: //cancel async
			iAsyncRequestHandler->Cancel(); 
			aMessage.Complete(KErrNone); 
			break;
			
		case EThreadServerUnloadDeviceDriver:
			aMessage.Complete(iServer.UnloadDevice());
			break;
			
		default:
			User::Leave(KErrNotSupported);
			break;
		}

	}
/*****************CAsyncHandler**********************/
/**
	Active object class to handle asynchronous requests
*/
CThreadServerSession::CAsyncHandler* CThreadServerSession::CAsyncHandler::NewL(CThreadServer& aServer)
	{
	CAsyncHandler* self = new(ELeave) CAsyncHandler(aServer);
	CleanupStack::PushL(self);
	self ->ConstructL();
	CleanupStack::Pop();
	return self;
	}
/**
	Destructor
*/
CThreadServerSession::CAsyncHandler::~CAsyncHandler()
	{
	Cancel();
	iMessageArray.Close();
	}
/**
	Handle the event
*/	
void CThreadServerSession::CAsyncHandler::RunL()
	{
	// complete the request
	Complete(iStatus.Int());
	
	//continue to execute next request if there is any
	ExecuteFirstRequestInArray();
	}
/**
	Cancel request	
*/	
void CThreadServerSession::CAsyncHandler::DoCancel()
	{
	iServer.CancelSendData();
	// complete the request
	Complete(iStatus.Int());
	}
/**
	Constructor
*/
CThreadServerSession::CAsyncHandler::CAsyncHandler(CThreadServer& aServer)
	:CActive(EPriorityStandard), iServer(aServer)
	{
	CActiveScheduler::Add(this);
	}

void CThreadServerSession::CAsyncHandler::ConstructL()
	{
	}
/**
	Implement simple message queue
	@param aMessage Message to be stored in the message queue
	@return KErrNone or standard error code. 
*/		
TInt CThreadServerSession::CAsyncHandler::ProcessRequest(const RMessage2& aMessage)
	{
	TInt ret;
	//store message
	TMessage newMessage;
	newMessage.Message() = aMessage;
	ret= iMessageArray.Append(newMessage);
	if (ret != KErrNone)
		return ret;
	
	// only kick off the first request in the array when there is only one in the array
	if (iMessageArray.Count() == 1) 
		ret = ExecuteFirstRequestInArray();

	return ret;
	}
/** 
	Read the first message in the message queue
	@return KErrNone or standard error code.
*/
TInt CThreadServerSession::CAsyncHandler::ExecuteFirstRequestInArray()
	{
	TInt r = KErrNone;
	if (iMessageArray.Count() != 0)
		{
		const RMessage2& message = iMessageArray[0].Message();
		switch (message.Function())
			{
			case EThreadServerSendData:
				{
				TBuf8<KThreadServerMaxMessageLen> sendObject;
				r= message.Read(0, sendObject);
				if (r == KErrNone)
					{
					r = iServer.SendDataToDevice(iStatus, sendObject);
					if (r==KErrNone)
						SetActive();
					}
				}
				break;
				
			default:
				break;
			}
		}
	return r; 
	}
/**
	Complete issued request
*/	
void CThreadServerSession::CAsyncHandler::Complete(TInt aResult)
	{
	iMessageArray[0].Message().Complete(aResult);
	iMessageArray.Remove(0);
	iServer.UpdateDriverState(CThreadServer::ELogicalChannelOpened);
	}

/***************TMessage****************/
/**
	Constructor
*/
const RMessage2& CThreadServerSession::CAsyncHandler::TMessage::Message() const
	{
	return iMessage;
	}
RMessage2& CThreadServerSession::CAsyncHandler::TMessage::Message()
	{
	return iMessage; 
	}

//EOF
