package com.forum.nokia.taskmanager.webui;

import java.io.IOException;
import java.io.PrintWriter;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import com.forum.nokia.taskmanager.beans.Task;
import com.forum.nokia.taskmanager.database.DBAccess;
import com.forum.nokia.taskmanager.database.UserInfo;

/**
 * This servlet acts as the controller in the system. The system implements the
 * MVC (or Java Model 2) architecture, meaning that model (data storage), view
 * (the look and feel of the system) and controller (decides the flow of
 * operations) are separated from one another.<br>
 * <br>
 * This servlets responsibility is to direct the user to the correct page and to
 * make queries to the database and validate the user input.
 */
public class Controller extends HttpServlet
{
    /**
     * Enum-type constants for the different actions offered by this servlet.
     * They also work as indexes to the array of actions below.
     */
    private static final int ERROR = -1;

    private static final int ADD_NEW_TASK = 0;

    private static final int CHANGE_PASSWORD = 1;

    private static final int DELETE_TASK = 2;

    private static final int LOGIN = 3;

    private static final int LOGOUT = 4;

    private static final int SAVE_TASK = 5;

    private static final int SEND_SMS = 6;

    private static final int ADD_NEW_USER = 7;

    private static final int SAVE_USER = 8;

    private static final int DELETE_USER = 9;

    private static final int SET_TASK_DONE = 10;

    private static final int GET_TASKS = 11;

    private static final int EMPTY = 12;
    
    // The separator constant.
    private static final char SEPARATOR = '#';

    /**
     * The strings recognized as valid actions by the servlet. Note the
     * correspondance between the strings and the indexes defined above.
     */
    private static final String[] COMMANDS = { "AddNewTaskAction",
            "ChangePasswordAction", "DeleteTaskAction", "LoginAction",
            "LogoutAction", "SaveTaskAction", "SendSMSAction",
            "AddNewUserAction", "SaveUserAction", "DeleteUserAction",
            "SetTaskDoneAction", "GetTasksAction", "" };

    /**
     * Default serial version UID.
     */
    private static final long serialVersionUID = 1L;

    /**
     * The database access object (DBAO). Offers services to the Controller
     * servlet.
     */
    private DBAccess database = null;

    private boolean databaseNotInUse = false;

    /**
     * This method is executed when the Controller object is instantiated. It
     * makes sure a DBAO can be found or that at least an error flag is raised.
     */
    public void init()
    {
        database = (DBAccess) getServletContext().getAttribute( "database" );
        if( database == null )
        { // This _should_ be true every time but we try to retrieve a
            // database object from the context anyway.
            try
            {
                database = new DBAccess();
                getServletContext().setAttribute( "database", database );
            }
            catch( Exception e )
            { // Database connection couldn't be opened so the database is not
                // in use.
                System.out.println( "database not in use" );
                databaseNotInUse = true;
            }
        }
    }

    /**
     * This method is called when the Controller object is about to be
     * destroyed. It ensures the DBAO's connection is terminated after it is no
     * longer needed.
     */
    public void destroy()
    {
        super.destroy();

        if( database != null )
        { // the database connection is terminated so there's no unnecessary
            // idle connections to the mysql server.
            database.terminateConnection();
        }
        getServletContext().removeAttribute( "database" );
    }

    /**
     * This function finds the corresponfing "enum" integer to the given action
     * 
     * @param action
     * @return
     */
    private int identifyAction( String action )
    {
        for( int i = 0; i < COMMANDS.length; ++i )
        {
            if( COMMANDS[i].equals( action ) )
            {
                return i;
            }
        }
        return ERROR;
    }

    /**
     * This function is called when a GET method is received from the client. It
     * merely forwards the call to doPost.
     */
    protected void doGet( HttpServletRequest request,
            HttpServletResponse response ) throws ServletException, IOException
    {
        doPost( request, response );
    }

    /**
     * This function is called when a GET method is received from the client. It
     * identifies the action request from the client and calls the function that
     * realizes the action. It then dispatches the request to the correct JSP
     * page.
     */
    protected void doPost( HttpServletRequest request,
            HttpServletResponse response ) throws ServletException, IOException
    {
        String nextPage = "";
        int action = identifyAction( request.getParameter( "action" ) );

        // We need to test if the database connection is alive.
        if( !database.testConnection() )
        {
            try
            { // We try to reconnect to the database.
                database.connect();
            }
            catch( SQLException e )
            { // the database couldn't be reconnected to.
                request.setAttribute( "loginError",
                        "Error: Database connection forming failed." );
                request.setAttribute( "cameFromController", "true" );

                // The client is shown the correct page.
                RequestDispatcher rd = getServletContext()
                        .getRequestDispatcher( "index.jsp" );
                rd.forward( request, response );
            }
        }

        if( request.getSession( false ) == null && action != LOGIN
                && action != SET_TASK_DONE && action != GET_TASKS )
        {
            action = ERROR;
        }
        switch( action )
        {
            case ERROR:
            {
                nextPage = doErrorAction( request, response );
                break;
            }
            case ADD_NEW_TASK:
            {
                nextPage = doAddNewTaskAction( request, response );
                break;
            }
            case CHANGE_PASSWORD:
            {
                nextPage = doChangePasswordAction( request, response );
                break;
            }
            case DELETE_TASK:
            {
                nextPage = doDeleteTaskAction( request, response );
                break;
            }
            case LOGIN:
            {
                nextPage = doLoginAction( request, response );
                break;
            }
            case LOGOUT:
            {
                nextPage = doLogoutAction( request, response );
                break;
            }
            case SAVE_TASK:
            {
                nextPage = doSaveTaskAction( request, response );
                break;
            }
            case SEND_SMS:
            {
                nextPage = doSendSMSAction( request, response );
                break;
            }
            case ADD_NEW_USER:
            {
                nextPage = doAddNewUserAction( request, response );
                break;
            }
            case SAVE_USER:
            {
                nextPage = doSaveUserAction( request, response );
                break;
            }
            case DELETE_USER:
            {
                nextPage = doDeleteUserAction( request, response );
                break;
            }
            case SET_TASK_DONE:
            {
                doSetTaskDoneAction( request, response );
                return;
            }
            case GET_TASKS:
            {
                doGetTasksAction( request, response );
                return;
            }
            case EMPTY:
            {
                nextPage = "/index.jsp";
                break;
            }
            default:
            {
                nextPage = doErrorAction( request, response );
                break;
            }
        }

        request.setAttribute( "cameFromController", "true" );

        // The client is shown the correct page.
        RequestDispatcher rd = getServletContext().getRequestDispatcher(
                nextPage );
        rd.forward( request, response );
    }

    /**
     * Implements the GetTasks action. Retrieves the users tasks from the
     * database and changes them to a string.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     */
    private void doGetTasksAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        PrintWriter out = null;
        try
        {
            out = response.getWriter();
        }
        catch( IOException e )
        {
        }

        // we do a check on the clients login info.
        String username = request.getParameter( "username" );
        String password = request.getParameter( "password" );

        if( username == null || username.equals( "" ) || password == null
                || password.equals( "" ) )
        {
            out.print( SEPARATOR + "Error: username and/or password not provided." );
            return;
        }

        UserInfo uInfo;
        try
        {
            // the username and password are validated from the database.
            uInfo = validateUser( username, password );
            if( !uInfo.userValid )
            {
                out.print( SEPARATOR + "Error: invalid username and/or password." );
                return;
            }
        }
        catch( SQLException e )
        {
            out.print( SEPARATOR + "Error: server-side database error." );
            return;
        }

        // the users tasks are retrieved.
        List tasks = database.getTasks( uInfo.userId );

        // The list starts with a separator sign
        out.print( SEPARATOR );

        if( tasks == null || tasks.size() == 0 )
        { // Empty tasklist
            return;
        }

        Iterator iter = tasks.iterator();
        String taskList = "";
        while( iter.hasNext() )
        { // the tasklist is formatted into a string.
            Task task = (Task) iter.next();
            if( task.getStatus().equals( "NOT_DONE" ) )
            {
                taskList += task.getTaskId() + SEPARATOR + task.getDescription()
                        + SEPARATOR;
            }
        }

        // finally, we update the userinfo so that the users tasks are marked as
        // up to date.
        if( !database.markTasksUpToDate( uInfo.userId ) )
        {
            out.print( SEPARATOR + "Error: server-side database error." );
            return;
        }

        out.print( taskList );
    }

    /**
     * Implements the SetTaskDone action. Marks a specific task as done in the
     * database.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     */
    private void doSetTaskDoneAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        PrintWriter out = null;
        try
        {
            out = response.getWriter();
        }
        catch( IOException e )
        {
        }

        // we do a check on the clients login info.
        String taskId = request.getParameter( "task_id" );
        String username = request.getParameter( "username" );
        String password = request.getParameter( "password" );

        if( username == null || username.equals( "" ) || password == null
                || password.equals( "" ) )
        {
            out.print( SEPARATOR + "Error: username and/or password not provided." );
            return;
        }

        if( taskId == null || taskId.equals( "" ) )
        {
            out.print( SEPARATOR + "Error: task id not provided." );
            return;
        }

        UserInfo uInfo;
        try
        {
            uInfo = validateUser( username, password );
            if( !uInfo.userValid )
            {
                out.print( SEPARATOR + "Error: invalid username and/or password." );
                return;
            }
        }
        catch( SQLException e )
        {
            out.print( SEPARATOR + "Error: server-side database error." );
            return;
        }

        if( !database.markTaskDone( taskId ) )
        {
            out.print( SEPARATOR + "Error: Task marking failed." );
        }
        else
        {
            out.print( SEPARATOR + "OK" );
        }
    }

    /**
     * Implements the DeleteUser action. Tries to remove the user from the
     * database.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doDeleteUserAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        String userId = (String) request.getParameter( "user_id" );

        request.getSession( false ).setAttribute( "sqlError", null );

        if( !database.deleteUser( userId ) )
        {
            request.getSession( false ).setAttribute( "sqlError",
                    "An SQL error occurred while modifying a user." );
        }

        return "/main.jsp";
    }

    /**
     * Implements the SaveUser action. Saves modified user data into the
     * database.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doSaveUserAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        String login = (String) request.getParameter( "user" );
        String pw = (String) request.getParameter( "pw" );
        String mobile = (String) request.getParameter( "phone" );
        String groupId = (String) request.getParameter( "group" );
        String userId = (String) request.getParameter( "user_id" );

        request.getSession( false ).setAttribute( "sqlError", null );

        if( !database.changeUser( login, pw, mobile, groupId, userId ) )
        {
            request.getSession( false ).setAttribute( "sqlError",
                    "An SQL error occurred while modifying a user." );
        }

        return "/main.jsp";
    }

    /**
     * Implements the AddNewUser action. Adds a new row to the user table in the
     * database.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doAddNewUserAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        String login = (String) request.getParameter( "user" );
        String pw = (String) request.getParameter( "pw" );
        String mobile = (String) request.getParameter( "phone" );
        String group = (String) request.getParameter( "group" );

        request.getSession( false ).setAttribute( "sqlError", null );

        if( !database.addNewUser( login, pw, mobile, group ) )
        {
            request.getSession( false ).setAttribute( "sqlError",
                    "An SQL error occurred while adding a user." );
        }

        return "/main.jsp";
    }

    /**
     * Implements the Error action. Sets the request attribute "loginError" so
     * that an error message can be seen on the login screen.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doErrorAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        request.setAttribute( "loginError", "Error: Not logged in" );
        return "/index.jsp";
    }

    /**
     * Implements the AddNewTask action. Adds a new row to the task table.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doAddNewTaskAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        String oldDescription = (String) request.getParameter( "description" );
        String owner_id = (String) request.getParameter( "owner" );
        String state = (String) request.getParameter( "state" );
        String description = null;

        request.getSession( false ).setAttribute( "sqlError", null );
        if( ( description = oldDescription.replace( '#', ' ' ) ) != oldDescription )
        {
            request.getSession( false ).setAttribute( "sqlError",
                    "#-characters are illegal and were removed." );
        }

        if( !database.addNewTask( owner_id, description, state ) )
        {
            request.getSession( false ).setAttribute( "sqlError",
                    "An SQL error occurred while adding a task." );
        }

        return "/main.jsp";
    }

    /**
     * Implements the ChangePassword action. Changes the users password in the
     * database.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doChangePasswordAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        String currentPw = (String) request.getParameter( "currentpw" );
        String newPw = (String) request.getParameter( "pw" );
        String userId = (String) request.getSession( false ).getAttribute(
                "userId" );

        request.getSession( false ).setAttribute( "sqlError", null );

        if( database.checkCurrentPassword( currentPw, userId ) )
        {
            if( !database.changePassword( newPw, userId ) )
            {
                request.getSession( false ).setAttribute( "sqlError",
                        "An SQL error occurred while changing password." );
            }
            else
            {
                request.getSession( false ).setAttribute( "sqlError",
                        "Password changed succesfully." );
            }
        }
        else
        {
            request.getSession( false ).setAttribute( "sqlError",
                    "Current password was incorrect." );
        }

        return "/main.jsp";
    }

    /**
     * Implements the DeleteTask action. Removes the specified task from the
     * task table.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doDeleteTaskAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        String taskId = (String) request.getParameter( "taskId" );
        String ownerId = (String) request.getParameter( "owner" );

        request.getSession( false ).setAttribute( "sqlError", null );

        if( !database.deleteTask( taskId, ownerId ) )
        {
            request.getSession( false ).setAttribute( "sqlError",
                    "An SQL error occurred while deleting a task." );
        }

        return "/main.jsp";
    }

    /**
     * This function implements the login actions. It makes sure the inserted
     * username and password are of valid format and can be found from the
     * database. It also sets up the necessary session attributes.
     * 
     * @param request
     *            Request received from the client.
     * @param response
     *            Response to the client.
     * @return A string depicting the name of the page to be shown next.
     * @throws ServletException
     * @throws IOException
     */
    private String doLoginAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        // Form data is extracted.
        String username = request.getParameter( "username" );
        String password = request.getParameter( "password" );
        UserInfo uInfo;

        // The field must not be empty.
        if( username == null || password == null || username == ""
                || password == "" )
        {
            request.setAttribute( "loginError",
                    "Error: username and/or password not provided." );
            return "/index.jsp";
        }
        // If true, the servlet couldn't connect to the database when
        // initialized
        else if( databaseNotInUse )
        {
            request.setAttribute( "loginError",
                    "Database connection couldn't be formed" );
            return "/index.jsp";
        }

        try
        {
            uInfo = validateUser( username, password );
        }
        catch( SQLException e )
        { // Something went wrong when communicating with the database.
            request.setAttribute( "loginError",
                    "An error occured while communicating with the database." );
            return "/index.jsp";
        }

        if( !uInfo.userValid )
        { // The received username and password couldn't be found from the
            // database.
            request.setAttribute( "loginError", "Invalid username or password" );
            return "/index.jsp";
        }
        // A new session is created.
        HttpSession userSession = request.getSession( true );

        userSession.setAttribute( "userId", uInfo.userId );
        userSession.setAttribute( "userRole", uInfo.userRole );
        userSession.setAttribute( "username", username );

        // The client is redirected to the main page after a succesful login.
        return "/main.jsp";
    }

    /**
     * Implements the Logout action. Invalidates the session and returns the
     * client to the login screen.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doLogoutAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        request.getSession( false ).invalidate();
        return "/index.jsp";

    }

    /**
     * Implements the SaveTask action.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doSaveTaskAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        String oldDescription = (String) request.getParameter( "description" );
        String taskId = (String) request.getParameter( "taskId" );
        String ownerId = (String) request.getParameter( "owner" );
        String state = (String) request.getParameter( "state" );
        String description = null;

        request.getSession( false ).setAttribute( "sqlError", null );
        if( ( description = oldDescription.replace( '#', ' ' ) ) != oldDescription )
        {
            request.getSession( false ).setAttribute( "sqlError",
                    "#-characters are illegal and were removed." );
        }

        if( !database.changeTask( ownerId, description, state, taskId ) )
        {
            request.getSession( false ).setAttribute( "sqlError",
                    "An SQL error occurred while modifying a task." );
        }

        return "/main.jsp";
    }

    /**
     * Implements the SendSMS action.
     * 
     * @param request
     *            The request object from the servlet.
     * @param response
     *            The response object from the servlet.
     * @return The page to be shown next.
     */
    private String doSendSMSAction( HttpServletRequest request,
            HttpServletResponse response )
    {
        request.setAttribute( "smsSent", "true" );

        return "/main.jsp";
    }

    /**
     * This function validates the username and password by accessing the
     * database.
     * 
     * @param username
     *            The username to be checked.
     * @param password
     *            The password to be checked.
     * @return A boolean value depending on wether the username and password
     *         were accepted.
     * @throws SQLException
     *             Thrown if an error occurred while communicating with the
     *             database.
     */
    private UserInfo validateUser( String username, String password )
            throws SQLException
    {
        if( username.length() > 20 || password.length() > 20 )
        {
            return new UserInfo( false );
        }

        // The user login is authenticated.
        UserInfo uInfo = database.validateUser( username, password );

        return uInfo;
    }
}
