/*
* ============================================================================
*  Name     : CTaskManagerEngine from TaskManagerEngine.cpp
*  Part of  : TaskManager
*  Created  : 15/03/2006 by Forum Nokia
*  Version  : 1.2
*  Copyright: Nokia Corporation
* ============================================================================
*/

// INCLUDE FILES
#include "TimeOutTimer.h"
#include "Response.h"
#include "Request.h"
#include "TaskManagerEngine.h"
#include "TaskManagerEngineReader.h"
#include "TaskManager.pan"

#include <msvids.h>				// KMsvGlobalInBoxIndexEntryId
#include <txtrich.h>			// CRichText
#include <mtclreg.h>			// CClientMtmRegistry
#include <mtclbase.h>			// CBaseMtm
#include <commdb.h>				// CCommsDatabase
#include <commdbconnpref.h>		// TCommDbConnPref
#include <es_enum.h>			// TConnectionInfoV2Buf
#include <in_sock.h>			// TInetAddr
#include <eikenv.h>
#include <s32mem.h>				// TDesBuf
#include <e32std.h>				// TLex
#include <securesocket.h>       // CSecureSocket
#include <SocketTaskManager.rsg>

// CONSTANTS

_LIT( KSSL3, "SSL3.0" );

_LIT( KGPRSStartError, "Error starting GPRS service" );
_LIT( KLookupError, "Error looking up server IP");
_LIT( KConnectionError, "Error in forming the connection" );
_LIT( KSSLConnError, "Error forming SSL link between client and server" );
_LIT( KWriteError, "Error writing data to server" );
_LIT( KErroneousPackage, "Package received was erroneous" );

_LIT(KUserNotSet, "Set username and/or password");
_LIT(KIapNotSet, "IAP not selected");
_LIT(KServerNotSet, "Server name not set");
_LIT(KSmsUpdateMessage, "TaskManagerUpdate");

const TInt CTaskManagerEngine::KTimeOut = 30000000; // 30 seconds time-out

// ================= MEMBER FUNCTIONS =======================

// constructor
CTaskManagerEngine::CTaskManagerEngine(MTransactionObserver& aObserver)
	: CActive( EPriorityStandard ), iState( ENotConnected ),
	iOperation( TRequest::EFetchTasks ), iMarkId( 0 ),
	iTransactionObserver( aObserver )
	
	{
	}
	
// destructor
CTaskManagerEngine::~CTaskManagerEngine()
	{
	Cancel(); // Closes the socket and resolver if engine is active

    if( iReader )
        {
        iReader->Cancel();
        }
	delete iReader;
	
	delete iTimer;
	delete iSecureSocket;
	
	iConnection.Close();
	iSocketServer.Close();
	
	delete iMsvEntry;
	delete iMsvSession;
	iIAPs.Close();
	}

// ----------------------------------------------------
// CTaskManagerEngine::NewL()
// Two-phased constructor.
// ----------------------------------------------------
//	
CTaskManagerEngine* CTaskManagerEngine::NewL(MTransactionObserver& aObserver)
	{
	CTaskManagerEngine* self = new (ELeave) CTaskManagerEngine(aObserver);
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

// ----------------------------------------------------
// CTaskManagerEngine::ConstructL()
// Symbian OS default constructor can leave.
// ----------------------------------------------------
//	
void CTaskManagerEngine::ConstructL()
	{
	User::LeaveIfError(iSocketServer.Connect());
	User::LeaveIfError(iConnection.Open(iSocketServer));
	
	// instantiate a timer for connection, socketread etc. timeouts 
	iTimer = CTimeOutTimer::NewL( EPriorityHigh, *this );
	
    iReader = CTaskManagerEngineReader::NewL( *this );
    
    CActiveScheduler::Add( this );
	
	iMsvSession = CMsvSession::OpenAsyncL(*this);
	
	LoadIapsL();
	
	// no IAPs defined in the device.
	if (iIAPs.Count() == 0)
		{
		CEikonEnv::Static()->LeaveWithInfoMsg(R_TASKMANAGER_NO_IAPS_DEFINED);
		}
	
	}

// ----------------------------------------------------
// CTaskManagerEngine::SetConnectionSettings()
// Sets all connections settings. 
// ----------------------------------------------------
//		
void CTaskManagerEngine::SetConnectionSettings(const TDesC& aServerName, 
												const TInt& aPort,
												const TDesC& aUsername, 
												const TDesC& aPassword)
	{
	iUsername.Copy(aUsername);
	iPassword.Copy(aPassword);
	iServer.Copy(aServerName);
	iPort = aPort;
	}

// ----------------------------------------------------
// CTaskManagerEngine::SetIap()
// Sets the IAP to be used in the connections.
// ----------------------------------------------------
//	
void CTaskManagerEngine::SetIap(const TUint32& aId)
	{
	iIap = aId;
	}
	
// ----------------------------------------------------
// CTaskManagerEngine::IapSet()
// Returns ETrue if the IAP is set, EFalse if not.
// ----------------------------------------------------
//	
TBool CTaskManagerEngine::IapSet() const
	{
	if (iIap == 0)
		{
		return EFalse;
		}
	else
		{
		return ETrue;
		}
	}

// ----------------------------------------------------
// CTaskManagerEngine::CheckAndReportErrorsL()
// Checks that all necessary connections settings are 
// set and reports from missing data.
// ----------------------------------------------------
//	
TBool CTaskManagerEngine::CheckAndReportErrorsL()
	{
	if (!IapSet())
		{
		iTransactionObserver.ErrorL(KIapNotSet);
		return ETrue;
		}
		
	if (iServer.Length() == 0)
		{
		iTransactionObserver.ErrorL(KServerNotSet);
		return ETrue;
		}

	return EFalse;
	}

// ----------------------------------------------------
// CTaskManagerEngine::FetchTasksL()
// Starts fetching tasks from the server.
// Called from UI.
// ----------------------------------------------------
//		
void CTaskManagerEngine::FetchTasksL()
	{
	iOperation = TRequest::EFetchTasks;
	
    iTransactionObserver.ConnectingToServerL( ETrue );
	
	iRunning = ETrue;
	GPRSConnectL();
	}
	
// ----------------------------------------------------
// CTaskManagerEngine::MarkTaskDoneL()
// Informs the server that a task has been completed.
// Server updates task info in the database.
// ----------------------------------------------------
//	
void CTaskManagerEngine::MarkTaskDoneL(const TInt& aTaskId)
	{
	iOperation = TRequest::ETaskDone;
	iMarkId.Num( aTaskId );
	
	iRunning = ETrue;
	GPRSConnectL();
	}

// ----------------------------------------------------
// CTaskManagerEngine::CancelTransaction()
// Can be used for cancelling a transaction at 
// any time.
// ----------------------------------------------------
//		
void CTaskManagerEngine::CancelTransaction()
	{
	Cancel();
	
	iRunning = EFalse;
	
	TRAPD( error, iTransactionObserver.CancelledL() );
	if( error != KErrNone )
	    {
	    Panic( error );
	    }
	
	}

// ----------------------------------------------------
// CTaskManagerEngine::CheckRefreshL()
// An SMS update message may have been received while 
// a transaction was running. If so, tasks will 
// be fetched.
// ----------------------------------------------------
//
void CTaskManagerEngine::CheckRefreshL()
	{
	if (iRunning)
	    {
	    return;
	    }

	if (iDoRefresh)
		{
		iDoRefresh = EFalse;
		FetchTasksL();
		}
	}

// ----------------------------------------------------
// CTaskManagerEngine::SetAutomaticUpdateL()
// Defines whether tasks are downloaded automatically 
// when an update SMS message arrives or not.
// ----------------------------------------------------
//
void CTaskManagerEngine::SetAutomaticUpdateL(const TBool& aOn)
	{
	iAutomaticUpdate = aOn;
	if (iAutomaticUpdate)
		{
		CheckRefreshL();
		}
	}

// ----------------------------------------------------
// CTaskManagerEngine::HandleSessionEventL()
// Indicates an event in the message server has occurred. 
// Used here for listening incoming SMS messages.
// ----------------------------------------------------
//	
void CTaskManagerEngine::HandleSessionEventL(TMsvSessionEvent aEvent, TAny* aArg1, TAny* aArg2, TAny* /*aArg3*/)
	{
	switch (aEvent)
		{
		case EMsvServerReady:
			if (!iMsvEntry)
				{
				iMsvEntry = CMsvEntry::NewL(*iMsvSession, KMsvGlobalInBoxIndexEntryId, TMsvSelectionOrdering());
				}
			break;
			
		case EMsvEntriesCreated:
			if (*(static_cast<TMsvId*>(aArg2)) == KMsvGlobalInBoxIndexEntryId)
				{
				CMsvEntrySelection* entries = static_cast<CMsvEntrySelection*>(aArg1);

				iMsvEntry->SetEntryL(entries->At(0));
				TMsvEntry msvEntry(iMsvEntry->Entry());
				msvEntry.SetVisible(EFalse);

				CClientMtmRegistry* mtmReg = CClientMtmRegistry::NewL(*iMsvSession);
				CleanupStack::PushL(mtmReg);
				CBaseMtm* smsMtm = mtmReg->NewMtmL(msvEntry.iMtm);
				smsMtm->SwitchCurrentEntryL(entries->At(0));
				smsMtm->LoadMessageL();
				TBool CorrectSms = EFalse;
				
				// received SMS message is a 'Task manager update' SMS.
				if (smsMtm->Body().Read(0,KSmsUpdateMessage().Length()).Compare(KSmsUpdateMessage)==0)
					{
					msvEntry.SetUnread(EFalse);
					CorrectSms = ETrue;
					}
				// not 'Task manager update' SMS, show it to user.
				else
					{
					msvEntry.SetVisible(ETrue);
					}
					
				iMsvEntry->ChangeL(msvEntry);

				CleanupStack::PopAndDestroy(smsMtm);

				// if received SMS message was a 'Task Manager update' SMS.
				if (CorrectSms)
					{
					// delete the received SMS message for it is not intended for the user to read.
					iMsvEntry->DeleteL(entries->At(0));
					
					// if a transaction is running or program is in the background (paused) 
					// don't fetch tasks yet.
					if (iRunning || iAutomaticUpdate)
						{
						iDoRefresh = ETrue;
						}
					// Transaction is not running, fetch tasks.
					else
						{
						FetchTasksL();
						}
					}
				}
			break;
			
			default:
			break;
		}
	}

// ----------------------------------------------------
// CTaskManagerEngine::LoadIapsL()
// Loads all IAPs of the device.
// ----------------------------------------------------
//	
void CTaskManagerEngine::LoadIapsL()
	{
	// open commdb
	CCommsDatabase* commDb = CCommsDatabase::NewL(EDatabaseTypeIAP);
	CleanupStack::PushL(commDb);

	// open IAP table
	CCommsDbTableView* commView = commDb->OpenIAPTableViewMatchingBearerSetLC(ECommDbBearerCSD|ECommDbBearerGPRS,ECommDbConnectionDirectionOutgoing);
	
	// search all IAPs
	if (commView->GotoFirstRecord() == KErrNone)
		{
		do
			{
			TIap iap;
			commView->ReadTextL(TPtrC(COMMDB_NAME), iap.iName);
			commView->ReadUintL(TPtrC(COMMDB_ID), iap.iId);
			User::LeaveIfError(iIAPs.Append(iap));
			}
		while (commView->GotoNextRecord() == KErrNone);
		}
	CleanupStack::PopAndDestroy(commView);
	CleanupStack::PopAndDestroy(commDb);
	}

// ----------------------------------------------------
// CTaskManagerEngine::Iaps()
// Returns all IAPs of the device.
// ----------------------------------------------------
//		
RArray<TIap>& CTaskManagerEngine::Iaps() 
	{
	return iIAPs;
	}

	
// ----------------------------------------------------
// CTaskManagerEngine::ConnectL() (overloaded)
// First phase of initiating a connection between sockets.
// Handles DNS-lookup etc.
// ----------------------------------------------------
void CTaskManagerEngine::ConnectL()
	{
	if( iState == ENotConnected )
	{
		TInetAddr addr;
		
		// determine if the server address is a valid ip address
		if( addr.Input( iServer ) == KErrNone )
		{
			ConnectL( addr.Address() );
		}
		else // if not, we must try DNS-lookup
		{
			// open a dns-resolver session
			User::LeaveIfError( iResolver.Open( iSocketServer, KAfInet, KProtocolInetUdp) );
			
			// DNS-lookup request
			iResolver.GetByName( iServer, iNameEntry, iStatus );
			
			iState = ELookingUp;
			iTimer->After( KTimeOut );
			SetActive();
		}
	}
}


// ----------------------------------------------------
// CTaskManagerEngine::ConnectL() (overloaded)
// True connection function. Takes a valid IP-address
// as parameter (valid as in xxx.xxx.xxx.xxx form)
// ----------------------------------------------------
void CTaskManagerEngine::ConnectL( TUint32 aAddr )
	{
	if( iState == ENotConnected )
		{
		// open a new socket
#ifdef __WINS__
		// Emulator can't handle RConnections
		User::LeaveIfError( iSocket.Open( iSocketServer,
										  KAfInet,
										  KSockStream,
										  KProtocolInetTcp ) );
#else
		User::LeaveIfError( iSocket.Open( iSocketServer,
										  KAfInet,
										  KSockStream,
										  KProtocolInetTcp,
										  iConnection ) );
#endif
		// set correct port and address
		iAddress.SetPort( iPort );
		iAddress.SetAddress( aAddr );
		
		// try to connect the socket
		iSocket.Connect( iAddress, iStatus );
		iState = EConnecting;
		
		iTimer->After( KTimeOut );
		
		SetActive();
		}
	}
	
// ----------------------------------------------------
// CTaskManagerEngine::DoCancel()
// Called when a cancel is done during activity.
// ----------------------------------------------------
//
void CTaskManagerEngine::DoCancel()
	{
	iTimer->Cancel();
	
	switch( iState )
		{
		case EStartingGPRS:
		    {
		    iConnection.Stop();
		    break;
		    }
		case ELookingUp:
			{
			iResolver.Cancel();
			iResolver.Close();
		    iState = ENotConnected;
			break;
			}
		case EConnecting:
			{
    		iSocket.CancelConnect();
			iSocket.Close();
			iState = ENotConnected;
			break;
			}
		case ESSLConnecting:
		    {
            iSecureSocket->CancelHandshake();
    	    
    	    EndConnection();
		    break;
		    }
		case EWriting:
		    {
		    iSecureSocket->CancelSend();
		    iReader->Cancel();
		    
            EndConnection();
		    break;
		    }
		default:
		    {
		    Panic( ETaskManagerInvalidState );
		    break;
		    }
		};
	}

// ----------------------------------------------------
// CTaskManagerEngine::RunL()
// Handles the states of the object and ensures everything
// is done in right order.
// ----------------------------------------------------
//
void CTaskManagerEngine::RunL()
	{
    iTimer->Cancel();
	
	switch( iState )
		{
		// After GPRS service has been started (or failed to start)
		case EStartingGPRS:
		    {
		    iState = ENotConnected;
		    
		    if( iStatus == KErrNone )
		        {
		        iOpeningConnection = EFalse;
		        iTransactionObserver.ConnectingToServerL( (iOperation == TRequest::EFetchTasks ) );
		        ConnectL();
		        }
		    else
		        {
		        iTransactionObserver.ErrorL( KGPRSStartError );
		        }
		    break;
		    }
		// After DNS resolver has finished
		case ELookingUp:
		    {
			iResolver.Close();
			iState = ENotConnected;
			
			if( iStatus == KErrNone ) // DNS-lookup was successful
				{
				iNameRecord = iNameEntry();
				
				TBuf<15> ipAddr;
				TInetAddr::Cast( iNameRecord.iAddr ).Output( ipAddr );
				
				ConnectL( TInetAddr::Cast( iNameRecord.iAddr ).Address() );
				}
			else // DNS-lookup failed
				{
				iTransactionObserver.ErrorL( KLookupError );
				}

		    break;
		    }
        // After socket has finished connecting
		case EConnecting:
		    {
			// everything went as planned
			if( iStatus == KErrNone )
				{
				iState = ESSLConnecting;
                iSecureSocket = CSecureSocket::NewL( iSocket, KSSL3 );

                #ifdef __SERIES60_30__
                // The CSecureSocket implementation has changed slightly in S60 3rd Edition,
                // this additional option with the server host name needs to be set (in a 8-bit
                // descriptor).
                TBuf8<KMaxServerNameLength> tempName;
                tempName.Copy(iServer);
                iSecureSocket->SetOpt(KSoSSLDomainName, KSolInetSSL, tempName);
                #endif

				iSecureSocket->StartClientHandshake( iStatus );
                iTimer->After( KTimeOut );
				SetActive();
				}
			else // we couldn't open a connection
				{
                iSocket.Close();
				iTransactionObserver.ErrorL( KConnectionError );
				iState = ENotConnected;
				}

		    break;
		    }
		// After the SSL-handshake has been completed
		case ESSLConnecting:
		    {
		    if( iStatus == KErrNone )
		        {
		        iState = EConnected;
		        iReader->SetSecureSocket( iSecureSocket );

                // These parameters are formatted into a string that is saved to iWriteBuffer
                TRequest::GetMessage( iUsername, iPassword, iOperation, iMarkId, iWriteBuffer );
                
                // We begin reading just before writing just so we won't miss anything
            	iReader->Start();
            	
            	// We send the command data to the server
            	iSecureSocket->Send( iWriteBuffer, iStatus );
            	iState = EWriting;

                // Start a timer
            	iTimer->After( KTimeOut );
            	SetActive();
		        }
	        else
		        {
		        // CSecureSocket can be instantiated only after a socket has been connected so
		        // it must be deleted after every connection
		        iSecureSocket->Close(); // this closes the regular socket as well
		        delete iSecureSocket;
		        iSecureSocket = NULL;

		        iState = ENotConnected;
		        iTransactionObserver.ErrorL( KSSLConnError );
		        }
		    break;
		    }
		case EWriting:
		    {
			if( iStatus == KErrNone )
			    {
    			iTimer->After( KTimeOut ); // So we won't wait forever for a response
    			iState = EReading;
			    }
		    else
				{ // writing failed
				iState = EConnected;
				
				iTransactionObserver.ErrorL( KWriteError );
				}
		    break;
		    }
		default:
		    {
		    User::Leave( ETaskManagerInvalidState );
		    }
		};
	}

TInt CTaskManagerEngine::RunError( TInt /*aError*/ )
    {
    return KErrNone;
    }

// ----------------------------------------------------
// CTaskManagerEngine::GPRSConnectL()
// Will open up a (GPRS) connection. We open it using the
// IAP the user chose at the beginning. (View)
// ----------------------------------------------------
//		
void CTaskManagerEngine::GPRSConnectL()
	{
// this functionality is not applicable for the emulator
#ifndef __WINS__ 
	TBool connected = EFalse;
	
	// Lets first check are we already connected.
	TUint connectionCount;
	User::LeaveIfError(iConnection.EnumerateConnections(connectionCount));
	TPckgBuf<TConnectionInfoV2> connectionInfo; 
	for (TUint i = 1; i <= connectionCount; i++)
		{
		User::LeaveIfError(iConnection.GetConnectionInfo(i, connectionInfo));
		if (connectionInfo().iIapId == iIap)
			{
			connected = ETrue;
			break;
			}
		}
	
	// Not yet connected, start connection
	if (!connected)
		{
		//Define preferences for connection
		TCommDbConnPref prefs;
		prefs.SetIapId(iIap);
		prefs.SetDialogPreference(ECommDbDialogPrefDoNotPrompt);

		//Start Connection
		iTransactionObserver.OpeningConnectionL();
		
        //Asnychronic call
        iOpeningConnection = ETrue;
		iConnection.Start(prefs, iStatus);

		iState = EStartingGPRS;
		iTimer->After( KTimeOut );
		SetActive();
		
		return;
		}
#endif // __WINS__

    ConnectL();
	}

// ----------------------------------------------------
// CTaskManagerEngine::PackageReceivedL()
// The socket reader calls this function to pass the received
// data to the engine. The function passes it on to CResponse
// which does the actual parsing.
// ----------------------------------------------------
//
TBool CTaskManagerEngine::PackageReceivedL( const TDesC8& aData )
    {
    // Timer is stopped
    iTimer->Cancel();
    
    if( !iResponse )
        {
        iResponse = CResponse::NewL();
        }
    
    iResponse->InputDataL( aData );
    
    switch( iResponse->GetState() )
        {
        case CResponse::ENotComplete:
            {
            iTimer->After( KTimeOut );
            return ETrue;
            }
        case CResponse::EError:
            {
            iTransactionObserver.ErrorL( KErroneousPackage );
            }
        case CResponse::EComplete:
            {
            iTransactionObserver.SuccessL( *iResponse );
            CheckRefreshL();
            }
        }

    EndConnection();
    return EFalse;
    }

// ----------------------------------------------------
// CTaskManagerEngine::EndConnection()
// Ends the connection to the server in a controlled manner.
// ----------------------------------------------------
//
void CTaskManagerEngine::EndConnection()
    {
    delete iResponse;
    iResponse = NULL;
	
	if( iSecureSocket )
	    {
	    iSecureSocket->Close(); // also closes the normal socket    
	    }
    else
        {
        iSocket.Close();
        }
    
    delete iSecureSocket;
    iSecureSocket = NULL;

	iRunning = EFalse;
	iState = ENotConnected;
    }

// ----------------------------------------------------
// CTaskManagerEngine::TimerExpired()
// Timer calls this function to notify a timeout with
// connection, write or read.
// ----------------------------------------------------
//
void CTaskManagerEngine::TimerExpired()
    {
    TRAPD( err, iTransactionObserver.FailedL( 1 ) );
    
    if( err != KErrNone )
        {
        Panic( err );
        }
    Cancel();
    }

// End of file
