/*
* ============================================================================
*  Name     : CTaskManagerAppUi from TaskManagerAppUi.cpp
*  Part of  : TaskManager
*  Created  : 08/31/2005 by Forum Nokia
*  Version  : 1.01
*  Copyright: Nokia Corporation
* ============================================================================
*/

// INCLUDE FILES
#include "TaskManager.pan"
#include "TaskManagerAppUi.h"
#include "TaskManagerAppView.h"
#include "TaskManager.hrh"
#include "TaskManagerEngine.h"
#include "TaskManagerConnForm.h"
#include "TaskManagerConnInfo.h"

#include <avkon.rsg>
#include <eikapp.h>

// CONSTANTS
_LIT(KSettingsFile, "socketsettings.txt");


// ================= MEMBER FUNCTIONS =======================

// ----------------------------------------------------------
// CTaskManagerAppUi::ConstructL()
// 
// ----------------------------------------------------------
//
void CTaskManagerAppUi::ConstructL()
	{
	// load the menu, etc. configuration from the resources
    BaseConstructL(EAknEnableSkin);

    // create the AppView - the control that will be able to draw on the screen
    iAppView = CTaskManagerAppView::NewL(ClientRect(), *this);    
    
    // required e.g. to display scroll buttons
    iAppView->SetMopParent(this);

    // add the control to the control stack - it will receive key press events
    AddToStackL(iAppView);
    
    // create an engine that handles socket transactions and SMS notifications.
    iEngine = CTaskManagerEngine::NewL(*iAppView);

#ifdef __SERIES60_30__    
	// Only in 3rd Edition
    User::LeaveIfError( iEikonEnv->FsSession().CreatePrivatePath( EDriveC ) );
	User::LeaveIfError( iEikonEnv->FsSession().SetSessionToPrivate( EDriveC ) );

#else
	// Only in 2nd Edition
	#ifndef __WINS__  // don't save settings to z-drive in emulator
    	TFileName appFullName = Application()->AppFullName();
    	TParsePtr appPath(appFullName);
    	iSettingsFile = appPath.DriveAndPath();
	#endif //__WINS__
#endif

    iSettingsFile.Append(KSettingsFile);

	// read settings    
    InternalizeConnInfoL();

	// Here we open the connections dialog. We needn't worry about the return
	// value because the dialog class has internal code that saves changes
	// to iConnectionInfo, if the user presses ok
    if (CTaskManagerConnForm::RunDlgLD( iConnectionInfo ))
    	{
    	iEngine->SetConnectionSettings(iConnectionInfo.ServerAddress(),
    									iConnectionInfo.Port(),
    									iConnectionInfo.Username(), 
    									iConnectionInfo.Password());
    	TUint32 iap;
#ifdef __SERIES60_30__    
			// Set an empty IAP, the device itself handles the correct IAP
    		iConnectionInfo.SetIap(iap);
	    	iEngine->SetIap(iap);
    		iEngine->FetchTasksL();

#else
    		// In 2nd Edition we need to do this manually
    	
    		// query the IAP to be used.
    		if (iAppView->QueryIapL(iap, iConnectionInfo.Iap()))
	    		{
    			iConnectionInfo.SetIap(iap);
	    		iEngine->SetIap(iap);
    			iEngine->FetchTasksL();
    			}
#endif
    		
    	// save settings to file
    	ExternalizeConnInfoL();
       	}
	iEngine->SetAutomaticUpdateL(ETrue);
	}

// ----------------------------------------------------
// CTaskManagerAppUi::~CTaskManagerAppUi()
// Destructor
// Frees reserved resources
// ----------------------------------------------------
//
CTaskManagerAppUi::~CTaskManagerAppUi()
	{
    if (iAppView)
	    {
        RemoveFromStack(iAppView);
        delete iAppView;
        iAppView = NULL;
	    }
    delete iEngine;
	}

// ----------------------------------------------------
// CTaskManagerAppUi::HandleCommandL()
// takes care of command handling
// ----------------------------------------------------
//
void CTaskManagerAppUi::HandleCommandL(TInt aCommand)
	{
    switch(aCommand)
 		{
       	case EEikCmdExit:
       	case EAknSoftkeyExit:
            Exit();
            break;

        case ETaskManagerConnectionSettingsCommand:
    	    {
    	    iUiBusy = ETrue;
    	    iEngine->SetAutomaticUpdateL(EFalse);
    	    
			// Open the connections dialog
			if (CTaskManagerConnForm::RunDlgLD( iConnectionInfo ))
				{
				TUint32 iap;
				// if IAP not yet selected, query the user.
				if (!iEngine->IapSet())
					{
					if (iAppView->QueryIapL(iap, iConnectionInfo.Iap()))
						{
						iConnectionInfo.SetIap(iap);
						iEngine->SetIap(iap);
						}
					}
				// update the connection settings.
				iEngine->SetConnectionSettings(iConnectionInfo.ServerAddress(),
												iConnectionInfo.Port(),
												iConnectionInfo.Username(), 
												iConnectionInfo.Password());
				
				// save the connection settings to a file.
				ExternalizeConnInfoL();
				}
				
			iUiBusy = EFalse;
			SetAutomaticUpdateL();
	        }
	        break;
        
 		case ETaskManagerLoadTasksCommand:
 			iEngine->FetchTasksL();
 			break;
 			
 		case EAknSoftkeyCancel:
 			iEngine->CancelTransaction();
 			break;

        default:
            Panic(ETaskManagerBasicUi);
            break;
    	}
	}

// ----------------------------------------------------
// CTaskManagerAppUi::HandleForegroundEventL()
// Called when an application switches to, or from, the 
// foreground.
// ----------------------------------------------------
//		
void CTaskManagerAppUi::HandleForegroundEventL(TBool aForeground)
	{
	CAknAppUi::HandleForegroundEventL(aForeground);

	// when coming in to foreground, unpause the engine and load 
	// tasks if an update SMS has arrived while we were in the background
	if (aForeground)
		{
		SetAutomaticUpdateL();
		}
	// when going in to background, pause engine so that tasks are not 
	// downloaded automatically if an update SMS arrives.
	else
		{
		iEngine->SetAutomaticUpdateL(EFalse);
		}
	}

// ----------------------------------------------------
// CTaskManagerAppUi::Model()
// Returns a reference to the engine.
// ----------------------------------------------------
//	
CTaskManagerEngine& CTaskManagerAppUi::Model()
	{
	return *iEngine;
	}

// ----------------------------------------------------
// CTaskManagerAppUi::ShowConnectingCbaL()
// While a transaction is running, show only a 
// cancel button.
// ----------------------------------------------------
//	
void CTaskManagerAppUi::ShowConnectingCbaL(const TBool& aShow)
	{
	if (aShow)
		{
		Cba()->SetCommandSetL(R_AVKON_SOFTKEYS_CANCEL);
		}
	else
		{
		Cba()->SetCommandSetL(R_AVKON_SOFTKEYS_OPTIONS_EXIT);
		}
	Cba()->DrawNow();
	}

// ----------------------------------------------------
// CTaskManagerAppUi::InternalizeConnInfoL()
// Reads connection settings from a settings file.
// ----------------------------------------------------
//
void CTaskManagerAppUi::InternalizeConnInfoL()
	{
	RFs& fs = CCoeEnv::Static()->FsSession();
    RFileReadStream readStream;
    TInt error = readStream.Open(fs, iSettingsFile, EFileRead);
    readStream.PushL();
    TInt internalizationError = KErrNone;
    // if settings file existed, try to read settings.
    if (error == KErrNone)
    	{
    	TRAP(internalizationError, iConnectionInfo.InternalizeL(readStream);)
    	}
    readStream.Pop();
    readStream.Release();

	// reading failed, settings file might be corrupted.    
    if (internalizationError != KErrNone)
    	{
		User::LeaveIfError(fs.Delete(iSettingsFile));
    	}

	}

// ----------------------------------------------------
// CTaskManagerAppUi::ExternalizeConnInfoL()
// Saves connection settings to a settings file.
// ----------------------------------------------------
//	
void CTaskManagerAppUi::ExternalizeConnInfoL()
	{
	RFs& fs = CCoeEnv::Static()->FsSession();
	RFileWriteStream writeStream;
	TInt error = writeStream.Open(fs, iSettingsFile, EFileWrite);
	// setting file did not exist, create one.
	if (error != KErrNone)
		{
		User::LeaveIfError(writeStream.Create(fs, iSettingsFile, EFileWrite));
		}
	writeStream.PushL();
	writeStream << iConnectionInfo;
	writeStream.CommitL();
	writeStream.Pop();
	writeStream.Release();
	}

// ----------------------------------------------------
// CTaskManagerAppUi::SetViewBusyL()
// AppView informs the AppUi that it is busy. While 
// AppView is busy, the engine does not update tasks.
// ----------------------------------------------------
//	
void CTaskManagerAppUi::SetViewBusyL(const TBool& aBusy)
	{
	iViewBusy = aBusy;
	if (aBusy)
		{
		iEngine->SetAutomaticUpdateL(EFalse);
		}
	else
		{
		SetAutomaticUpdateL();
		}
	}

// ----------------------------------------------------
// CTaskManagerAppUi::SetAutomaticUpdateL()
// If view and ui aren't busy, tasks can be loaded 
// automatically.
// ----------------------------------------------------
//		
void CTaskManagerAppUi::SetAutomaticUpdateL()
	{
	if (!iViewBusy && !iUiBusy)
		{
		iEngine->SetAutomaticUpdateL(ETrue);
		}
	}
	
// End of file
