/*
* ============================================================================
*  Name     : CTaskManagerEngine from TaskManagerEngine.h
*  Part of  : TaskManager
*  Created  : 08/31/2005 by Forum Nokia
*  Version  : 1.0
*  Copyright: Nokia Corporation
* ============================================================================
*/

#ifndef __TASKMANAGERENGINE_H
#define __TASKMANAGERENGINE_H

// INCLUDE FILES
#include "TaskManager.hrh"
#include "TaskManagerEngineReader.h"
#include "timeouttimer.h"
#include "request.h"

#include <in_sock.h>						//TInetAddr
#include <msvapi.h>							//MMsvSessionObserver
#include <cdblen.h>							//KCommsDbSvrMaxFieldLength
#include <es_sock.h>						//TSockXfrLength

// FORWARD DECLARATIONS
class CSecureSocket;
class CResponse;

// CLASS DECLARATION

/**
* A mixin class for notifying about transactions.
*/
class MTransactionObserver
	{
public:
	/**
	* Called when a GPRS connection is being opened.
	*/
	virtual void OpeningConnectionL() = 0;
	
	/**
	* Called when aa transaction has successfully finished.
	* @param aResponse contains the response received from the server.
	*/
	virtual void SuccessL(const CResponse& aResponse) = 0;
	
	/**
	* Called when a transaction has failed.
	* @param aError the error code.
	*/
	virtual void FailedL(const TInt& aError) = 0;
	
	/**
	* Called when a transaction is initiated.
	* @param aLoadingTasks ETrue if downloading tasks, EFalse if completing a task.
	*/
	virtual void ConnectingToServerL(const TBool& aLoadingTasks) = 0;
	
	/**
	* Called when user cancelled the transaction.
	*/
	virtual void CancelledL() = 0;
	
	/**
	* Called when the user tries to initialise a transaction, but 
	* settings are faulty.
	* @param aErrorMsg error message.
	*/
	virtual void ErrorL(const TDesC& aErrorMsg) = 0;
	};

// CLASS DECLARATION

/**
* A class holding IAP data.
*/
class TIap
	{
public:
	TUint32 iId;
	TBuf<KCommsDbSvrMaxFieldLength> iName;
	};
	

// CLASS DECLARATION

/**
* An engine class.
* Used for communicating with the server.
* Listens also for incoming SMS messages.
*/
class CTaskManagerEngine :	public CActive,
							public MMsvSessionObserver,
							public MEngineNotifier,
							public MTimeoutNotifier
	{
public: // Constructors and destructor

	/**
	* Two-phased constructor.
	*/
	static CTaskManagerEngine* NewL(MTransactionObserver& aObserver);

	/**
	* Destructor
	*/
	~CTaskManagerEngine();
	
protected: // from CActive

	/**
  	* Cancel any outstanding operation
  	*/
   void DoCancel();

	/**
	* Called when an operation completes
  	*/
   void RunL();
   
   /**
   * Handles a leave occurring in the request completion event handler RunL()
   */
   TInt RunError( TInt aError );
	
public: // Main interface for engine logic
	
	/**
	* Sets the connections settings.
	* @param aServerName name of the server.
	* @param aPort Port number of the server.
	* @param aUsername Username.
	* @param aPassword Password.
	*/
	void SetConnectionSettingsL(const TDesC& aServerName, 
								const TInt& aPort,
								const TDesC& aUsername, 
								const TDesC& aPassword);
								
	/**
	* Will make a socket call to the server. As a result, we will get all tasks 
	* of the user.
	*/
	void FetchTasksL();
	
	/**
	* Will make a socket call to the server that is used for completing a wanted task.
	* @param aTaskId The id of the task that is to be completed.
	*/
	void MarkTaskDoneL(const TInt& aTaskId);

	/**
	* Checks if tasks needs to be reloaded. If so, starts the downloading process.
	*/
	void CheckRefreshL();

public: // from MEngineNotifier

    /**
    * Socket reader notifies the engine of a new package through this function.
    * @param aData The package received.
    */
    TBool PackageReceivedL( const TDesC8& aData );

public: // MTimeoutNotifier

    /**
    * Called by the timer whenever a timeout occurs.
    */
    void TimerExpired();

public: // New functions

	/**
	* With this function you can select which IAP is used for the connection.
	* @param aId id of the IAP to be used.
	*/
	void SetIap(const TUint32& aId);
	
	/**
	* This method is used for cancelling a socket transaction.
	*/
	void CancelTransaction();
	
	/**
	* Returns whether IAP has been set or not.
	* @return ETrue if IAP is set, EFalse if not.
	*/
	TBool IapSet() const;
	
	/**
	* Returns all found IAPs.
	* @return Reference to an array of IAPs.
	*/
	RArray<TIap>& Iaps();

    /**
	* This function sets the automatic task loading on or off. If this functionality 
	* is set on, tasks are loaded automatically when an update SMS message is received.
	* @param aOn ETrue if tasks are loaded automatically, EFalse if not.
	*/
    void SetAutomaticUpdateL(const TBool& aOn);
	
private: // Functions from base classes

	/**
	* From MMsvSessionObserver
	*/
	void HandleSessionEventL(TMsvSessionEvent aEvent, TAny* aArg1, TAny* aArg2, TAny* aArg3);
		
private: // New functions

	/**
	* Loads all IAPs of a device into an array.
	*/		 		
	void LoadIapsL();
	
	/**
	* Open a connection in accordance to the connection settings
	*/
	void ConnectL();
	
	/**
	* True connect function, connects to the given IP-addressa
	* @param aAddr The IP-address in 32-bit form.
	*/
	void ConnectL( TUint32 aAddr );
	
	/**
	* Checks that server name and an IAP is set. Reports about "errors" with 
	* MTransactionObserver::ErrorL();
	* @return ETrue if errors existed, EFalse if not.
	*/
	TBool CheckAndReportErrorsL();
	
	/**
	* Opens the GPRS connection.
	*/
	void GPRSConnectL();
	
    /**
    * Closes the connection in a controlled manner.
    */
	void EndConnection();
	
private:

	/**
	* Symbian OS default constructor
	*/
	CTaskManagerEngine(MTransactionObserver& aObserver);
	void ConstructL();
	
	enum TEngineState
   		{
   		// Connection enums
		ENotConnected = 1,
   		EStartingGPRS,
        ELookingUp,
		EConnecting,
		ESSLConnecting,
		EConnected,
		
		// Transmission enums
		EWriting,
		EReading
   		};
   	
private: // Data members

    // Timeout-constant, defined in .cpp
    static const TInt           KTimeOut;

    // Socket handles and variables
	RSocketServ					iSocketServer;
	RSocket						iSocket;
	RConnection					iConnection;
	CSecureSocket*              iSecureSocket;
	TInetAddr					iAddress;

    // State of engine
	TEngineState				iState;
	
    // Timeout timer
	CTimeOutTimer*              iTimer;
	
	// Which operation is being performed
	TRequest::TOperationType	iOperation;
	
	// If the operation is "mark task", the id of
	// the task is stored here.
	TBuf<5> 					iMarkId;
	
	// DNS resolver
	RHostResolver 				iResolver;

	// Result from DNS resolver
	TNameEntry 					iNameEntry;
	TNameRecord 				iNameRecord;

    // Message server observation
	CMsvSession*				iMsvSession;
	CMsvEntry*					iMsvEntry;

    // Handle to inform view
	MTransactionObserver&		iTransactionObserver;
	
	// User variables
	TBuf<KMaxUsernameLength>	iUsername;
	TBuf<KMaxPasswordLength>	iPassword;
	TBuf<KMaxServerNameLength>	iServer;
	TInt						iPort;
	
	// Write message buffer
	TBuf8<KMaxWriteBufferSize>	iWriteBuffer;
	
	// Active object for reading inbound streams
    CTaskManagerEngineReader*   iReader;
    
	// Parser class, hides the structure of messages from the engine
	CResponse*                  iResponse;
	
	// The amount of data read is stored here
	TSockXfrLength				iDummyLength;
	
	TBool						iRunning;
	TBool						iDoRefresh;
	TBool                       iOpeningConnection;
	RArray<TIap>				iIAPs;
	TUint32						iIap;
	TBool						iAutomaticUpdate;
	};

#endif

// End of file