/*
* ============================================================================
*  Name     : CTaskManagerAppUi from TaskManagerAppUi.cpp
*  Part of  : TaskManager
*  Created  : 08/31/2005 by Forum Nokia
*  Version  : 1.0
*  Copyright: Nokia Corporation
* ============================================================================
*/

// INCLUDE FILES
#include <eikon.rsg>
#include <eikapp.h>
#include <eikbtgpc.h> // CEikButtonGroupContainer
#include <eikmenup.h> // CEikMenuPane
#include "TaskManager.pan"
#include "TaskManagerAppUi.h"
#include "TaskManagerAppView.h"
#include "TaskManager.hrh"
#include "TaskManagerEngine.h"
#include "TaskManagerConnForm.h"
#include "TaskManagerConnInfo.h"
#include <sockettaskmanager.rsg>

// CONSTANTS
_LIT(KSettingsFile, "socketsettings.txt");


// ================= MEMBER FUNCTIONS =======================

// ----------------------------------------------------------
// CTaskManagerAppUi::ConstructL()
// 
// ----------------------------------------------------------
//
void CTaskManagerAppUi::ConstructL()
	{
	// load the menu, etc. configuration from the resources
    BaseConstructL();

    // create the AppView - the control that will be able to draw on the screen
    iAppView = CTaskManagerAppView::NewL(ClientRect(), *this);    
    
    // add the control to the control stack - it will receive key press events
    AddToStackL(iAppView);
    
    // create an engine that handles transactions and SMS notifications.
    iEngine = CTaskManagerEngine::NewL(*iAppView);
    
#ifndef __WINS__  // don't save settings to z-drive in emulator  
    TFileName appFullName = Application()->AppFullName();
    TParsePtr appPath(appFullName);
    iSettingsFile = appPath.DriveAndPath();
#endif //__WINS__

    iSettingsFile.Append(KSettingsFile);

	// read settings    
    InternalizeConnInfoL();

	// Here we open the connections dialog. We needn't worry about the return
	// value because the dialog class has internal code that saves changes
	// to iConnectionInfo, if the user presses ok
    if (CTaskManagerConnForm::RunDlgLD( iConnectionInfo ))
    	{
    	iEngine->SetConnectionSettingsL(iConnectionInfo.ServerAddress(),
    									iConnectionInfo.Port(),
    									iConnectionInfo.Username(), 
    									iConnectionInfo.Password());
    	TUint32 iap;
    	// query the IAP to be used.
    	if (iAppView->QueryIapL(iap, iConnectionInfo.Iap())) 
    		{
    		iConnectionInfo.SetIap(iap);
	    	iEngine->SetIap(iap);
    		iEngine->FetchTasksL();
    		}
    		
    	// save settings to file
    	ExternalizeConnInfoL();
       	}
	iEngine->SetAutomaticUpdateL(ETrue);
	
	TasksExist(EFalse);
	}

// ----------------------------------------------------
// CTaskManagerAppUi::~CTaskManagerAppUi()
// Destructor
// Frees reserved resources
// ----------------------------------------------------
//
CTaskManagerAppUi::~CTaskManagerAppUi()
	{
    if (iAppView)
	    {
        RemoveFromStack(iAppView);
        delete iAppView;
        iAppView = NULL;
	    }
    delete iEngine;
	}

// ----------------------------------------------------
// CTaskManagerAppUi::DynInitMenuPaneL()
// Dynamically initialises a menu pane. The Uikon 
// framework calls this function, if it is implemented 
// in a menus observer, immediately before the menu 
// pane is activated.
// ----------------------------------------------------
//	
void CTaskManagerAppUi::DynInitMenuPaneL(TInt aResourceId,CEikMenuPane* aMenuPane)
    {
    if (aResourceId == R_TASKMANAGER_MENU)
    	{
    	aMenuPane->SetItemDimmed(ETaskManagerCompleteTaskCmd, iCompleteTaskDimmed);
    	}
    }

// ----------------------------------------------------
// CTaskManagerAppUi::HandleCommandL()
// takes care of command handling
// ----------------------------------------------------
//
void CTaskManagerAppUi::HandleCommandL(TInt aCommand)
	{
    switch(aCommand)
 		{
       	case EEikCmdExit:
             Exit();
            break;

        case ETaskManagerConnectionSettingsCmd:
    	    {
    	    iUiBusy = ETrue;
    	    iEngine->SetAutomaticUpdateL(EFalse);
    	    
			// Open the connections dialog
			if (CTaskManagerConnForm::RunDlgLD( iConnectionInfo ))
				{
				TUint32 iap;
				// if IAP not yet selected, query the user.
				if (!iEngine->IapSet())
					{
					if (iAppView->QueryIapL(iap, iConnectionInfo.Iap()))
						{
						iConnectionInfo.SetIap(iap);
						iEngine->SetIap(iap);
						}
					}
				// update the connection settings.
				iEngine->SetConnectionSettingsL(iConnectionInfo.ServerAddress(),
												iConnectionInfo.Port(),
												iConnectionInfo.Username(), 
												iConnectionInfo.Password());
				
				// save the connection settings to a file.
				ExternalizeConnInfoL();
				}
				
			iUiBusy = EFalse;
			SetAutomaticUpdateL();
	        }
	        break;
        
 		case ETaskManagerLoadTasksCmd:
 			iEngine->FetchTasksL();
 			break;
 			
 		case EEikBidCancel:
 			iEngine->CancelTransaction();
 			break;
 			
 		case ETaskManagerCompleteTaskCmd:
 			iAppView->CompleteTaskL();
 			break;
    	}
	}

// ----------------------------------------------------
// CTaskManagerAppUi::HandleForegroundEventL()
// Called when an application switches to, or from, the 
// foreground.
// ----------------------------------------------------
//		
void CTaskManagerAppUi::HandleForegroundEventL(TBool aForeground)
	{
	CEikAppUi::HandleForegroundEventL(aForeground);

	// when coming in to foreground, unpause the engine and load 
	// tasks if an update SMS has arrived while we were in the background
	if (aForeground)
		{
		SetAutomaticUpdateL();
		}
	// when going in to background, pause engine so that tasks are not 
	// downloaded automatically if an update SMS arrives.
	else
		{
		iEngine->SetAutomaticUpdateL(EFalse);
		}
	}

// ----------------------------------------------------
// CTaskManagerAppUi::Model()
// Returns a reference to the engine.
// ----------------------------------------------------
//	
CTaskManagerEngine& CTaskManagerAppUi::Model()
	{
	return *iEngine;
	}

// ----------------------------------------------------
// CTaskManagerAppUi::ShowConnectingCbaL()
// While transaction is running, show only a 
// cancel button.
// ----------------------------------------------------
//	
void CTaskManagerAppUi::ShowConnectingCbaL(const TBool& aShow)
	{
	if (aShow)
		{
		CEikButtonGroupContainer::Current()->SetCommandSetL(R_TASKMANAGER_CANCEL_CBA); 
		}
	else
		{
		CEikButtonGroupContainer::Current()->SetCommandSetL(R_TASKMANAGER_CBA);
		CEikButtonGroupContainer::Current()->DimCommand(ETaskManagerCompleteTaskCmd, iCompleteTaskDimmed);
		}
	CEikButtonGroupContainer::Current()->DrawNow();
	}

// ----------------------------------------------------
// CTaskManagerAppUi::TasksExist()
// AppView informs the AppUi whether tasks exist in the
// listbox or not. AppUi then dimms the 'Complete task'
// command accordingly.
// ----------------------------------------------------
//	
void CTaskManagerAppUi::TasksExist(const TBool& aTasksExist)
	{
	iCompleteTaskDimmed = !aTasksExist;
	CEikButtonGroupContainer::Current()->DimCommand(ETaskManagerCompleteTaskCmd, iCompleteTaskDimmed);
	}

// ----------------------------------------------------
// CTaskManagerAppUi::InternalizeConnInfoL()
// Reads connection settings from a settings file.
// ----------------------------------------------------
//
void CTaskManagerAppUi::InternalizeConnInfoL()
	{
	RFs& fs = CCoeEnv::Static()->FsSession();
    RFileReadStream readStream;
    TInt error = readStream.Open(fs, iSettingsFile, EFileRead);
    readStream.PushL();
    TInt internalizationError = KErrNone;
    // if settings file existed, try to read settings.
    if (error == KErrNone)
    	{
    	TRAP(internalizationError, iConnectionInfo.InternalizeL(readStream);)
    	}
    readStream.Pop();
    readStream.Release();

	// reading failed, settings file might be corrupted.    
    if (internalizationError != KErrNone)
    	{
		User::LeaveIfError(fs.Delete(iSettingsFile));
    	}
	}

// ----------------------------------------------------
// CTaskManagerAppUi::ExternalizeConnInfoL()
// Saves connection settings to a settings file.
// ----------------------------------------------------
//	
void CTaskManagerAppUi::ExternalizeConnInfoL()
	{
	RFs& fs = CCoeEnv::Static()->FsSession();
	RFileWriteStream writeStream;
	TInt error = writeStream.Open(fs, iSettingsFile, EFileWrite);
	// setting file did not exist, create one.
	if (error != KErrNone)
		{
		User::LeaveIfError(writeStream.Create(fs, iSettingsFile, EFileWrite));
		}
	writeStream.PushL();
	writeStream << iConnectionInfo;
	writeStream.CommitL();
	writeStream.Pop();
	writeStream.Release();
	}

// ----------------------------------------------------
// CTaskManagerAppUi::SetViewBusy()
// AppView informs the AppUi that it is busy. While 
// AppView is busy, the engine does not update tasks.
// ----------------------------------------------------
//	
void CTaskManagerAppUi::SetViewBusy(const TBool& aBusy)
	{
	iViewBusy = aBusy;
	if (aBusy)
		{
		iEngine->SetAutomaticUpdateL(EFalse);
		}
	else
		{
		SetAutomaticUpdateL();
		}
	}

// ----------------------------------------------------
// CTaskManagerAppUi::SetAutomaticUpdateL()
// If view and ui aren't busy, tasks can be loaded 
// automatically.
// ----------------------------------------------------
//		
void CTaskManagerAppUi::SetAutomaticUpdateL()
	{
	if (!iViewBusy && !iUiBusy)
		{
		iEngine->SetAutomaticUpdateL(ETrue);
		}
	}
	
// End of file